package live.attach.ui.roomactivity;

import android.app.AlertDialog;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.provider.Browser;
import android.support.annotation.RequiresApi;
import android.support.v4.content.ContextCompat;
import android.support.v7.widget.AppCompatImageView;
import android.util.AttributeSet;
import android.view.View;
import android.widget.Button;
import android.widget.TextView;

import java.util.Map;

import live.attach.repackaged.io.reactivex.disposables.Disposable;
import live.attach.domain.model.exception.AttachException;
import live.attach.domain.model.exception.BannerException;
import live.attach.domain.model.exception.warning.AttachWarning;
import live.attach.lib.live.attach.repackaged.io.reactivex.android.schedulers.AndroidSchedulers;
import live.attach.sdk.AttachSdkInternal;
import live.attach.sdk.R;
import live.attach.ui.view.AttachBrandedView;

public class WarningView extends AttachBrandedView {
    private AppCompatImageView icon;
    private Model model;

    private AttachSdkInternal attachSdkInternal;
    private Disposable subscription;

    public WarningView(Context context) {
        super(context);
        init();
    }

    public WarningView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public WarningView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    public WarningView(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        init();
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        subscription = attachSdkInternal
            .getExceptions()
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(e -> this.setMessage(e.getException()));
    }

    @Override
    protected void onDetachedFromWindow() {
        subscription.dispose();
        super.onDetachedFromWindow();
    }

    private void init() {
        attachSdkInternal = (AttachSdkInternal) getAttachActivity().getAttachSdk();

        View root = inflate(getContext(), R.layout.roomactivity_warning_view, this);
        icon = root.findViewById(R.id.warningIcon);

        icon.setOnClickListener(view -> {
            View dialogView = inflate(getContext(), R.layout.roomactivity_warning_dialog_layout, null);
            AlertDialog.Builder builder = new AlertDialog.Builder(getContext());
            builder.setView(dialogView);
            builder.setNegativeButton("Close", (dialog, id) -> dialog.dismiss());
            AlertDialog dialog = builder.create();
            dialog.show();

            AppCompatImageView learnMoreIcon = dialogView.findViewById(R.id.warningDialogLearnMoreIcon);
            Button learMoreButton = dialogView.findViewById(R.id.warningDialogLearnMoreButton);

            learnMoreIcon.setColorFilter(model.color);
            learMoreButton.setTextColor(model.color);

            dialog.getButton(AlertDialog.BUTTON_NEGATIVE).setTextColor(model.color);
            dialog.getButton(AlertDialog.BUTTON_POSITIVE).setTextColor(model.color);

            ((TextView) dialogView.findViewById(R.id.warningDialogMessageText)).setText(model.message);
            String url = model.url;
            if (url == null) {
                dialogView.findViewById(R.id.warningDialogContainer).setVisibility(GONE);
            } else {
                learMoreButton.setOnClickListener(v -> {
                    String url1 = attachSdkInternal.getEnvironment().getPlatformUrl() + model.url;
                    Intent i = new Intent(Intent.ACTION_VIEW);
                    i.setData(Uri.parse(url1));

                    Map<String, String> headers = attachSdkInternal.getEnvironment().getPlatformHeaders();
                    if (headers.size() > 0) {
                        Bundle bundle = new Bundle();
                        for (Map.Entry<String, String> entry : headers.entrySet()) {
                            bundle.putString(entry.getKey(), entry.getValue());
                        }
                        i.putExtra(Browser.EXTRA_HEADERS, bundle);
                    }

                    getAttachActivity().startActivity(i);
                    dialog.dismiss();
                });
            }
        });
    }

    private void setMessage(AttachException exception) {
        if (exception == null) {
            setVisibility(INVISIBLE);
            return;
        }

        if (exception instanceof BannerException) {
            setVisibility(VISIBLE);
            BannerException bannerException = (BannerException) exception;
            icon.setImageResource(bannerException.getIconRes());
            icon.setColorFilter(ContextCompat.getColor(getContext(), bannerException.getColorRes()));
            this.model = new Model(
                bannerException.getBannerMessage(),
                bannerException.getUrl(),
                ContextCompat.getColor(getContext(), bannerException.getColorRes())
            );
        } else if (!(exception instanceof AttachWarning)) {
            setVisibility(VISIBLE);
            icon.setImageResource(R.drawable.ic_bug_report);
            icon.setColorFilter(ContextCompat.getColor(getContext(), R.color.error));
            this.model = new Model(
                "You have error. Look at your console",
                "/assist/debug",
                ContextCompat.getColor(getContext(), R.color.error)
            );
        }
    }

    private static class Model {
        String message;
        String url;
        int color;

        Model(String message, String url, int color) {
            this.message = message;
            this.url = url;
            this.color = color;
        }
    }
}
