package live.attach.ui.roomactivity;

import android.content.Context;
import android.os.Build;
import android.support.annotation.RequiresApi;
import android.util.AttributeSet;
import android.view.View;
import android.widget.TextView;

import live.attach.repackaged.io.reactivex.disposables.Disposable;
import live.attach.lib.live.attach.repackaged.io.reactivex.android.schedulers.AndroidSchedulers;
import live.attach.sdk.AttachSdkInternal;
import live.attach.sdk.R;
import live.attach.infrastructure.android.utils.AndroidUtils;
import live.attach.domain.model.application.AttachProperties;
import live.attach.domain.model.application.Color;
import live.attach.domain.model.exception.error.AttachError;
import live.attach.domain.model.exception.warning.AttachDetailedWarning;
import live.attach.ui.view.AttachBrandedView;

public class ErrorView extends AttachBrandedView {
    private AttachSdkInternal attachSdkInternal;
    private Disposable subscription;

    private View errorMessageContainer;
    private TextView errorMessageText;

    private View warningMessageContainer;
    private TextView warningMessageText;

    public ErrorView(Context context) {
        super(context);
        init();
    }

    public ErrorView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public ErrorView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    public ErrorView(Context context, AttributeSet attrs, int defStyleAttr, int defStyleRes) {
        super(context, attrs, defStyleAttr, defStyleRes);
        init();
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        subscription = attachSdkInternal
            .getExceptions()
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(exception -> handleException(exception.getException()));
    }

    private void handleException(Throwable error) {
        if (error == null) {
            errorMessageContainer.setVisibility(GONE);
            warningMessageContainer.setVisibility(GONE);
        }

        if (error instanceof AttachError) {
            AttachError attachError = (AttachError) error;
            errorMessageContainer.setVisibility(VISIBLE);
            errorMessageText.setText(AndroidUtils.getStringResourceByName(
                attachError.getAttachMessage(),
                getContext(),
                R.string.SDK_SOMETHING_WENT_WRONG
            ));
        } else if (error instanceof AttachDetailedWarning) {
            AttachDetailedWarning warning = (AttachDetailedWarning) error;
            warningMessageText.setText(AndroidUtils.getStringResourceByName(
                warning.getAttachMessageRes(),
                getContext(),
                R.string.SDK_SOMETHING_WENT_WRONG
            ));
            warningMessageContainer.setVisibility(VISIBLE);
            postDelayed(() -> warningMessageContainer.setVisibility(GONE), 1500);
        }
    }

    @Override
    protected void applyBranding(AttachProperties attachProperties) {
        Color bgColor = attachProperties.getParticipantsFocusBackgroundColor();
        warningMessageContainer.setBackgroundColor(android.graphics.Color.parseColor(bgColor.value));

        Color textColor = Color.getForegroundColor(bgColor);
        warningMessageText.setTextColor(android.graphics.Color.parseColor(textColor.value));
    }

    @Override
    protected void onDetachedFromWindow() {
        subscription.dispose();
        super.onDetachedFromWindow();
    }

    private void init() {
        attachSdkInternal = (AttachSdkInternal) getAttachActivity().getAttachSdk();
        View root = inflate(getContext(), R.layout.roomactivity_error_view, this);

        errorMessageText = root.findViewById(R.id.errorMessageText);
        errorMessageContainer = root.findViewById(R.id.errorMessageContainer);
        errorMessageContainer.setOnClickListener(view -> {
            errorMessageContainer.setVisibility(GONE);
            attachSdkInternal.restartRoom();
        });

        warningMessageContainer = root.findViewById(R.id.warningMessageContainer);
        warningMessageText = root.findViewById(R.id.warningMessageText);
    }
}
