package live.attach.sdk;

import live.attach.infrastructure.logger.AttachLogger;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.lib.TextUtils;

/**
 * This is the User class.   It is used to pass the app user into the SDK.
 * @see  <a href="https://documentation.attach.live/android/concepts/users/">Documentation</a>
 */
public class User {
    private static final AttachLogger log = AttachLoggerFactory.getLogger();

    public final String id;
    public final String username;
    public final String avatar;
    public final String token;

    /**
     * Class constructor specifying properties.
     *
     * @param id        the id of the app user, optional
     * @param username  the username of the app user, optional
     * @param avatar    the avatar of the app user, optional
     */
    public User(String id, String username, String avatar) {
        if ("".equals(id)) id = null;
        if ("".equals(username)) username = null;
        if ("".equals(avatar)) avatar = null;

        if (avatar != null && !avatar.startsWith("https")) {
            log.warning("Only https urls are supported for user avatars");
            avatar = null;
        }

        this.id = id;
        this.username = username;
        this.avatar = avatar;
        this.token = null;
    }

    /**
     * Class constructor specifying properties using a signed JSON Web Token (JWT).   The token must contain
     * a valid <code>user</code> claim containing the properties.
     *
     * @param token  the JWT token containing a user claim
     */
    public User(String token) {
        if (TextUtils.isEmpty(token)) {
            throw new IllegalArgumentException("'token' cannot be 'null'.");
        }
        this.id = null;
        this.username = null;
        this.avatar = null;
        this.token = token;
    }

    /**
     * Builder for constructing an instance of the <code>User</code> class.
     */
    public static class Builder {
        private String id;
        private String username;
        private String avatar;
        private String token;

        /**
         * Sets the <code>id</code> property.
         *
         * @param id  the id of the app user, optional
         * @throws IllegalArgumentException  if the token is combined with individual properties
         */
        public Builder withId(String id) {
            if (token != null) {
                throw new IllegalArgumentException("Cannot set 'id' when 'token' is already set.");
            }
            this.id = id;
            return this;
        }

        /**
         * Sets the <code>username</code> property.
         *
         * @param username  the username of the app user, optional
         * @throws IllegalArgumentException  if the token is combined with individual properties
         */
        public Builder withUsername(String username) {
            if (token != null) {
                throw new IllegalArgumentException("Cannot set 'username' when 'token' is already set.");
            }
            this.username = username;
            return this;
        }

        /**
         * Sets the <code>avatar</code> property.
         *
         * @param avatar  the avatar of the app user, optional
         * @throws IllegalArgumentException  if the token is combined with individual properties
         */
        public Builder withAvatar(String avatar) {
            if (token != null) {
                throw new IllegalArgumentException("Cannot set 'avatar' when 'token' is already set.");
            }
            this.avatar = avatar;
            return this;
        }

        /**
         * Sets the user properties using a signed JSON Web Token (JWT).   The token cannot be set in
         * conjunction with other properties.
         *
         * @param token  the JWT token containing a user claim
         * @throws IllegalArgumentException  if the token is combined with individual properties
         */
        public Builder withToken(String token) {
            if (id != null || username != null || avatar != null) {
                throw new IllegalArgumentException("Cannot set 'token' when 'id', 'username' or 'avatar' is already set.");
            }
            this.token = token;
            return this;
        }

        /**
         * Constructs an instance of the <code>user</code> class.
         *
         * @return  the user
         */
        public User build() {
            if (token != null) {
                return new User(token);
            } else {
                return new User(id, username, avatar);
            }
        }
    }
}
