package live.attach.sdk;

import java.util.List;

import live.attach.repackaged.io.reactivex.Observable;
import live.attach.application.AttachRoomInteractor;
import live.attach.application.AttachSessionInteractor;
import live.attach.application.AttachEnvironment;
import live.attach.infrastructure.logger.AttachLogger;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.infrastructure.octopus.socket.video.VideoConnectionClient;
import live.attach.infrastructure.octopus.socket.video.WebRtcVideoController;
import live.attach.domain.model.application.AttachProperties;
import live.attach.domain.model.chat.Message;
import live.attach.domain.model.exception.error.HttpSessionExpiredException;
import live.attach.domain.model.item.ItemId;
import live.attach.domain.model.participant.Participant;
import live.attach.domain.model.session.RoomPresence;
import live.attach.domain.model.session.SessionExceptionInfo;
import live.attach.domain.model.video.VideoCall;

public class AttachSdkInternal extends AttachSdk {
    private static final AttachLogger log = AttachLoggerFactory.getLogger();

    AttachSdkInternal(
        AttachSessionInteractor sessionInteractor,
        AttachRoomInteractor roomInteractor,
        AttachEnvironment environment
    ) {
        super(sessionInteractor, roomInteractor, environment);
    }

    void enterRoom(ItemId itemId) {
        if (!this.sessionInteractor.validSessionExist()) {
            sessionInteractor.refreshOrUseGuest();
        }
        if (itemId != null) {
            roomInteractor.enterRoom(itemId, attachPropertiesSubject.getValue());
        }
    }

    public boolean restartRoom() {
        if (!this.sessionInteractor.validSessionExist()) {
            sessionInteractor.refreshSession();
        }
        return roomInteractor.restartRoom();
    }

    public AttachEnvironment getEnvironment() {
        return environment;
    }

    void pauseRoom() {
        roomInteractor.pauseRoom();
    }

    public void fetchMessages() {
        roomInteractor.requestMessages();
    }

    void stopCall() {
        roomInteractor.stopCall(VideoCall.ByeReason.CALL_FINISH);
    }

    void rejectCall() {
        roomInteractor.stopCall(VideoCall.ByeReason.INVITE_REJECT);
    }

    void callParticipant(Participant participant) {
        roomInteractor.callParticipant(participant);
    }

    void acceptCall() {
        roomInteractor.acceptCall();
    }

    void setProperties(AttachProperties attachProperties) {
        if (attachProperties == null) return;

        AttachProperties currentProperties = this.attachPropertiesSubject.getValue();
        if (currentProperties == null) {
            currentProperties = attachProperties;
        } else {
            currentProperties.merge(attachProperties);
        }
        this.attachPropertiesSubject.onNext(currentProperties);
    }

    public void sendMessage(String message) {
        roomInteractor.sendMessage(message);
    }

    public Observable<List<Participant>> getRoomParticipants() {
        return roomInteractor.getParticipantsStream();
    }

    public Observable<AttachProperties> getAttachProperties() {
        return attachPropertiesSubject;
    }

    public Observable<List<Message>> getRoomMessages() {
        return roomInteractor.getMessagesStream();
    }

    public Observable<SessionExceptionInfo> getExceptions() {
        return roomInteractor.getExceptionsStream();
    }

    Observable<RoomPresence> getRoomPresence() {
        return roomInteractor.getRoomPresenceStream()
            .doOnNext(roomPresence -> {
                if (roomPresence.getError() instanceof HttpSessionExpiredException) {
                    sessionInteractor.expireSession();
                }
            });
    }

    Observable<VideoCall> getVideoCall() {
        return roomInteractor.getVideoCallStream();
    }

    WebRtcVideoController getWebRtcController() {
        return roomInteractor;
    }

    void subscribe(VideoConnectionClient peerConnectionClient) {
        roomInteractor.setVideoConnectionClient(peerConnectionClient);
    }

    void unsubscribe(VideoConnectionClient peerConnectionClient) {
        roomInteractor.setVideoConnectionClient(null);
    }
}
