package live.attach.sdk;

import android.content.Context;

import live.attach.application.AttachEnvironment;
import live.attach.application.AttachRoomInteractor;
import live.attach.application.AttachSessionInteractor;
import live.attach.application.RoomLiveConnection;
import live.attach.application.RoomSessionResolver;
import live.attach.domain.model.application.AttachProperties;
import live.attach.infrastructure.android.cache.Cache;
import live.attach.infrastructure.env.AttachAndroidEnvironment;
import live.attach.infrastructure.http.JavaHttpService;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.infrastructure.logger.RemoteLogger;
import live.attach.infrastructure.octopus.rest.OctopusHttpApi;
import live.attach.infrastructure.octopus.rest.OctopusRoomSessionResolver;
import live.attach.infrastructure.octopus.rest.OctopusSessionResolver;
import live.attach.infrastructure.octopus.socket.OctopusLiveConnection;
import live.attach.lib.TextUtils;
import live.attach.repackaged.com.google.gson.Gson;
import live.attach.repackaged.io.reactivex.schedulers.Schedulers;
import live.attach.repackaged.io.reactivex.subjects.BehaviorSubject;

/**
 * This is the controller for the ATTACH SDK.
 *
 * @see  <a href="https://documentation.attach.live/android/">Documentation</a>
 * @see  <a href="https://github.com/attach-live/attach-sdk-properties/">Properties collection</a>
 */
public class AttachSdk {
    private static AttachSdk instance;

    public static AttachSdk getInstance() {
        if (instance == null) {
            throw new RuntimeException("You need to initialize AttachSdk by calling setContext(context)");
        }
        return instance;
    }

    /**
     * Initializes the ATTACH SDK with the application context.   When using this constructor, the
     * API key must be present in <code>AndroidManifest.xml</code>. Must be initialized once when the app starts.
     *
     * @param context  the application context as provided by the Android <code>Application</code> class
     * @see            <a href="https://documentation.attach.live/android/concepts/setup/">Setting up</a>
     */
    public static void init(Context context) throws RuntimeException {
        init(context, null);
    }

    /**
     * Initializes the ATTACH SDK with the application context and the apiKey.   Must be initialized once
     * when the app starts.
     *
     * @param context  the application context as provided by the Android <code>Application</code> class
     * @param apiKey   the API key
     * @see            <a href="https://documentation.attach.live/android/concepts/setup/">Setting up</a>
     */
    public static void init(Context context, String apiKey) throws RuntimeException {
        instance = createInstance(context, apiKey);
    }

    /**
     * Determines if anonymous error reports are sent.
     *
     * @param enabled  <code>true</code> to report errors (default); <code>false</code> otherwise.
     * @see            <a href="https://documentation.attach.live/android/concepts/error-reporting/">Error reporting</a>
     */
    public static void setErrorReporting(boolean enabled) {
        AttachLoggerFactory.setErrorReporting(enabled);
    }

    private static AttachSdk createInstance(Context context, String apiKey) throws RuntimeException {
        try {
            AttachEnvironment environment = AttachAndroidEnvironment.createEnvironment(context, apiKey);
            JavaHttpService httpService = new JavaHttpService();
            OctopusHttpApi octopusHttpApi = new OctopusHttpApi(httpService, environment);
            Cache cache = new Cache(context);

            AttachLoggerFactory.registerLogger(new RemoteLogger(environment, httpService, new Gson()));

            OctopusSessionResolver sessionResolver = new OctopusSessionResolver(octopusHttpApi, environment, cache);
            RoomSessionResolver roomSessionResolver = new OctopusRoomSessionResolver(octopusHttpApi);
            RoomLiveConnection roomLiveConnection = new OctopusLiveConnection(environment);

            AttachSessionInteractor sessionInteractor = new AttachSessionInteractor(sessionResolver, environment, Schedulers.io());
            AttachRoomInteractor roomInteractor = new AttachRoomInteractor(
                roomSessionResolver,
                roomLiveConnection,
                sessionInteractor.getSessionStream()
            );

            return new AttachSdkInternal(sessionInteractor, roomInteractor, environment);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    AttachSessionInteractor sessionInteractor;
    AttachRoomInteractor roomInteractor;
    AttachEnvironment environment;
    BehaviorSubject<AttachProperties> attachPropertiesSubject;

    AttachSdk(
        AttachSessionInteractor sessionInteractor,
        AttachRoomInteractor roomInteractor,
        AttachEnvironment environment
    ) {
        this.sessionInteractor = sessionInteractor;
        this.roomInteractor = roomInteractor;
        this.environment = environment;
        this.attachPropertiesSubject = BehaviorSubject.createDefault(new AttachProperties());
    }

    /**
     * Signs the app user into the SDK.   Should be called every time the app user changes. If not called,
     * a guest user with a random avatar will be used for ATTACH.
     *
     * @param user  the user who is signed into the app
     */
    public static void signIn(User user) {
        if (user == null) {
            signOut();
            return;
        }

        getInstance().sessionInteractor.signIn(
            TextUtils.isEmpty(user.token) ?
                new live.attach.domain.model.session.User(
                    TextUtils.isEmpty(user.id) ? "__anonymous__" : user.id,
                    user.avatar,
                    user.username)
                :
                new live.attach.domain.model.session.User(user.token)
        );
    }

    /**
     * Signs the app user out of the SDK.   Must be called when the app user signs out in the app.
     */
    public static void signOut() {
        getInstance().sessionInteractor.signIn(null);
    }
}
