package live.attach.sdk;

import android.Manifest;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.LayoutRes;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v7.widget.AppCompatImageView;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import org.webrtc.EglBase;
import org.webrtc.SurfaceViewRenderer;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import live.attach.repackaged.io.reactivex.disposables.Disposable;
import live.attach.domain.model.application.AttachProperties;
import live.attach.domain.model.application.Color;
import live.attach.domain.model.exception.error.AttachError;
import live.attach.domain.model.session.RoomPresence;
import live.attach.domain.model.video.VideoCall;
import live.attach.infrastructure.logger.AttachLogger;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.infrastructure.octopus.video.RemoteStreamListener;
import live.attach.infrastructure.octopus.video.audio.AppRTCAudioManager;
import live.attach.infrastructure.octopus.video.webrtc.CameraListener;
import live.attach.infrastructure.octopus.video.webrtc.PeerConnectionClient;
import live.attach.infrastructure.octopus.video.webrtc.PeerConnectionParameters;
import live.attach.lib.live.attach.repackaged.io.reactivex.android.schedulers.AndroidSchedulers;
import live.attach.ui.SoundNotifier;
import live.attach.ui.ViewUtils;
import live.attach.ui.bottomsheet.viewpager.ViewPagerBottomSheetBehavior;
import live.attach.ui.roomactivity.call.CallView;
import live.attach.ui.roomactivity.layout.RoomActivityLayoutManager;
import live.attach.ui.roomactivity.overlay.OverlayContentView;
import live.attach.ui.roomactivity.participant.ParticipantsView;
import live.attach.ui.roomactivity.video.CameraPermissionsView;
import live.attach.ui.roomactivity.video.RemoteVideoView;
import live.attach.ui.roomactivity.video.VideoOverlayView;

/**
 * This is the ATTACH overlay activity.   It is used instead <code>AppCompatActivity</code> to extend
 * any <code>Activity</code> to display an ATTACH overlay layer above the activity.
 *
 * @see  <a href="https://documentation.attach.live/android/walkthroughs/overlay/">Walkthrough</a>
 */
public abstract class AttachOverlayActivity extends AttachActivity {
    private static final AttachLogger log = AttachLoggerFactory.getLogger();

    private static final int CAMERA_PERMISSIONS = 100;
    private static final int SCREEN_VIDEO = android.graphics.Color.argb(0, 0, 0, 0);
    private static final int SCREEN_BLACK = android.graphics.Color.argb(255, 0, 0, 0);

    private AttachSdkInternal attachSdk = (AttachSdkInternal) getAttachSdk();

    private EglBase rootEglBase;
    private PeerConnectionClient peerConnectionClient;
    private AppRTCAudioManager audioManager;
    private List<SurfaceViewRenderer> renderers = new ArrayList<>();

    private ParticipantsView participantsView;
    private CallView callView;

    private AppCompatImageView videoToggleButton;

    private CameraPermissionsView cameraPermissionsView;
    private OverlayContentView overlayContent;

    private RoomActivityLayoutManager layoutManager;
    private SoundNotifier sound;

    private Disposable videoStreamSubscription;
    private Disposable roomPresenceSubscription;
    private Disposable brandingSubscription;

    private OnPermissionChangeListener onPermissionChangeListener;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        super.setContentView(R.layout.activity_roomactivity);
        layoutManager = new RoomActivityLayoutManager(this);
        rootEglBase = EglBase.create();

        participantsView = findViewById(R.id.participantsView);
        overlayContent = findViewById(R.id.overlayContent);

        RemoteVideoView remoteVideoView = findViewById(R.id.remoteVideoView);
        videoToggleButton = findViewById(R.id.videoToggleButton);
        callView = findViewById(R.id.callView);
        cameraPermissionsView = findViewById(R.id.cameraPermissionsView);


        VideoOverlayView videoOverlayView = findViewById(R.id.videoOverlayControls);
        videoOverlayView.setSwitchVideoClickListener(view -> {
            if (peerConnectionClient != null) {
                peerConnectionClient.switchCamera();
            }
        });

        renderers.add(findViewById(R.id.videoLocal));
        renderers.add(remoteVideoView.getSurfaceRenderer());

        sound = new SoundNotifier(this);

        cameraPermissionsView.setGrantPermissionListener(view ->
            askAttachPermissions((permissions, grantResults) -> {
                cameraPermissionsView.setVisibility(View.GONE);
                showLocalVideo();
            })
        );
        videoToggleButton.setOnClickListener(view -> {
            if (layoutManager.isVideoVisible()) {
                hideLocalVideo();
            } else {
                showLocalVideo();
            }
        });

        participantsView.setOnParticipantClick(participant -> {
            if (hasAttachPermissions()) {
                startVideo();
                attachSdk.callParticipant(participant);
            } else {
                showLocalVideo();
            }
        });
        overlayContent.setOnParticipantClickListener(participant -> {
            if (hasAttachPermissions()) {
                startVideo();
                attachSdk.callParticipant(participant);
            } else {
                showLocalVideo();
            }
        });

        participantsView.setOnShowMoreClickListener(view -> {
            layoutManager.bottomSheetBehavior().setState(ViewPagerBottomSheetBehavior.STATE_EXPANDED);
            overlayContent.enablePaging();
            overlayContent.postDelayed(() -> overlayContent.showParticipants(), 300);
        });

        callView.setOnAcceptCallClickListener(view -> {
            if (hasAttachPermissions()) {
                startVideo();
                peerConnectionClient.createOffer();
                attachSdk.acceptCall();
            } else {
                showLocalVideo();
            }
        });
        callView.setOnDismissCallClickListener(view -> attachSdk.rejectCall());
        remoteVideoView.setOnDismissCallClickListener(view -> attachSdk.stopCall());
        layoutManager.onCreate();
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        layoutManager.onConfigurationChanged(newConfig);
    }


    private void applyProperties(AttachProperties attachProperties) {
        Color participantsPrimary = attachProperties.getParticipantsFocusBackgroundColor();
        Color participantsForeground = Color.getForegroundColor(participantsPrimary);
        videoToggleButton.setColorFilter(android.graphics.Color.parseColor(participantsForeground.value));

        if (!attachProperties.isVideocallEnabled()) {
            videoToggleButton.setVisibility(View.GONE);
            participantsView.setOnParticipantClick(null);
        }
    }

    private AttachSdkInternal sdk() {
        return (AttachSdkInternal) getAttachSdk();
    }

    @Override
    protected void onStart() {
        super.onStart();
        videoStreamSubscription = sdk()
            .getVideoCall()
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(this::handleVideoCallChange);

        roomPresenceSubscription = sdk()
            .getRoomPresence()
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(this::handleRoomPresence);

        brandingSubscription = sdk()
            .getAttachProperties()
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(this::applyProperties);
    }

    @Override
    protected void onStop() {
        videoStreamSubscription.dispose();
        roomPresenceSubscription.dispose();
        brandingSubscription.dispose();

        stopVideo();
        layoutManager.hideVideo();
        attachSdk.stopCall();
        sound.stopAll();
        super.onStop();
    }

    @Override
    protected void onDestroy() {
        rootEglBase.release();
        super.onDestroy();
    }

    @Override
    public void setContentView(@LayoutRes int layoutResID) {
        View view = LayoutInflater.from(this).inflate(layoutResID, null, false);
        ((ViewGroup) findViewById(R.id.contentContainer)).addView(view);
    }

    private void handleRoomPresence(RoomPresence roomPresence) {
        if (roomPresence.getStatus() == RoomPresence.Status.OFFLINE) {
            overlayContent.reset();
        }
        if (roomPresence.getError() instanceof AttachError) {
            ViewUtils.hideKeyboard(this);
            layoutManager.bottomSheetBehavior().disable();
        } else {
            layoutManager.bottomSheetBehavior().enable();
        }
    }

    private void showLocalVideo() {
        layoutManager.showLocalVideo();
        if (hasAttachPermissions()) {
            startVideo();
        }
        videoToggleButton.setImageResource(R.drawable.ic_videocam_off);
    }

    private void hideLocalVideo() {
        stopVideo();
        layoutManager.hideVideo();
        videoToggleButton.setImageResource(R.drawable.ic_videocam);
    }

    private void handleVideoCallChange(VideoCall videoCall) {
        log.info("Video call state: " + videoCall);
        switch (videoCall.state) {
            case NONE:
                sound.stopAll();
                stopVideo();
                participantsView.setVisibility(View.VISIBLE);
                layoutManager.hideVideo();
                callView.hide();
                videoToggleButton.setImageResource(R.drawable.ic_videocam);
                break;
            case IN_PROGRESS:
                sound.playConnect();
                callView.hide();
                startVideo();
                layoutManager.showRemoteVideo(videoCall);
                videoToggleButton.setImageResource(R.drawable.ic_videocam_off);
                break;
            case INCOMING:
                sound.playRing();
                callView.setParticipant(videoCall.participant, true);
                videoToggleButton.setImageResource(R.drawable.ic_videocam_off);
                break;
            case OUTGOING:
                sound.playRing();
                callView.setParticipant(videoCall.participant, false);
                videoToggleButton.setImageResource(R.drawable.ic_videocam_off);
                break;
            case STOPPED:
                sound.playDisconnect();
                stopVideo();
                participantsView.setVisibility(View.GONE);
                layoutManager.hideVideo();
                callView.setStatus(videoCall.byeReason);
                videoToggleButton.setImageResource(R.drawable.ic_videocam_off);
                break;
        }
    }

    private void askAttachPermissions(OnPermissionChangeListener onPermissionChangeListener) {
        this.onPermissionChangeListener = onPermissionChangeListener;
        String[] permissions = {
            Manifest.permission.CAMERA,
            Manifest.permission.RECORD_AUDIO
        };
        ActivityCompat.requestPermissions(this, permissions, CAMERA_PERMISSIONS);
    }

    public boolean hasAttachPermissions() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            return checkSelfPermission(Manifest.permission.CAMERA) == PackageManager.PERMISSION_GRANTED
                && checkSelfPermission(Manifest.permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED;
        } else {
            return true;
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == CAMERA_PERMISSIONS) {
            if (onPermissionChangeListener != null) {
                onPermissionChangeListener.onPermissionChange(permissions, grantResults);
                onPermissionChangeListener = null;
            }
        }
    }

    private CameraListener cameraListener = new CameraListener() {
        @Override
        public void onCameraStarted(final String camera, boolean mirrorLocal) {
            runOnUiThread(() -> {
                for (SurfaceViewRenderer renderer : renderers) {
                    renderer.setBackgroundColor(SCREEN_VIDEO);
                }
            });
        }

        @Override
        public void onCameraStopped() {
            runOnUiThread(() -> {
                for (SurfaceViewRenderer renderer : renderers) {
                    renderer.setBackgroundColor(SCREEN_BLACK);
                }
            });
        }

        @Override
        public void onCameraError(final String error) {
            runOnUiThread(() -> log.warning("On Camera Error: " + error));
        }
    };

    private PeerConnectionClient createPeerConnectionClient() {
        PeerConnectionParameters peerConnectionParameters = new PeerConnectionParameters(
            true, // video call enabled
            false, // loopback
            false, // tracing
            false, // use camera 2
            640, // video width
            480, // video height
            30, // video fps
            550, // video start bitrate 550
            "VP8", // video codec  vp8
            true, // video hardware acceleration
            false, // capture to texture
            0, // audio start bitrate
            "opus", // audio codec
            false, // no audio processing
            false, // AEC dump
            false, // use Open SLES
            false // disable built-in AEC
        );
        PeerConnectionClient peerConnectionClient = PeerConnectionClient.getInstance();
        attachSdk.subscribe(peerConnectionClient);
        peerConnectionClient.createPeerConnectionFactory(
            getApplicationContext(),
            peerConnectionParameters,
            attachSdk.getWebRtcController(),
            cameraListener,
            new RemoteStreamListener()
        );
        return peerConnectionClient;
    }

    private void startVideo() {
        if (peerConnectionClient != null) return;
        this.peerConnectionClient = createPeerConnectionClient();
        audioManager = AppRTCAudioManager.create(getApplicationContext());
        audioManager.start((audioDevice, availableAudioDevices) -> log.debug("Audio mode changed"));

        for (SurfaceViewRenderer renderer : renderers) {
            renderer.init(rootEglBase.getEglBaseContext(), null);
        }
        for (int index = 1; index < renderers.size(); index++) {
            renderers.get(index).setZOrderMediaOverlay(true);
        }

        peerConnectionClient.createPeerConnection(
            rootEglBase.getEglBaseContext(),
            renderers,
            new ArrayList<>()
        );
    }

    private void stopVideo() {
        if (peerConnectionClient != null) {
            attachSdk.unsubscribe(peerConnectionClient);
            peerConnectionClient.releaseRenderers();
            peerConnectionClient.close();
            peerConnectionClient = null;
        }

        if (audioManager != null) {
            audioManager.stop();
            audioManager = null;
        }
    }

    /**
     * Sets the overlay properties when the activity loads.   Override.
     *
     * @return  return the services and properties you wish to set for the overlay
     * @see     <a href="https://documentation.attach.live/android/walkthroughs/overlay#customize-previews-and-colors">Walkthrough</a>
     * @see     <a href="https://github.com/attach-live/attach-sdk-properties/">Properties collection</a>
     */
    protected OverlayActivityProperties withProperties() {
        return null;
    }

    /**
     * Dynamically changes a property of the overlay activity.
     *
     * @param property  the property name
     * @param value     the new value for the property
     */
    protected void setProperty(String property, String value) {
        Map<String, String> map = new HashMap<>();
        map.put(property, value);
        this.setProperties(map);
    }

    /**
     * Dynamically changes multiple properties of the overlay activity.
     *
     * @param properties  a map with key-value pairs of property name and value
     */
    protected void setProperties(Map<String, String> properties) {
        sdk().setProperties(new AttachProperties(properties));
    }

    public interface OnPermissionChangeListener {
        void onPermissionChange(@NonNull String[] permissions, @NonNull int[] grantResults);
    }

    /**
     * Services available in the overlay activity.
     */
    public enum OverlayService {
        VIDEOCALL("videocall"), CHAT("chat");

        private final String service;

        OverlayService(final String service) {
            this.service = service;
        }

        public String getService() {
            return service;
        }
    }

    /**
     * Properties of the overlay activity.
     * @see  <a href="https://github.com/attach-live/attach-sdk-properties/">Properties collection</a>
     */
    public static class OverlayActivityProperties extends Properties {
        /**
         * Class constructor.
         */
        public OverlayActivityProperties() {
            super();
        }

        /**
         * Class constructor specifying properties for the overlay activity.
         *
         * @param properties  a map with key-value pairs of property name and value
         */
        public OverlayActivityProperties(Map<String, String> properties) {
            super(properties);
        }

        /**
         * Class constructor specifying services for the overlay activity.
         *
         * @param services  a list of <code>OverlayService</code>
         */
        public OverlayActivityProperties(AttachOverlayActivity.OverlayService... services) {
            super();
            setServices(services);
        }

        /**
         * Class constructor specifying properties and services for the overlay activity.
         *
         * @param properties  a map with key-value pairs of property name and value
         * @param services    a list of <code>OverlayService</code>
         */
        public OverlayActivityProperties(Map<String, String> properties, AttachOverlayActivity.OverlayService... services) {
            super(properties);
            setServices(services);
        }

        /**
         * Sets services for the overlay activity.   Defaults to all services if omitted.
         *
         * @param services  a list of <code>OverlayService</code>
         */
        public OverlayActivityProperties setServices(AttachOverlayActivity.OverlayService... services) {
            List<String> list = new ArrayList<>();
            for (AttachOverlayActivity.OverlayService service : services) {
                list.add(service.getService());
            }
            String value = TextUtils.join(",", list);
            if (!TextUtils.isEmpty(value)) {
                setProperty(AttachProperties.attachOverlayServices, value);
            }
            return this;
        }
    }
}
