package live.attach.sdk;

import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v7.app.AppCompatActivity;

import java.util.Map;

import live.attach.infrastructure.android.theme.ThemeResolver;
import live.attach.infrastructure.logger.AttachLogger;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.lib.TextUtils;
import live.attach.domain.model.application.AttachProperties;
import live.attach.domain.model.item.ItemId;

/**
 * This is the ATTACH activity.   It is not used in this version.
 * Use the <code>AttachOverlayActivity</code> instead.
 */
public abstract class AttachActivity extends AppCompatActivity {
    private static final AttachLogger log = AttachLoggerFactory.getLogger();

    private AttachSdkInternal attachSdk = (AttachSdkInternal) AttachSdk.getInstance();
    private AttachProperties defaultProperties;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        defaultProperties = ThemeResolver.resolveTheme(this);
        attachSdk.setProperties(defaultProperties);
        attachSdk.setProperties(asAttachProperties(withProperties()));
    }

    @Override
    protected void onStart() {
        super.onStart();
        if (!attachSdk.restartRoom()) {
            attachSdk.enterRoom(asItemId(withRoom()));
        }
    }

    public final AttachSdk getAttachSdk() {
        return attachSdk;
    }

    @Override
    protected void onStop() {
        attachSdk.pauseRoom();
        super.onStop();
    }

    /**
     * Sets the room when the activity loads.   Override. Assigns a room to the activity.
     *
     * @return  return the room to be entered
     * @see     <a href="https://documentation.attach.live/android/concepts/rooms/">Documentation</a>
     */
    protected Room withRoom() {
        return null;
    }

    /**
     * Leaves the room.
     */
    protected final void leaveRoom() {
        attachSdk.pauseRoom();
    }

    /**
     * Dynamically changes the room.
     *
     * @param  room the room you wish to enter
     */
    protected final void enterRoom(Room room) {
        attachSdk.enterRoom(asItemId(room));
    }

    /**
     * Sets the room properties when the activity loads.   Override.
     *
     * @return  return the properties to be set for the activity
     * @see     <a href="https://documentation.attach.live/android/concepts/rooms/properties">Documentation</a>
     * @see     <a href="https://github.com/attach-live/attach-sdk-properties/">Properties collection</a>
     */
    protected Properties withProperties() {
        return null;
    }

    public final AttachProperties getDefaultProperties() {
        return defaultProperties;
    }

    private ItemId asItemId(Room room) {
        if (room == null) return null;
        if (TextUtils.isEmpty(room.url)) {
            return new ItemId(
                room.provider,
                room.type,
                room.identifier
            );
        } else {
            return new ItemId(room.url);
        }
    }

    private AttachProperties asAttachProperties(Properties properties) {
        if (properties == null) return null;

        AttachProperties attachProperties = new AttachProperties();
        for (Map.Entry<String, String> entry : properties.getProperties().entrySet()) {
            attachProperties.set(entry.getKey(), entry.getValue());
        }
        return attachProperties;
    }
}
