package live.attach.infrastructure.octopus.video.webrtc;

/*
 *  Copyright 2014 The WebRTC Project Authors. All rights reserved.
 *
 *  Use of this source code is governed by a BSD-style license
 *  that can be found in the LICENSE file in the root of the source
 *  tree. An additional intellectual property rights grant can be found
 *  in the file PATENTS.  All contributing project authors may
 *  be found in the AUTHORS file in the root of the source tree.
 */

/*
 *  Modified portions Copyright 2016 Closeup, Inc.
 */

import android.content.Context;
import android.os.Environment;
import android.os.ParcelFileDescriptor;
import android.support.annotation.NonNull;

import org.webrtc.AudioSource;
import org.webrtc.AudioTrack;
import org.webrtc.Camera1Enumerator;
import org.webrtc.Camera2Enumerator;
import org.webrtc.CameraEnumerator;
import org.webrtc.CameraVideoCapturer;
import org.webrtc.DataChannel;
import org.webrtc.EglBase;
import org.webrtc.IceCandidate;
import org.webrtc.Logging;
import org.webrtc.MediaConstraints;
import org.webrtc.MediaStream;
import org.webrtc.PeerConnection;
import org.webrtc.PeerConnectionFactory;
import org.webrtc.RtpReceiver;
import org.webrtc.SdpObserver;
import org.webrtc.SessionDescription;
import org.webrtc.StatsObserver;
import org.webrtc.StatsReport;
import org.webrtc.SurfaceViewRenderer;
import org.webrtc.VideoCapturer;
import org.webrtc.VideoSource;
import org.webrtc.VideoTrack;
import org.webrtc.voiceengine.WebRtcAudioManager;
import org.webrtc.voiceengine.WebRtcAudioUtils;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import live.attach.infrastructure.logger.AttachLogger;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.infrastructure.octopus.socket.video.VideoConnectionClient;
import live.attach.infrastructure.octopus.socket.video.WebRtcVideoController;
import live.attach.domain.model.video.Candidate;
import live.attach.domain.model.video.IceServer;
import live.attach.domain.model.video.Sdp;
import live.attach.infrastructure.octopus.video.manager.SurfaceManager;
import live.attach.infrastructure.octopus.video.webrtc.mapper.IceServerConverter;

/**
 * Peer connection client implementation.
 *
 * <p>All public methods are routed to local looper thread.
 * All PeerConnectionEvents callbacks are invoked from the same looper thread.
 * This class is a singleton.
 */
public class PeerConnectionClient implements VideoConnectionClient {
    private static final AttachLogger log = AttachLoggerFactory.getLogger();

    private static final String VIDEO_TRACK_ID = "ARDAMSv0";
    private static final String AUDIO_TRACK_ID = "ARDAMSa0";
    private static final String TAG = PeerConnectionClient.class.getSimpleName();
    private static final String VIDEO_CODEC_VP8 = "VP8";
    private static final String VIDEO_CODEC_VP9 = "VP9";
    private static final String VIDEO_CODEC_H264 = "H264";
    private static final String AUDIO_CODEC_OPUS = "opus";
    private static final String AUDIO_CODEC_ISAC = "ISAC";
    private static final String VIDEO_CODEC_PARAM_START_BITRATE = "x-google-start-bitrate";
    private static final String AUDIO_CODEC_PARAM_BITRATE = "maxaveragebitrate";
    private static final String AUDIO_ECHO_CANCELLATION_CONSTRAINT = "googEchoCancellation";
    private static final String AUDIO_AUTO_GAIN_CONTROL_CONSTRAINT = "googAutoGainControl";
    private static final String AUDIO_HIGH_PASS_FILTER_CONSTRAINT = "googHighpassFilter";
    private static final String AUDIO_NOISE_SUPPRESSION_CONSTRAINT = "googNoiseSuppression";
    private static final String DTLS_SRTP_KEY_AGREEMENT_CONSTRAINT = "DtlsSrtpKeyAgreement";
    private static final int HD_VIDEO_WIDTH = 1280;
    private static final int HD_VIDEO_HEIGHT = 720;
    private static final int MAX_VIDEO_WIDTH = 1280;
    private static final int MAX_VIDEO_HEIGHT = 1280;
    private static final int MAX_VIDEO_FPS = 30;
    private static final PeerConnectionClient instance = new PeerConnectionClient();
    private final PCObserver pcObserver = new PCObserver();
    private final SDPObserver sdpObserver = new SDPObserver();
    private final ScheduledExecutorService executor;
    private Context context;
    private PeerConnectionFactory factory;
    private PeerConnection peerConnection;
    private PeerConnectionFactory.Options options = null;
    private AudioSource audioSource;
    private VideoSource videoSource;
    private boolean videoCallEnabled;
    private boolean preferIsac;
    private String preferredVideoCodec;
    private boolean videoCapturerStopped;
    private boolean iceConnected = false;
    private boolean iceClosed = true;
    private boolean isError;
    private Timer statsTimer;
    private SurfaceManager surfaceManager;
    private List<PeerConnection.IceServer> iceServers;
    private PeerConnection.RTCConfiguration rtcConfig;
    private MediaConstraints pcConstraints;
    private int videoWidth;
    private int videoHeight;
    private int videoFps;
    private MediaConstraints audioConstraints;
    private ParcelFileDescriptor aecDumpFileDescriptor;
    private MediaConstraints sdpMediaConstraints;
    private PeerConnectionParameters peerConnectionParameters;
    private WebRtcVideoController sdpConnector;
    private CameraListener cameraListener;
    private RemoteStreamListener remoteStreamListener;
    private int numberOfCameras;
    private String currentCamera = null;
    private CameraVideoCapturer videoCapturer;
    private VideoTrack localVideoTrack;
    private AudioTrack localAudioTrack;
    private boolean mirrorLocal;
    private boolean enableVideo;
    private boolean enableAudio;

    private PeerConnectionClient() {
        // Executor thread is started once in private ctor and is used for all
        // peer connection API calls to ensure new peer connection factory is
        // created on the same thread as previously destroyed factory.
        executor = Executors.newSingleThreadScheduledExecutor();
    }

    public static PeerConnectionClient getInstance() {
        return instance;
    }

    public void setPeerConnectionFactoryOptions(PeerConnectionFactory.Options options) {
        this.options = options;
    }

    public void createPeerConnectionFactory(
        @NonNull final Context context,
        @NonNull final PeerConnectionParameters peerConnectionParameters,
        @NonNull final WebRtcVideoController sdpConnector,
        @NonNull final CameraListener cameraListener,
        @NonNull final RemoteStreamListener remoteStreamListener) {
        this.peerConnectionParameters = peerConnectionParameters;
        this.sdpConnector = sdpConnector;
        this.cameraListener = cameraListener;
        this.remoteStreamListener = remoteStreamListener;
        videoCallEnabled = peerConnectionParameters.videoCallEnabled;
        // Reset variables to initial states.
        this.context = null;
        factory = null;
        peerConnection = null;
        preferIsac = false;
        videoCapturerStopped = false;
        isError = false;
        videoCapturer = null;
        enableVideo = true;
        enableAudio = true;
        localAudioTrack = null;
        statsTimer = new Timer();
        executor.execute(new Runnable() {
            @Override
            public void run() {
                createPeerConnectionFactoryInternal(context);
            }
        });
    }

    public void createPeerConnection(
        final EglBase.Context renderEGLContext,
        final List<SurfaceViewRenderer> surfaceViewRenderers,
        final List<PeerConnection.IceServer> iceServers
    ) {
        if (peerConnectionParameters == null) {
            log.error("Creating peer connection without initializing factory.");
            return;
        }
        this.iceServers = iceServers;
        this.surfaceManager = new SurfaceManager(surfaceViewRenderers);
        executor.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    createMediaConstraintsInternal();
                    createPeerConnectionInternal(renderEGLContext);
                } catch (Exception e) {
                    reportError("Failed to create peer connection: " + e.getMessage());
                    throw e;
                }
            }
        });
    }

    public void close() {
        executor.execute(new Runnable() {
            @Override
            public void run() {
                closeInternal();
            }
        });
    }

    public boolean isVideoCallEnabled() {
        return videoCallEnabled;
    }

    public boolean isIceConnected() {
        return iceConnected;
    }

    private void createPeerConnectionFactoryInternal(Context context) {
        PeerConnectionFactory.initializeInternalTracer();
        if (peerConnectionParameters.tracing) {
            PeerConnectionFactory.startInternalTracingCapture(
                Environment.getExternalStorageDirectory().getAbsolutePath() + File.separator
                    + "webrtc-trace.txt");
        }
        log.info("Create peer connection factory. Use video: " +
            peerConnectionParameters.videoCallEnabled);
        isError = false;
        // Initialize field trials.
        PeerConnectionFactory.initializeFieldTrials("");
        // Check preferred video codec.
        preferredVideoCodec = VIDEO_CODEC_VP8;
        if (videoCallEnabled && peerConnectionParameters.videoCodec != null) {
            if (peerConnectionParameters.videoCodec.equals(VIDEO_CODEC_VP9)) {
                preferredVideoCodec = VIDEO_CODEC_VP9;
            } else if (peerConnectionParameters.videoCodec.equals(VIDEO_CODEC_H264)) {
                preferredVideoCodec = VIDEO_CODEC_H264;
            }
        }
        log.info("Pereferred video codec: " + preferredVideoCodec);
        // Check if ISAC is used by default.
        preferIsac = peerConnectionParameters.audioCodec != null
            && peerConnectionParameters.audioCodec.equals(AUDIO_CODEC_ISAC);
        // Enable/disable OpenSL ES playback.
        if (!peerConnectionParameters.useOpenSLES) {
            log.info("Disable OpenSL ES audio even if device supports it");
            WebRtcAudioManager.setBlacklistDeviceForOpenSLESUsage(true /* enable */);
        } else {
            log.info("Allow OpenSL ES audio if device supports it");
            WebRtcAudioManager.setBlacklistDeviceForOpenSLESUsage(false);
        }
        if (peerConnectionParameters.disableBuiltInAEC) {
            log.info("Disable built-in AEC even if device supports it");
            WebRtcAudioUtils.setWebRtcBasedAcousticEchoCanceler(true);
        } else {
            log.info("Enable built-in AEC if device supports it");
            WebRtcAudioUtils.setWebRtcBasedAcousticEchoCanceler(false);
        }
        // Create peer connection factory.
        if (!PeerConnectionFactory.initializeAndroidGlobals(context, true, true,
            peerConnectionParameters.videoCodecHwAcceleration)) {
            //sdpConnector.sendPeerConnectionError("Failed to initializeAndroidGlobals");
        }
        if (options != null) {
            log.info("Factory networkIgnoreMask option: " + options.networkIgnoreMask);
        }
        this.context = context;
        factory = new PeerConnectionFactory(options);
        log.info("Peer connection factory created.");
    }

    private void createMediaConstraintsInternal() {
        // Create peer connection constraints.
        pcConstraints = new MediaConstraints();
        // Enable DTLS for normal calls and disable for loopback calls.
        if (peerConnectionParameters.loopback) {
            pcConstraints.optional.add(
                new MediaConstraints.KeyValuePair(DTLS_SRTP_KEY_AGREEMENT_CONSTRAINT, "false"));
        } else {
            pcConstraints.optional.add(
                new MediaConstraints.KeyValuePair(DTLS_SRTP_KEY_AGREEMENT_CONSTRAINT, "true"));
        }
        // Check if there is a camera on device and disable video call if not.
//        numberOfCameras = CameraEnumerationAndroid.getDeviceCount();
//        if (numberOfCameras == 0) {
//            log.warning("No camera on device. Switch to audio only call.");
//            videoCallEnabled = false;
//        }
        // Create video constraints if video call is enabled.
        if (videoCallEnabled) {
            videoWidth = peerConnectionParameters.videoWidth;
            videoHeight = peerConnectionParameters.videoHeight;
            videoFps = peerConnectionParameters.videoFps;
            // If video resolution is not specified, default to HD.
            if (videoWidth == 0 || videoHeight == 0) {
                videoWidth = HD_VIDEO_WIDTH;
                videoHeight = HD_VIDEO_HEIGHT;
            }
            // If fps is not specified, default to 30.
            if (videoFps == 0) {
                videoFps = 30;
            }
            videoWidth = Math.min(videoWidth, MAX_VIDEO_WIDTH);
            videoHeight = Math.min(videoHeight, MAX_VIDEO_HEIGHT);
            videoFps = Math.min(videoFps, MAX_VIDEO_FPS);
        }
        // Create audio constraints.
        audioConstraints = new MediaConstraints();
        // added for audio performance measurements
        if (peerConnectionParameters.noAudioProcessing) {
            log.info("Disabling audio processing");
            audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(
                AUDIO_ECHO_CANCELLATION_CONSTRAINT, "false"));
            audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(
                AUDIO_AUTO_GAIN_CONTROL_CONSTRAINT, "false"));
            audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(
                AUDIO_HIGH_PASS_FILTER_CONSTRAINT, "false"));
            audioConstraints.mandatory.add(new MediaConstraints.KeyValuePair(
                AUDIO_NOISE_SUPPRESSION_CONSTRAINT, "false"));
        }
        // Create SDP constraints.
        sdpMediaConstraints = new MediaConstraints();
        sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair(
            "OfferToReceiveAudio", "true"));
        if (videoCallEnabled || peerConnectionParameters.loopback) {
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair(
                "OfferToReceiveVideo", "true"));
        } else {
            sdpMediaConstraints.mandatory.add(new MediaConstraints.KeyValuePair(
                "OfferToReceiveVideo", "false"));
        }
    }

    private void createCapturer(CameraEnumerator enumerator) {
        final String[] deviceNames = enumerator.getDeviceNames();
        // First, try to find front facing camera
        Logging.d(TAG, "Looking for front facing cameras.");
        for (String deviceName : deviceNames) {
            if (enumerator.isFrontFacing(deviceName)) {
                Logging.d(TAG, "Creating front facing camera capturer.");
                videoCapturer = enumerator.createCapturer(deviceName, cameraEventsHandler);
                mirrorLocal = true;
                if (videoCapturer != null) {
                    return;
                }
            }
        }
        // Front facing camera not found, try something else
        Logging.d(TAG, "Looking for other cameras.");
        for (String deviceName : deviceNames) {
            if (!enumerator.isFrontFacing(deviceName)) {
                Logging.d(TAG, "Creating other camera capturer.");
                videoCapturer = enumerator.createCapturer(deviceName, cameraEventsHandler);
                mirrorLocal = false;
                if (videoCapturer != null) {
                    return;
                }
            }
        }
    }

    private CameraVideoCapturer.CameraEventsHandler cameraEventsHandler = new CameraVideoCapturer.CameraEventsHandler() {
        @Override
        public void onCameraError(String error) {
            currentCamera = null;
            if (cameraListener != null) {
                cameraListener.onCameraError(error);
            }
        }

        @Override
        public void onCameraDisconnected() {

        }

        @Override
        public void onCameraFreezed(String error) {
            currentCamera = null;
            if (cameraListener != null) {
                cameraListener.onCameraError(error);
            }
        }

        @Override
        public void onCameraOpening(String s) {

        }


//        public void onCameraOpening(int camera) {
//            currentCamera = camera;
//        }

        @Override
        public void onFirstFrameAvailable() {
            if (cameraListener != null) {
                cameraListener.onCameraStarted(currentCamera, mirrorLocal);
                surfaceManager.mirrorLocal(mirrorLocal);
            }
        }

        @Override
        public void onCameraClosed() {
            currentCamera = null;
            if (cameraListener != null) {
                cameraListener.onCameraStopped();
            }
        }
    };

    private void createPeerConnectionInternal(EglBase.Context renderEGLContext) {
        if (factory == null || isError) {
            log.error("Peerconnection factory is not created");
            return;
        }
        log.info("Create peer connection.");
        log.info("PCConstraints: " + pcConstraints.toString());
        if (videoCallEnabled) {
            log.info("EGLContext: " + renderEGLContext);
            factory.setVideoHwAccelerationOptions(renderEGLContext, renderEGLContext);
        }
        rtcConfig = new PeerConnection.RTCConfiguration(iceServers);
        // TCP candidates are only useful when connecting to a server that supports
        // ICE-TCP.
        rtcConfig.tcpCandidatePolicy = PeerConnection.TcpCandidatePolicy.DISABLED;
        rtcConfig.bundlePolicy = PeerConnection.BundlePolicy.MAXBUNDLE;
        rtcConfig.rtcpMuxPolicy = PeerConnection.RtcpMuxPolicy.REQUIRE;
        rtcConfig.continualGatheringPolicy = PeerConnection.ContinualGatheringPolicy.GATHER_CONTINUALLY;
        // Use ECDSA encryption.
        rtcConfig.keyType = PeerConnection.KeyType.ECDSA;
        peerConnection = factory.createPeerConnection(
            rtcConfig, pcConstraints, pcObserver);
        // Set default WebRTC tracing and INFO libjingle logging.
        // NOTE: this _must_ happen while |factory| is alive!
//        Logging.enableTracing(
//                "logcat:",
//                EnumSet.of(Logging.TraceLevel.TRACE_DEFAULT));
//        Logging.enableLogToDebugOutput(Logging.Severity.LS_INFO);
        MediaStream localMediaStream = factory.createLocalMediaStream("ARDAMS");
        if (videoCallEnabled) {
            if (peerConnectionParameters.useCamera2) {
                if (!peerConnectionParameters.captureToTexture) {
                    reportError("camera2_texture_only_error");
                    return;
                }
                Logging.d(TAG, "Creating capturer using camera2 API.");
                createCapturer(new Camera2Enumerator(context));
            } else {
                Logging.d(TAG, "Creating capturer using camera1 API.");
                createCapturer(new Camera1Enumerator(peerConnectionParameters.captureToTexture));
            }
            if (videoCapturer == null) {
                reportError("Failed to open camera");
                return;
            }
            localMediaStream.addTrack(createVideoTrack(videoCapturer));
        }
        localMediaStream.addTrack(createAudioTrack());
        surfaceManager.addLocalStream(localMediaStream);
        peerConnection.addStream(localMediaStream);
        if (peerConnectionParameters.aecDump) {
            try {
                aecDumpFileDescriptor = ParcelFileDescriptor.open(
                    new File(Environment.getExternalStorageDirectory().getPath()
                        + File.separator
                        + "Download/audio.aecdump"),
                    ParcelFileDescriptor.MODE_READ_WRITE |
                        ParcelFileDescriptor.MODE_CREATE |
                        ParcelFileDescriptor.MODE_TRUNCATE);
                factory.startAecDump(aecDumpFileDescriptor.getFd(), -1);
            } catch (IOException e) {
                log.error("Can not open aecdump file", e);
            }
        }
        log.info("Peer connection created.");
    }

    private void closeInternal() {
        if (factory != null && peerConnectionParameters.aecDump) {
            factory.stopAecDump();
        }
        log.info("Closing peer connection.");
        statsTimer.cancel();
        if (peerConnection != null) {
            peerConnection.dispose();
            peerConnection = null;
        }
        log.info("Closing audio source.");
        if (audioSource != null) {
            audioSource.dispose();
            audioSource = null;
        }
        log.info("Stopping capture.");
        if (videoCapturer != null) {
            try {
                videoCapturer.stopCapture();
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            videoCapturer.dispose();
            videoCapturer = null;
        }
        log.info("Closing video source.");
        if (videoSource != null) {
            videoSource.dispose();
            videoSource = null;
        }
        log.info("Closing peer connection factory.");
        if (factory != null) {
            factory.dispose();
            factory = null;
        }
        options = null;
        log.info("Closing peer connection done.");
        PeerConnectionFactory.stopInternalTracingCapture();
        PeerConnectionFactory.shutdownInternalTracer();
        this.context = null;
    }

    public boolean isHDVideo() {
        if (!videoCallEnabled) {
            return false;
        }
        return videoWidth * videoHeight >= 1280 * 720;
    }

    private void getStats() {
        if (peerConnection == null || isError) {
            return;
        }
        boolean success = peerConnection.getStats(new StatsObserver() {
            @Override
            public void onComplete(final StatsReport[] reports) {
            }
        }, null);
        if (!success) {
            log.error("getStats() returns false!");
        }
    }

    public void enableStatsEvents(boolean enable, int periodMs) {
        if (enable) {
            try {
                statsTimer.schedule(new TimerTask() {
                    @Override
                    public void run() {
                        executor.execute(new Runnable() {
                            @Override
                            public void run() {
                                getStats();
                            }
                        });
                    }
                }, 0, periodMs);
            } catch (Exception e) {
                log.error("Can not schedule statistics timer", e);
            }
        } else {
            statsTimer.cancel();
        }
    }

    public void updateIceServers(List<PeerConnection.IceServer> iceServers) {
        if (this.rtcConfig == null) { // TODO this should be a while loop with a much shorter sleep time
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
            }
        }
        this.iceServers = iceServers;
        // TODO Nullpointer here
        this.rtcConfig.iceServers = iceServers;
        executor.execute(new Runnable() {
            @Override
            public void run() {
                if (peerConnection != null && !isError) {
                    log.info("PC Update IceServers");
                    peerConnection.setConfiguration(rtcConfig);
                }
            }
        });
    }

    public void createOffer() {
        executor.execute(() -> {
            if (peerConnection != null && !isError) {
                log.info("PC Create OFFER");
                peerConnection.createOffer(sdpObserver, sdpMediaConstraints);
            }
        });
    }

    public void createAnswer() {
        executor.execute(() -> {
            if (peerConnection != null && !isError) {
                log.info("PC create ANSWER");
                peerConnection.createAnswer(sdpObserver, sdpMediaConstraints);
            }
        });
    }

    public void addRemoteIceCandidate(final IceCandidate candidate) {
        executor.execute(() -> {
            if (peerConnection != null && !isError) {
                peerConnection.addIceCandidate(candidate);
            }
        });
    }

    public void removeRemoteIceCandidates(final IceCandidate[] candidates) {
        executor.execute(() -> {
            if (peerConnection == null || isError) {
                return;
            }
            peerConnection.removeIceCandidates(candidates);
        });
    }

    public void setRemoteDescription(final SessionDescription sessionDescription) {
        log.info("set remote description");
        executor.execute(() -> {
            if (peerConnection == null || isError) {
                return;
            }

            String description = sessionDescription.description;
            if (preferIsac) {
                description = preferCodec(description, AUDIO_CODEC_ISAC, true);
            }
            if (videoCallEnabled) {
                description = preferCodec(description, preferredVideoCodec, false);
            }
            if (videoCallEnabled && peerConnectionParameters.videoStartBitrate > 0) {
                description = setStartBitrate(VIDEO_CODEC_VP8, true,
                    description, peerConnectionParameters.videoStartBitrate);
                description = setStartBitrate(VIDEO_CODEC_VP9, true,
                    description, peerConnectionParameters.videoStartBitrate);
                description = setStartBitrate(VIDEO_CODEC_H264, true,
                    description, peerConnectionParameters.videoStartBitrate);
            }
            if (peerConnectionParameters.audioStartBitrate > 0) {
                description = setStartBitrate(AUDIO_CODEC_OPUS, false,
                    description, peerConnectionParameters.audioStartBitrate);
            }
            log.info("Set remote SDP.");
            SessionDescription sdpRemote = new SessionDescription(sessionDescription.type, description);
            peerConnection.setRemoteDescription(sdpObserver, sdpRemote);
        });
    }

    public void stopVideoSource() {
        executor.execute(() -> {
            if (videoCapturer != null && !videoCapturerStopped) {
                log.info("Stop video source.");
                try {
                    videoCapturer.stopCapture();
                } catch (InterruptedException e) {
                }
                videoCapturerStopped = true;
            }
        });
    }

    public void startVideoSource() {
        executor.execute(() -> {
            if (videoCapturer != null && videoCapturerStopped) {
                log.info("Restart video source.");
                videoCapturer.startCapture(videoWidth, videoHeight, videoFps);
                videoCapturerStopped = false;
            }
        });
    }

    private void reportError(final String errorMessage) {
        log.error("Peerconnection error: " + errorMessage);
        executor.execute(() -> {
            if (!isError) {
                // TODO we do nothing right now
                isError = true;
            }
        });
    }

    private AudioTrack createAudioTrack() {
        audioSource = factory.createAudioSource(audioConstraints);
        localAudioTrack = factory.createAudioTrack(AUDIO_TRACK_ID, audioSource);
        localAudioTrack.setEnabled(enableAudio);
        return localAudioTrack;
    }

    private VideoTrack createVideoTrack(VideoCapturer capturer) {
        videoSource = factory.createVideoSource(capturer);
        capturer.startCapture(videoWidth, videoHeight, videoFps);
        localVideoTrack = factory.createVideoTrack(VIDEO_TRACK_ID, videoSource);
        localVideoTrack.setEnabled(enableVideo);
        return localVideoTrack;
    }

    public void enableLocalStreams(boolean enabled) {
        localVideoTrack.setEnabled(enabled && enableVideo);
        localAudioTrack.setEnabled(enabled && enableAudio);
    }

    private static String setStartBitrate(String codec, boolean isVideoCodec,
                                          String sdpDescription, int bitrateKbps) {
        String[] lines = sdpDescription.split("\r\n");
        int rtpmapLineIndex = -1;
        boolean sdpFormatUpdated = false;
        String codecRtpMap = null;
        // Search for codec rtpmap in format
        // a=rtpmap:<payload type> <encoding name>/<clock rate> [/<encoding parameters>]
        String regex = "^a=rtpmap:(\\d+) " + codec + "(/\\d+)+[\r]?$";
        Pattern codecPattern = Pattern.compile(regex);
        for (int i = 0; i < lines.length; i++) {
            Matcher codecMatcher = codecPattern.matcher(lines[i]);
            if (codecMatcher.matches()) {
                codecRtpMap = codecMatcher.group(1);
                rtpmapLineIndex = i;
                break;
            }
        }
        if (codecRtpMap == null) {
            log.warning("No rtpmap for " + codec + " codec");
            return sdpDescription;
        }
        log.info("Found " + codec + " rtpmap " + codecRtpMap
            + " at " + lines[rtpmapLineIndex]);
        // Check if a=fmtp string already exist in remote SDP for this codec and
        // update it with new bitrate parameter.
        regex = "^a=fmtp:" + codecRtpMap + " \\w+=\\d+.*[\r]?$";
        codecPattern = Pattern.compile(regex);
        for (int i = 0; i < lines.length; i++) {
            Matcher codecMatcher = codecPattern.matcher(lines[i]);
            if (codecMatcher.matches()) {
                log.info("Found " + codec + " " + lines[i]);
                if (isVideoCodec) {
                    lines[i] += "; " + VIDEO_CODEC_PARAM_START_BITRATE
                        + "=" + bitrateKbps;
                } else {
                    lines[i] += "; " + AUDIO_CODEC_PARAM_BITRATE
                        + "=" + (bitrateKbps * 1000);
                }
                log.info("Update remote SDP line: " + lines[i]);
                sdpFormatUpdated = true;
                break;
            }
        }
        StringBuilder newSdpDescription = new StringBuilder();
        for (int i = 0; i < lines.length; i++) {
            newSdpDescription.append(lines[i]).append("\r\n");
            // Append new a=fmtp line if no such line exist for a codec.
            if (!sdpFormatUpdated && i == rtpmapLineIndex) {
                String bitrateSet;
                if (isVideoCodec) {
                    bitrateSet = "a=fmtp:" + codecRtpMap + " "
                        + VIDEO_CODEC_PARAM_START_BITRATE + "=" + bitrateKbps;
                } else {
                    bitrateSet = "a=fmtp:" + codecRtpMap + " "
                        + AUDIO_CODEC_PARAM_BITRATE + "=" + (bitrateKbps * 1000);
                }
                log.info("Add remote SDP line: " + bitrateSet);
                newSdpDescription.append(bitrateSet).append("\r\n");
            }
        }
        return newSdpDescription.toString();
    }

    private static String preferCodec(
        String sdpDescription, String codec, boolean isAudio) {
        String[] lines = sdpDescription.split("\r\n");
        int mLineIndex = -1;
        String codecRtpMap = null;
        // a=rtpmap:<payload type> <encoding name>/<clock rate> [/<encoding parameters>]
        String regex = "^a=rtpmap:(\\d+) " + codec + "(/\\d+)+[\r]?$";
        Pattern codecPattern = Pattern.compile(regex);
        String mediaDescription = "m=video ";
        if (isAudio) {
            mediaDescription = "m=audio ";
        }
        for (int i = 0; (i < lines.length)
            && (mLineIndex == -1 || codecRtpMap == null); i++) {
            if (lines[i].startsWith(mediaDescription)) {
                mLineIndex = i;
                continue;
            }
            Matcher codecMatcher = codecPattern.matcher(lines[i]);
            if (codecMatcher.matches()) {
                codecRtpMap = codecMatcher.group(1);
            }
        }
        if (mLineIndex == -1) {
            log.warning("No " + mediaDescription + " line, so can't prefer " + codec);
            return sdpDescription;
        }
        if (codecRtpMap == null) {
            log.warning("No rtpmap for " + codec);
            return sdpDescription;
        }
        log.info("Found " + codec + " rtpmap " + codecRtpMap + ", prefer at "
            + lines[mLineIndex]);
        String[] origMLineParts = lines[mLineIndex].split(" ");
        if (origMLineParts.length > 3) {
            StringBuilder newMLine = new StringBuilder();
            int origPartIndex = 0;
            // Format is: m=<media> <port> <proto> <fmt> ...
            newMLine.append(origMLineParts[origPartIndex++]).append(" ");
            newMLine.append(origMLineParts[origPartIndex++]).append(" ");
            newMLine.append(origMLineParts[origPartIndex++]).append(" ");
            newMLine.append(codecRtpMap);
            for (; origPartIndex < origMLineParts.length; origPartIndex++) {
                if (!origMLineParts[origPartIndex].equals(codecRtpMap)) {
                    newMLine.append(" ").append(origMLineParts[origPartIndex]);
                }
            }
            lines[mLineIndex] = newMLine.toString();
            log.info("Change media description: " + lines[mLineIndex]);
        } else {
            log.error("Wrong SDP media description format: " + lines[mLineIndex]);
        }
        StringBuilder newSdpDescription = new StringBuilder();
        for (String line : lines) {
            newSdpDescription.append(line).append("\r\n");
        }
        return newSdpDescription.toString();
    }

    private void switchCameraInternal() {
        if (!videoCallEnabled || isError || videoCapturer == null) {
//        if (!videoCallEnabled || numberOfCameras < 2 || isError || videoCapturer == null) {
            log.error("Failed to switch camera. Video: " + videoCallEnabled + ". Error : "
                + isError + ". Number of cameras: " + numberOfCameras);
            return;  // No video is sent or only one camera is available or error happened.
        }
        log.info("Switch camera");
        videoCapturer.switchCamera(null);
        mirrorLocal = !mirrorLocal;
        surfaceManager.mirrorLocal(mirrorLocal);
    }

    public void switchCamera() {
        executor.execute(new Runnable() {
            @Override
            public void run() {
                switchCameraInternal();
            }
        });
    }

    public void changeCaptureFormat(final int width, final int height, final int framerate) {
        executor.execute(new Runnable() {
            @Override
            public void run() {
                changeCaptureFormatInternal(width, height, framerate);
            }
        });
    }

    private void changeCaptureFormatInternal(int width, int height, int framerate) {
        if (!videoCallEnabled || isError || videoCapturer == null) {
            log.error("Failed to change capture format. Video: " + videoCallEnabled + ". Error : "
                + isError);
            return;
        }
        log.info("changeCaptureFormat: " + width + "x" + height + "@" + framerate);
//        videoCapturer.onOutputFormatRequest(width, height, framerate);
        videoCapturer.changeCaptureFormat(width, height, framerate);
    }

//    public void updateRenderers(List<SurfaceViewRenderer> renderers) {
//        this.surfaceManager.updateRenderers(renderers);
//    }

    public void releaseRenderers() {
        this.surfaceManager.releaseRenderers();
    }

    @Override
    public void onParticipantOffer(String socketId, Sdp sdp, List<IceServer> iceServers) {
        updateIceServers(IceServerConverter.toPcIceServers(iceServers));

        SessionDescription sessionDescription = new SessionDescription(
            SessionDescription.Type.valueOf(sdp.type.toString().toUpperCase()),
            sdp.description
        );
        setRemoteDescription(sessionDescription);
        createAnswer();
    }

    @Override
    public void onParticipantAnswer(String socketId, Sdp sdp) {
        SessionDescription sessionDescription = new SessionDescription(
            SessionDescription.Type.valueOf(sdp.type.toString().toUpperCase()),
            sdp.description
        );
        setRemoteDescription(sessionDescription);
    }

    @Override
    public void onParticipantCandidate(String socketId, Candidate candidate) {
        addRemoteIceCandidate(new IceCandidate(
            candidate.sdpMid,
            candidate.sdpMLineIndex,
            candidate.description
        ));
    }

    // Implementation detail: observe ICE & stream changes and react accordingly.
    private class PCObserver implements PeerConnection.Observer {
        @Override
        public void onIceCandidate(final IceCandidate candidate) {
            executor.execute(new Runnable() {
                @Override
                public void run() {
                    sdpConnector.sendIceCandidate(new Candidate(
                        candidate.sdp,
                        candidate.sdpMid,
                        candidate.sdpMLineIndex
                    ));
                }
            });
        }

        @Override
        public void onIceCandidatesRemoved(final IceCandidate[] candidates) {
//            executor.execute(new Runnable() {
//                @Override
//                public void run() {
//                    sdpConnector.onIceCandidatesRemoved(candidates);
//                }
//            });
        }

        @Override
        public void onSignalingChange(PeerConnection.SignalingState newState) {
        }

        @Override
        public void onIceConnectionChange(
            final PeerConnection.IceConnectionState newState) {
//            executor.execute(new Runnable() {
//                @Override
//                public void run() {
//                    log.info("IceConnectionState: " + newState);
//                    switch (newState) {
//                        case CONNECTED:
//                            iceConnected = true;
//                            sdpConnector.onIceConnected();
//                            if (remoteStreamListener != null) {
//                                remoteStreamListener.onIceConnected();
//                            }
//                            break;
//                        case DISCONNECTED:
//                            iceConnected = false;
//                            sdpConnector.onIceDisconnected();
//                            break;
//                        case CLOSED:
//                            iceClosed = true;
//                            if (remoteStreamListener != null) {
//                                remoteStreamListener.onIceDisconnected();
//                            }
//                            break;
//                        case FAILED:
//                            reportError("ICE connection failed.");
//                            break;
//                    }
//                }
//            });
        }

        @Override
        public void onIceGatheringChange(
            PeerConnection.IceGatheringState newState) {
            log.info("IceGatheringState: " + newState);
        }

        @Override
        public void onIceConnectionReceivingChange(boolean receiving) {
            log.info("IceConnectionReceiving changed to " + receiving);
        }

        @Override
        public void onAddStream(final MediaStream stream) {
            executor.execute(new Runnable() {
                @Override
                public void run() {
                    if (peerConnection == null || isError) {
                        return;
                    }
                    if (stream.audioTracks.size() > 1 || stream.videoTracks.size() > 1) {
                        reportError("Weird-looking stream: " + stream);
                        return;
                    }
                    if (stream.videoTracks.size() == 1) {
                        surfaceManager.addRemoteStream(stream);
                        if (remoteStreamListener != null) {
                            remoteStreamListener.onAddStream(stream);
                        }
                    }
                }
            });
        }

        @Override
        public void onRemoveStream(final MediaStream stream) {
            executor.execute(new Runnable() {
                @Override
                public void run() {
                    if (peerConnection == null || isError) {
                        return;
                    }
                    surfaceManager.removeRemoteStream(stream);
                    if (remoteStreamListener != null) {
                        remoteStreamListener.onRemoveStream(stream);
                    }
                }
            });
        }

        @Override
        public void onDataChannel(final DataChannel dataChannel) {
            if (remoteStreamListener != null) {
                remoteStreamListener.onDataChannel(dataChannel);
            }
        }

        @Override
        public void onRenegotiationNeeded() {
            // No need to do anything; AppRTC follows a pre-agreed-upon
            // signaling/negotiation protocol.
        }

        @Override
        public void onAddTrack(RtpReceiver rtpReceiver, MediaStream[] mediaStreams) {

        }
    }

    // Implementation detail: handle offer creation/signaling and answer setting,
    // as well as adding remote ICE candidates once the answer SDP is set.
    private class SDPObserver implements SdpObserver {
        @Override
        public void onCreateSuccess(SessionDescription sessionDescription) {
            String sdp = sessionDescription.description;
            if (preferIsac) {
                sdp = preferCodec(sdp, AUDIO_CODEC_ISAC, true);
            }
            if (videoCallEnabled) {
                sdp = preferCodec(sdp, preferredVideoCodec, false);
            }
            final SessionDescription patchedSessionDescription = new SessionDescription(sessionDescription.type, sdp);
            executor.execute(new Runnable() {
                @Override
                public void run() {
                    if (peerConnection != null && !isError) {
                        log.info("Set local SDP from " + patchedSessionDescription.type);
                        peerConnection.setLocalDescription(sdpObserver, patchedSessionDescription);
                        switch (patchedSessionDescription.type) {
                            case OFFER:
                                sdpConnector.sendOfferCreated(patchedSessionDescription.description);
                                break;
                            case ANSWER:
                                sdpConnector.sendAnswerCreated(patchedSessionDescription.description);
                                break;
                        }
                    }
                }
            });
        }

        @Override
        public void onSetSuccess() {
            executor.execute(() -> {
                // TODO we do nothing
            });
        }

        @Override
        public void onCreateFailure(final String error) {
            reportError("createSDP error: " + error);
        }

        @Override
        public void onSetFailure(final String error) {
            reportError("setSDP error: " + error);
        }
    }
}
