package live.attach.infrastructure.octopus.video.manager;

import android.util.Log;

import org.webrtc.MediaStream;
import org.webrtc.RendererCommon;
import org.webrtc.SurfaceViewRenderer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

import static live.attach.infrastructure.octopus.video.manager.SurfaceManager.State.INIT;
import static live.attach.infrastructure.octopus.video.manager.SurfaceManager.State.INIT_LOCAL;
import static live.attach.infrastructure.octopus.video.manager.SurfaceManager.State.LOCAL;
import static live.attach.infrastructure.octopus.video.manager.SurfaceManager.State.REMOTE;

public class SurfaceManager {

    public static final int TICK_START_DELAY_MSEC = 4000;
    public static final int TICK_FREQUENCY_MSEC = 1500;
    public static final int TICKS_TO_SWITCH_LOCAL_STREAM = 1;
    public static final int TICKS_TO_SWITCH_REMOTE_STREAM = 2;
    public static final int TICKS_TO_SWITCH_LATEST_STREAM = 4;

    private State state;

    enum State {
        INIT,
        INIT_LOCAL, // workaround: must add stream to all renderers during setup
        LOCAL,
        REMOTE
    }

    private void assertState(State... states) {
        if (!Arrays.asList(states).contains(state)) {
            throw new RuntimeException("Wrong state.");
        }
    }

    private List<SurfaceViewRenderer> surfaceViewRenderers;
    private Integer[] positions;
    private List<Boolean> visibility;
    private boolean mirrorLocal = true;
    private List<ManagedStream> managedStreams = new LinkedList<>(); // 0 = local; 1, 2, 3 = remote
    private ManagedStream dummyStream;

    public SurfaceManager(List<SurfaceViewRenderer> surfaceViewRenderers) {
        this.state = INIT;
        initRenderers(surfaceViewRenderers);
    }

    public synchronized SurfaceManager releaseRenderers() {
        stopTimer();
        if (surfaceViewRenderers == null) {
            throw new RuntimeException("surfaceViewRenderers already released.");
        }

        for (ManagedStream managedStream : managedStreams) {
            managedStream.removeAllSurfaces();
        }
        if (dummyStream != null) {
            dummyStream.removeAllSurfaces();
        }
        for (SurfaceViewRenderer surfaceViewRenderer : surfaceViewRenderers) {
            surfaceViewRenderer.release();
        }
        surfaceViewRenderers = null;
        reassignSurfaces();
        return this;
    }

    public synchronized SurfaceManager updateRenderers(List<SurfaceViewRenderer> surfaceViewRenderers) {
        if (this.surfaceViewRenderers != null) {
            throw new RuntimeException("surfaceViewRenderers not released.");
        }
        if (surfaceViewRenderers.size() < 1) {
            throw new RuntimeException("At least one surfaceViewRenderer required.");
        }
        this.surfaceViewRenderers = surfaceViewRenderers;
        if (dummyStream != null) {
            dummyStream.addSurfaceDisabled(surfaceViewRenderers.get(0));
        }
        initRenderers(surfaceViewRenderers);
        return this;
    }

    private void initRenderers(List<SurfaceViewRenderer> surfaceViewRenderers) {
        for (SurfaceViewRenderer surfaceViewRenderer : surfaceViewRenderers) {
            surfaceViewRenderer.setMirror(mirrorLocal);
            surfaceViewRenderer.setScalingType(RendererCommon.ScalingType.SCALE_ASPECT_FILL);
            surfaceViewRenderer.requestLayout();
        }
        this.surfaceViewRenderers = surfaceViewRenderers;
        reassignSurfaces();
        startTimer();
    }

    public synchronized SurfaceManager mirrorLocal(boolean enabled) {
        mirrorLocal = enabled;
        ManagedStream managedStream = getLocalStream();
        for (SurfaceViewRenderer surfaceViewRenderer : managedStream.getSurfaces()) {
            surfaceViewRenderer.setMirror(mirrorLocal);
        }
        return this;
    }

    public synchronized SurfaceManager addLocalStream(MediaStream localStream) {
        Log.d("VIDEO_TAG", "ADD LOCAL STREAM");
        assertState(INIT);
        state = INIT_LOCAL;
        ManagedStream managedStream = new ManagedStream(localStream);
        managedStream.isLocal = true;
        managedStreams.add(managedStream);
        managedStream.position = 0;
        SurfaceViewRenderer localSurfaceViewRenderer = surfaceViewRenderers.get(0);
        managedStream.addSurface(localSurfaceViewRenderer);
        reassignSurfaces();
        return this;
    }

    public synchronized SurfaceManager addRemoteStream(MediaStream remoteStream) {
        Log.d("VIDEO_TAG", "ADD REMOTE STREAM");
        assertState(INIT_LOCAL, LOCAL, REMOTE);
        switch (state) {
            case INIT_LOCAL:
                if ("mcu".equals(remoteStream.label())) {
                    // ManagedStream localManagedStream = getLocalStream();
                    SurfaceViewRenderer localSurfaceViewRenderer = surfaceViewRenderers.get(0);
                    // avoid warnings
                    dummyStream = new ManagedStream(remoteStream);
                    dummyStream.addSurfaceDisabled(localSurfaceViewRenderer);
                    state = LOCAL;
                    break;
                } else {
                    state = REMOTE;
                    // continue
                }
                // continue
            case LOCAL:
            case REMOTE:
                ManagedStream remoteManagedStream = new ManagedStream(remoteStream);
                for (ManagedStream managedStream : managedStreams) {
                    managedStream.isLatest = false;
                }
                remoteManagedStream.isLatest = true;
                managedStreams.add(remoteManagedStream);
                remoteManagedStream.position = managedStreams.size() - 1;
                rotate();
                state = REMOTE;
                break;
        }
        return this;
    }

    public synchronized SurfaceManager removeRemoteStream(MediaStream remoteStream) {
        Log.d("VIDEO_TAG", "REMOVE REMOTE STREAM");
        assertState(REMOTE);
        List<ManagedStream> remoteManagedStreams = getRemoteStreams();
        for (ManagedStream managedStream : remoteManagedStreams) {
            if (remoteStream.equals(managedStream.getStream())) {
                managedStreams.remove(managedStream);
                int position = managedStream.position;
                for (ManagedStream managedStream1 : managedStreams) {
                    if (managedStream1.position > position) {
                        managedStream1.position = managedStream1.position - 1;
                    }
                }
                managedStream.removeAllSurfaces();
                reassignSurfaces();
                break;
            }
        }
        return this;
    }

    public synchronized ManagedStream getLocalStream() {
        for (ManagedStream managedStream : managedStreams) {
            if (managedStream.isLocal) {
                return managedStream;
            }
        }
        throw new RuntimeException("No local stream.");
    }

    public synchronized List<ManagedStream> getRemoteStreams() {
        List<ManagedStream> managedStreams = new LinkedList<>(this.managedStreams);
        Iterator<ManagedStream> iterator = managedStreams.iterator();
        while (iterator.hasNext()) {
            ManagedStream managedStream = iterator.next();
            if (managedStream.isLocal) {
                iterator.remove();
            }
        }
        return managedStreams;
    }

    public synchronized ManagedStream getLatestStream() {
        for (ManagedStream managedStream : managedStreams) {
            if (managedStream.isLatest) {
                return managedStream;
            }
        }
        return null;
    }

    public synchronized List<ManagedStream> getRemoteStreamsExceptLatest() {
        List<ManagedStream> managedStreams = new LinkedList<>(this.managedStreams);
        Iterator<ManagedStream> iterator = managedStreams.iterator();
        while (iterator.hasNext()) {
            ManagedStream managedStream = iterator.next();
            if (managedStream.isLocal || managedStream.isLatest) {
                iterator.remove();
            }
        }
        return managedStreams;
    }

    private synchronized void reassignSurfaces() {
        int renderersSize = surfaceViewRenderers != null ? surfaceViewRenderers.size() : 0;
        int visibleRenderersSize = Math.min(renderersSize, managedStreams.size());
        for (int index = 0; index < visibleRenderersSize; index++) {
            ManagedStream managedStream = managedStreams.get(index);
            SurfaceViewRenderer surfaceViewRenderer = surfaceViewRenderers.get(index);
            managedStream.replaceAllSurfacesWith(surfaceViewRenderer);
        }
        for (int index = visibleRenderersSize; index < renderersSize; index++) {
            SurfaceViewRenderer surfaceViewRenderer = surfaceViewRenderers.get(index);
            for (ManagedStream managedStream : managedStreams) {
                managedStream.removeSurface(surfaceViewRenderer);
            }
        }
        for (int index = visibleRenderersSize; index < managedStreams.size(); index++) {
            ManagedStream managedStream = managedStreams.get(index);
            managedStream.removeAllSurfaces();
        }

        // listener
        Integer[] positions = new Integer[visibleRenderersSize];
        for (int index = 0; index < visibleRenderersSize; index++) {
            positions[index] = managedStreams.get(index).position;
        }
        if (!Arrays.equals(this.positions, positions)) {
            Log.d("VIDEO_TAG", "RENDERER POSITIONS = " + Arrays.toString(positions));
            if (listener != null) {
                List<Boolean> visibility = new ArrayList<>(renderersSize);
                for (int index = 0; index < renderersSize ; index++) {
                    visibility.add(index < visibleRenderersSize);
                }
                if (!visibility.equals(this.visibility)) {
                    Log.d("VIDEO_TAG", "VISIBILITY = " + Arrays.toString(visibility.toArray()));
                    listener.updateLayout(visibility);
                    this.visibility = visibility;
                }
            }
            this.positions = positions;
        }
    }
    
    private synchronized void maybeRotate() {
        ManagedStream managedStream = managedStreams.get(0);
        managedStream.ticksUntilRotate--;
        if (managedStream.ticksUntilRotate <= 0) {
            rotate();
        }
    }

    private int swapStream = 0;

    private synchronized void rotate() {
        if (surfaceViewRenderers == null) {
            return;
        }

        Log.d("VIDEO_TAG", "ROTATE");
        int streamsSize = managedStreams.size();
        int renderersSize = surfaceViewRenderers.size();
        if (streamsSize > 1) {
            if (renderersSize == 2) {
                // duo
                if (streamsSize == 2) {
                    // local and remote
                    if (!managedStreams.get(0).isLocal) {
                        ManagedStream lastManagedStream = managedStreams.remove(streamsSize - 1);
                        managedStreams.add(0, lastManagedStream);
                    }
                    swapStream = 1;
                } else if (streamsSize == 3) {
                    // local plus 2 remote streams
                    ManagedStream managedStream0 = managedStreams.get(0);
                    ManagedStream managedStream1 = managedStreams.get(1);
                    if (!(managedStream0.isLocal || managedStream1.isLocal)) {
                        // both remote
                        ManagedStream swappedStream;
                        if (swapStream == 0) {
                            swappedStream = managedStream0;
                            swapStream = 1;
                        } else {
                            swappedStream = managedStream1;
                            swapStream = 0;
                        }
                        ManagedStream lastManagedStream = managedStreams.remove(streamsSize - 1);
                        managedStreams.add(managedStreams.indexOf(swappedStream), lastManagedStream);
                        managedStreams.remove(swappedStream);
                        managedStreams.add(swappedStream);
                    } else {
                        for (ManagedStream managedStream : managedStreams) {
                            if (managedStream.isLocal) {
                                // swap local
                                ManagedStream lastManagedStream = managedStreams.remove(streamsSize - 1);
                                managedStreams.add(managedStreams.indexOf(managedStream), lastManagedStream);
                                managedStreams.remove(managedStream);
                                managedStreams.add(managedStream);
                                break;
                            }
                        }
                    }
                } else {
                    // more streams
                    ManagedStream lastManagedStream = managedStreams.remove(streamsSize - 1);
                    ManagedStream swappedStream;
                    if (swapStream == 0) {
                        swappedStream = managedStreams.remove(0);
                        managedStreams.add(0, lastManagedStream);
                        swapStream = 1;
                    } else {
                        swappedStream = managedStreams.remove(1);
                        managedStreams.add(1, lastManagedStream);
                        swapStream = 0;
                    }
                    managedStreams.add(2, swappedStream);
                }
                // schedule next rotate
                ManagedStream managedStream = managedStreams.get(0);
                managedStream.ticksUntilRotate = SurfaceManager.TICKS_TO_SWITCH_LOCAL_STREAM;
            } else {
                // single, multi
                ManagedStream lastManagedStream = managedStreams.remove(streamsSize - 1);
                managedStreams.add(0, lastManagedStream);
                if (renderersSize != 1) {
                    // multi
                    for (ManagedStream managedStream : managedStreams) {
                        if (managedStream.isLocal) {
                            // keep local stream at first small surface if multiple surfaces
                            managedStreams.remove(managedStream);
                            managedStreams.add(1, managedStream);
                            break;
                        }
                    }
                }
                // schedule next rotate
                ManagedStream managedStream = managedStreams.get(0);
                if (managedStream.isLocal) {
                    managedStream.ticksUntilRotate = SurfaceManager.TICKS_TO_SWITCH_LOCAL_STREAM;
                } else if (managedStream.isLatest) {
                    managedStream.ticksUntilRotate = SurfaceManager.TICKS_TO_SWITCH_LATEST_STREAM;
                } else {
                    managedStream.ticksUntilRotate = SurfaceManager.TICKS_TO_SWITCH_REMOTE_STREAM;
                }
            }
        }
        reassignSurfaces();
    }

    private Listener listener;

    public interface Listener {
        void updateLayout(List<Boolean> visibility);
    }

    private Timer timer = null;

    private void startTimer() {
        stopTimer();
        timer = new Timer();
        timer.scheduleAtFixedRate(new TimerTask() {
                                      @Override
                                      public void run() {
                                          maybeRotate();
                                      }
                                  },
                TICK_START_DELAY_MSEC,
                TICK_FREQUENCY_MSEC
        );
    }

    private void stopTimer() {
        if (timer != null) {
            timer.cancel();
            timer = null;
        }
    }
}
