package live.attach.infrastructure.octopus.video.manager;

import org.webrtc.MediaStream;
import org.webrtc.SurfaceViewRenderer;
import org.webrtc.VideoRenderer;
import org.webrtc.VideoTrack;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

public class ManagedStream {
    private MediaStream stream;
    private Map<SurfaceViewRenderer, VideoRenderer> renderers = new HashMap<>();

    // managed by SurfaceManager
    boolean isLocal = false;
    boolean isLatest = false;
    Integer position = null;
    int ticksUntilRotate = 0;

    ManagedStream(MediaStream mediaStream) {
        this.stream = mediaStream;
    }

    ManagedStream addSurface(SurfaceViewRenderer surfaceViewRenderer) {
        addSurfaceDisabled(surfaceViewRenderer);
        enableStream();
        return this;
    }

    ManagedStream addSurfaceDisabled(SurfaceViewRenderer surfaceViewRenderer) {
        VideoRenderer videoRenderer = new VideoRenderer(surfaceViewRenderer);
        stream.videoTracks.get(0).addRenderer(videoRenderer);
        renderers.put(surfaceViewRenderer, videoRenderer);
        return this;
    }

    ManagedStream enableStream() {
        if (stream.videoTracks.size() > 0) {
            VideoTrack videoTrack = stream.videoTracks.get(0);
            if (!videoTrack.enabled()) {
                videoTrack.setEnabled(true);
            }
        }
        return this;
    }

    ManagedStream disableStream() {
        if (!isLocal && stream.videoTracks.size() > 0) {
            VideoTrack videoTrack = stream.videoTracks.get(0);
            if (videoTrack.enabled()) {
                videoTrack.setEnabled(false);
            }
        }
        return this;
    }

    ManagedStream removeSurface(SurfaceViewRenderer surfaceViewRenderer) {
        VideoRenderer videoRenderer = renderers.remove(surfaceViewRenderer);
        if (videoRenderer != null) {
            if (stream.videoTracks.size() > 0) {
                disableStream();
                VideoTrack videoTrack = stream.videoTracks.get(0);
                videoTrack.removeRenderer(videoRenderer);
            }
        }
        return this;
    }

    ManagedStream removeAllSurfaces() {
        if (stream.videoTracks.size() > 0) {
            disableStream();
            VideoTrack videoTrack = stream.videoTracks.get(0);
            for (VideoRenderer videoRenderer : renderers.values()) {
                videoTrack.removeRenderer(videoRenderer);
            }
        }
        renderers.clear();
        return this;
    }

    ManagedStream removeAllSurfacesExcept(SurfaceViewRenderer surfaceViewRenderer) {
        if (stream.videoTracks.size() > 0) {
            VideoTrack videoTrack = stream.videoTracks.get(0);
            Iterator<Map.Entry<SurfaceViewRenderer, VideoRenderer>> iterator = renderers.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<SurfaceViewRenderer, VideoRenderer> entry = iterator.next();
                SurfaceViewRenderer existingSurfaceViewRenderer = entry.getKey();
                if (!existingSurfaceViewRenderer.equals(surfaceViewRenderer)) {
                    VideoRenderer videoRenderer = entry.getValue();
                    videoTrack.removeRenderer(videoRenderer);
                    iterator.remove();
                }
            }
        }
        return this;
    }

    ManagedStream replaceAllSurfacesWith(SurfaceViewRenderer surfaceViewRenderer) {
        removeAllSurfacesExcept(surfaceViewRenderer);
        if (renderers.size() == 0) {
            addSurface(surfaceViewRenderer);
        }
        return this;
    }

    MediaStream getStream() {
        return stream;
    }

    Set<SurfaceViewRenderer> getSurfaces() {
        return renderers.keySet();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof ManagedStream)) return false;

        ManagedStream that = (ManagedStream) o;

        return stream.equals(that.stream);
    }

    @Override
    public int hashCode() {
        return stream.hashCode();
    }
}
