package live.attach.infrastructure.octopus.socket;

import android.os.Handler;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import live.attach.application.AttachEnvironment;
import live.attach.application.RoomLiveConnection;
import live.attach.domain.model.exception.AttachException;
import live.attach.domain.model.exception.error.AttachInternalException;
import live.attach.domain.model.session.RoomEvent;
import live.attach.domain.model.session.RoomPresence;
import live.attach.domain.model.session.RoomSession;
import live.attach.domain.model.video.Candidate;
import live.attach.domain.model.video.Sdp;
import live.attach.domain.model.video.VideoCall;
import live.attach.infrastructure.logger.AttachLogger;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.infrastructure.octopus.socket.dto.request.RequestFactory;
import live.attach.infrastructure.octopus.socket.dto.request.RequestMessagesPayload;
import live.attach.infrastructure.octopus.socket.dto.request.RequestParticipantsPayload;
import live.attach.infrastructure.octopus.socket.dto.request.SendMessageRequestPayload;
import live.attach.repackaged.com.google.gson.Gson;
import live.attach.repackaged.io.reactivex.Observable;
import live.attach.repackaged.io.reactivex.subjects.BehaviorSubject;
import live.attach.repackaged.io.reactivex.subjects.PublishSubject;
import live.attach.repackaged.io.socket.client.Ack;
import live.attach.repackaged.io.socket.client.Manager;
import live.attach.repackaged.io.socket.client.Socket;
import live.attach.repackaged.io.socket.client.SocketIOException;
import live.attach.repackaged.io.socket.engineio.client.Transport;

public class OctopusLiveConnection implements RoomLiveConnection {
    private static final AttachLogger log = AttachLoggerFactory.getLogger();
    private static final List<String> eventsToListen = Arrays.asList(
        "participants", "messages", "message", "participant-here", "participant-away", "participant-invite",
        "participant-offer", "participant-answer", "participant-candidate", "participant-bye",
        "error", "server-status"
    );
    private static final Gson gson = new Gson();

    private final AttachEnvironment environment;
    private final BehaviorSubject<RoomPresence> roomPresenceStream;
    private final PublishSubject<RoomEvent> roomEventsStream;
    private Socket socket;

    public OctopusLiveConnection(AttachEnvironment env) {
        environment = env;
        roomPresenceStream = BehaviorSubject.createDefault(RoomPresence.initial);
        roomEventsStream = PublishSubject.create();
    }

    @Override
    public void connect(RoomSession roomSession) {
        try {
            log.debug("Connecting to the octopus socket");
            String url = roomSession.getServer() + "?room-token=" + roomSession.getToken();
            socket = OctopusSocketIO.getSocket(url);

            for (final String event : eventsToListen) {
                socket.on(event, args -> {
                    if (args.length > 0 && args[0] instanceof String) {
                        String payload = (String) args[0];
                        log.info("Socket data received for '" + event + "': " + payload);
                        roomEventsStream.onNext(new RoomEvent(event, payload));
                    }
                });
            }

            socket.io().on(Manager.EVENT_TRANSPORT, args -> {
                log.debug("Transport event: " + Arrays.toString(args));
                Transport transport = (Transport) args[0];
                transport.on(Transport.EVENT_REQUEST_HEADERS, tArgs -> {
                    @SuppressWarnings("unchecked")
                    Map<String, List<String>> headers = (Map<String, List<String>>) tArgs[0];
                    headers.put("Authorization", Collections.singletonList("Bearer " + roomSession.getToken()));
                    headers.put("X-Attach-ApiKey", Collections.singletonList(environment.getApiKey()));
                    headers.put("X-Octopus-Version", Collections.singletonList("2"));
                    headers.put("X-Octopus-Scope", Collections.singletonList("room"));
                });
            });

            socket.on(Socket.EVENT_CONNECT, args -> {
                log.debug("Event Connect");
                roomPresenceStream.onNext(RoomPresence.online(roomSession));
            });

            socket.on(Socket.EVENT_DISCONNECT, args -> {
                log.debug("Event Disconnect");
                roomPresenceStream.onNext(RoomPresence.error(roomSession, new AttachInternalException()));
            });

            socket.on(Socket.EVENT_CONNECT_TIMEOUT, args -> {
                log.debug("Event Connect Timeout");
                roomPresenceStream.onNext(RoomPresence.error(roomSession, new AttachInternalException()));
            });

            socket.on(Socket.EVENT_ERROR, args -> {
                log.debug("Event Error");
                // socket.io should deliver SocketIOException as args[0] but sometimes delivers String: handle gracefully
                SocketIOException socketIOException;
                try {
                    if (args[0] != null && args[0] instanceof String) {
                        String message = (String) args[0];
                        socketIOException = new SocketIOException(message);
                    } else if (args[0] != null && args[0] instanceof SocketIOException) {
                        socketIOException = (SocketIOException) args[0];
                    } else {
                        String message = "unknown socket.io error";
                        socketIOException = new SocketIOException(message);
                    }
                } catch (Exception e) {
                    String message = "unknown socket.io error type";
                    socketIOException = new SocketIOException(message);
                }
                roomPresenceStream.onNext(
                    RoomPresence.error(roomSession, new AttachInternalException(socketIOException))
                );
            });

            socket.connect();
        } catch (Exception e) {
            log.error("Socket connection error", e);
            roomPresenceStream.onNext(RoomPresence.error(roomSession, new AttachInternalException(e)));
        }
    }

    @Override
    public void disconnect() {
        if (socket != null) {
            socket.off();
            socket.disconnect();
            socket = null;
            log.debug("Socket disconnected");
            roomPresenceStream.onNext(RoomPresence.initial);
        }
    }

    private void send(String event, Object object) {
        if (socket == null || !socket.connected()) {
            return;
        }
        String toSend = gson.toJson(object);
        log.debug("Sending '" + event + "': " + toSend);
        socket.emit(event, toSend, (Ack) args -> {
            log.debug("Acknowledgement. Event " + event + " Args: " + Arrays.toString(args));
            if (args == null || args.length == 0 || (args.length == 1 && args[0] == null)) {
                // Ignore SUCCESS
            } else {
                Object arg = args[0];
                if ("error".equals(arg)) {
                    String error = (String) args[1];
                    RoomSession currentSession = roomPresenceStream.getValue().getRoomSession();
                    roomPresenceStream.onNext(RoomPresence.error(currentSession, AttachException.exception(error)));
                }
            }
        });
    }

    @Override
    public Observable<RoomPresence> getRoomPresence() {
        return roomPresenceStream;
    }

    @Override
    public Observable<RoomEvent> getRoomEvents() {
        return roomEventsStream;
    }

    @Override
    public void requestParticipants() {
        send(
            "request-participants",
            new RequestParticipantsPayload(new RequestParticipantsPayload.Meta(50, null))
        );
    }

    @Override
    public void requestMessages(Long timestamp) {
        send(
            "request-messages",
            new RequestMessagesPayload(new RequestMessagesPayload.Meta(50, timestamp))
        );
    }

    @Override
    public void sendMessage(String message) {
        send(
            "message",
            new SendMessageRequestPayload(new SendMessageRequestPayload.Message(message))
        );
    }


    @Override
    public void sendIceCandidate(String socketId, Candidate candidate) {
        send("participant-candidate", RequestFactory.participantCandidate(socketId, candidate));
    }

    @Override
    public void inviteParticipant(String socketId) {
        send("participant-invite", RequestFactory.participantInvite(socketId));
    }

    @Override
    public void sendAnswerCreated(String socketId, String patchedSessionDescription) {
        send(
            "participant-answer",
            RequestFactory.participantAnswer(
                socketId,
                new Sdp(Sdp.Type.answer, patchedSessionDescription)
            )
        );
    }

    @Override
    public void sendOfferCreated(String socketId, String patchedSessionDescription) {
        send(
            "participant-offer",
            RequestFactory.participantOffer(
                socketId,
                new Sdp(Sdp.Type.offer, patchedSessionDescription)
            )
        );
    }

    @Override
    public void sendParticipantBye(String socketId, VideoCall.ByeReason reason) {
        send(
            "participant-bye",
            RequestFactory.participantBye(
                socketId,
                reason.toString()
            )
        );
    }


}
