package live.attach.infrastructure.octopus.rest;

import live.attach.application.AttachEnvironment;
import live.attach.application.SessionResolver;
import live.attach.domain.model.exception.error.AttachInternalException;
import live.attach.domain.model.session.Session;
import live.attach.domain.model.session.User;
import live.attach.infrastructure.android.cache.Cache;
import live.attach.infrastructure.logger.AndroidDebugLogger;
import live.attach.infrastructure.logger.AndroidErrorLogger;
import live.attach.infrastructure.logger.AndroidInfoLogger;
import live.attach.infrastructure.logger.AttachLogger;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.infrastructure.octopus.rest.dto.SessionResponse;
import live.attach.infrastructure.octopus.rest.dto.mapper.SessionMapper;
import live.attach.lib.TextUtils;
import live.attach.lib.com.auth0.android.jwt.JWT;
import live.attach.repackaged.com.google.gson.Gson;
import live.attach.repackaged.io.reactivex.Observable;

public class OctopusSessionResolver implements SessionResolver {
    private static final AttachLogger log = AttachLoggerFactory.getLogger();

    private final OctopusHttpApi octopusApi;
    private final AttachEnvironment environment;
    private final Cache cache;
    private final Gson gson;

    public OctopusSessionResolver(
        OctopusHttpApi octopusApi,
        AttachEnvironment environment,
        Cache cache
    ) {
        this.octopusApi = octopusApi;
        this.environment = environment;
        this.cache = cache;
        this.gson = new Gson();
    }

    @Override
    public Observable<Session> resolveSession(User user) {
        return createSession(environment.getBaseUrl() + "/session", user).onErrorReturn(throwable -> Session.error(new AttachInternalException(throwable)));
    }

    private Observable<Session> createSession(String url, User user) {
        String userString = user.isGuest() ? null : gson.toJson(user);
        String userCookie = environment.createSessionCookie(getCookie(
            user.isJwtBased() ? getUserIdFromToken(user.getToken()) : user.getId()
        ));


        return octopusApi.getSession(url, userString, userCookie)
            .switchMap(sessionResponse -> {
                if (sessionResponse.meta.warnings != null) {
                    for (SessionResponse.Meta.Warning w : sessionResponse.meta.warnings) {
                        log.warning(w.message + ": " + w.code);
                    }
                }
                if (sessionResponse.meta.sessionExist()) {
                    String sessionCookie = environment.extractSessionCookie(sessionResponse.cookie);
                    String providedUserId = "__anonymous__";
                    if (TextUtils.isEmpty(sessionResponse.meta.session.options.providedUserId)) {
                        providedUserId = sessionResponse.meta.session.options.providedUserId;
                    }
                    storeCookie(providedUserId, sessionCookie);

                    Session session = SessionMapper.mapSessionResponse(sessionResponse);
                    if (environment.isAttachDevelopment()) {
                        AttachLoggerFactory.registerLogger(new AndroidDebugLogger());
                    } else {
                        if (session.isDevelopment()) {
                            AttachLoggerFactory.registerLogger(new AndroidInfoLogger());
                        } else {
                            AttachLoggerFactory.registerLogger(new AndroidErrorLogger());
                        }
                    }
                    return Observable.just(session);
                } else {
                    return createSession(
                        "https://" + sessionResponse.meta.server.hostname + "/session",
                        user
                    );
                }
            });
    }

    private String getUserIdFromToken(String token) {
        try {
            JWT jwt = new JWT(token);
            return jwt.getClaim("user").getSubClaim("id").asString();
        } catch (Exception e) {
            throw new RuntimeException("Invalid JWT token");
        }
    }

    private void storeCookie(String userId, String cookie) {
        if (TextUtils.isEmpty(userId) || TextUtils.isEmpty(cookie)) return;
        cache.putString("cookie_" + userId, cookie);
    }

    private String getCookie(String userId) {
        if (TextUtils.isEmpty(userId)) return null;
        return cache.getString("cookie_" + userId);
    }
}
