package live.attach.infrastructure.octopus.rest;

import live.attach.lib.TextUtils;
import live.attach.repackaged.com.google.gson.Gson;

import live.attach.repackaged.io.reactivex.Observable;
import live.attach.application.RoomSessionResolver;
import live.attach.domain.model.exception.AttachException;
import live.attach.domain.model.exception.error.AttachInternalException;
import live.attach.domain.model.exception.error.HttpSessionExpiredException;
import live.attach.domain.model.item.ItemId;
import live.attach.domain.model.session.RoomSession;
import live.attach.domain.model.session.Session;
import live.attach.infrastructure.logger.AttachLogger;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.infrastructure.octopus.rest.dto.ErrorBody;
import live.attach.infrastructure.octopus.rest.dto.ResponseError;
import live.attach.infrastructure.octopus.rest.dto.RoomSessionResponse;
import live.attach.infrastructure.octopus.rest.dto.mapper.RoomSessionMapper;
import okhttp3.ResponseBody;

public class OctopusRoomSessionResolver implements RoomSessionResolver {
    private static final AttachLogger log = AttachLoggerFactory.getLogger();

    private final OctopusHttpApi octopusApi;
    private final Gson gson;
    private String lastServer;

    public OctopusRoomSessionResolver(OctopusHttpApi octopusApi) {
        this.octopusApi = octopusApi;
        this.gson = new Gson();
    }

    @Override
    public Observable<RoomSession> resolveRoomSession(Session session, ItemId itemId) {
        if (session.hasError()) return Observable.just(RoomSession.error(session, null));

        String url = session.getServer();
        if (!TextUtils.isEmpty(lastServer)) {
            url = lastServer;
        }
        url += "/room";

        String itemIdString = itemId.getUrl();
        if (!itemId.isWebItem()) {
            itemIdString = String.format("%s/%s/%s", itemId.getProvider(), itemId.getType(), itemId.getIdentifier());
        }

        return createRoomSession(url, itemIdString, session)
            .onErrorReturn(throwable -> {
                AttachException error = new AttachInternalException(throwable);
                if (throwable instanceof HttpSessionExpiredException) {
                    error = (HttpSessionExpiredException) throwable;
                }
                return RoomSession.error(session, error);
            });
    }

    private Observable<RoomSession> createRoomSession(String url, String itemIdString, Session session) {
        String token = session.getToken();
        return octopusApi.getRoom(url, itemIdString, token)
            .switchMap(roomSessionResponse -> {
                if (roomSessionResponse.meta.warnings != null) {
                    for (RoomSessionResponse.Meta.Warning w : roomSessionResponse.meta.warnings) {
                        log.warning(w.message + ": " + w.code);
                    }
                }
                if (roomSessionResponse.meta.roomExist()) {
                    lastServer = "https://" + roomSessionResponse.meta.server.hostname;
                    return Observable.just(RoomSessionMapper.mapRoomSessionResponse(session, roomSessionResponse));
                } else {
                    return createRoomSession(
                        "https://" + roomSessionResponse.meta.server.hostname + "/room",
                        itemIdString,
                        session
                    );
                }
            });
    }
}
