package live.attach.infrastructure.http;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import live.attach.infrastructure.logger.AttachLogger;
import live.attach.infrastructure.logger.AttachLoggerFactory;

public class JavaHttpService {
    private static AttachLogger log = AttachLoggerFactory.getLogger();

    public HttpResponse get(String endpoint, Map<String, String> params, Map<String, String> headers) throws IOException {
        log.debug("Making get request: " + endpoint);
        if (params == null) params = new HashMap<>();
        if (headers == null) headers = new HashMap<>();

        String paramsString = Query.urlEncodeUTF8(params);
        if (paramsString.length() > 0) endpoint += "?" + paramsString;

        URL url = new URL(endpoint);
        HttpURLConnection httpURLConnection = null;
        try {
            httpURLConnection = (HttpURLConnection) url.openConnection();
            for (Map.Entry<String, String> header : headers.entrySet()) {
                httpURLConnection.setRequestProperty(header.getKey(), header.getValue());
            }
            httpURLConnection.setRequestMethod("GET");
            httpURLConnection.connect();

            int responseCode = httpURLConnection.getResponseCode();
            log.debug("Response status: " + responseCode);
            if (responseCode != 200) {
                log.debug("Response message: " + httpURLConnection.getResponseMessage());
                return new HttpResponse(responseCode, httpURLConnection.getResponseMessage());
            }

            StringBuilder content;
            try (BufferedReader in = new BufferedReader(new InputStreamReader(httpURLConnection.getInputStream()))) {
                String line;
                content = new StringBuilder();

                while ((line = in.readLine()) != null) {
                    content.append(line);
                }
            }

            Map<String, String> responseHeaders = new HashMap<>();

            for (Map.Entry<String, List<String>> entry : httpURLConnection.getHeaderFields().entrySet()) {
                if (entry.getValue().size() > 0) {
                    if (entry.getKey() != null) {
                        responseHeaders.put(entry.getKey(), entry.getValue().get(0));
                    }
                }
            }

            log.debug("Response message: " + content.toString());

            return new HttpResponse(
                responseCode,
                content.toString(),
                responseHeaders
            );
        } finally {
            if (httpURLConnection != null) {
                httpURLConnection.disconnect();
            }
        }
    }

    public HttpResponse post(String endpoint, String body, Map<String, String> headers) throws IOException {
        log.debug("Making post request: " + endpoint);

        if (headers == null) headers = new HashMap<>();
        if (body == null) body = "";

        URL url = new URL(endpoint);
        HttpURLConnection httpURLConnection = null;
        try {
            httpURLConnection = (HttpURLConnection) url.openConnection();
            for (Map.Entry<String, String> header : headers.entrySet()) {
                httpURLConnection.setRequestProperty(header.getKey(), header.getValue());
            }
            httpURLConnection.setRequestMethod("POST");
            httpURLConnection.setDoOutput(true);
            byte[] out = body.getBytes(Charset.forName("UTF-8"));
            httpURLConnection.setFixedLengthStreamingMode(out.length);
            httpURLConnection.setRequestProperty("Content-Type", "application/json; charset=UTF-8");
            httpURLConnection.connect();

            try (OutputStream os = httpURLConnection.getOutputStream()) {
                os.write(out);
            }

            int responseCode = httpURLConnection.getResponseCode();
            log.debug("Response status: " + responseCode);
            if (responseCode != 200) {
                log.debug("Response message: " + httpURLConnection.getResponseMessage());
                return new HttpResponse(responseCode, httpURLConnection.getResponseMessage());
            }

            StringBuilder content;
            try (BufferedReader in = new BufferedReader(new InputStreamReader(httpURLConnection.getInputStream()))) {
                String line;
                content = new StringBuilder();

                while ((line = in.readLine()) != null) {
                    content.append(line);
                }
            }

            log.debug("Response message: " + content.toString());

            return new HttpResponse(
                responseCode,
                content.toString(),
                new HashMap<>()
            );
        } finally {
            if (httpURLConnection != null) {
                httpURLConnection.disconnect();
            }
        }
    }
}
