package live.attach.infrastructure.env;

import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.util.Base64;

import java.math.BigInteger;
import java.net.HttpCookie;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.UUID;

import live.attach.application.AttachEnvironment;
import live.attach.infrastructure.android.cache.Cache;
import live.attach.infrastructure.android.utils.AndroidUtils;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.lib.TextUtils;
import live.attach.sdk.BuildConfig;

public abstract class AttachAndroidEnvironment implements AttachEnvironment {
    private Cache cache;

    private final String packageName;
    private final String apiKey;
    private final String clientApiKey;
    private final String deviceId;
    private final String acceptLanguage;
    private final String baseUrl;
    private final String platformUrl;
    private final String loggingUrl;

    AttachAndroidEnvironment(Context context, String providedClientApiKey) throws Exception {
        cache = new Cache(context);

        packageName = context.getPackageName();
        Bundle bundle = context.getPackageManager().getApplicationInfo(packageName, PackageManager.GET_META_DATA).metaData;

        if (TextUtils.isEmpty(providedClientApiKey) && bundle == null)
            throw new RuntimeException("Can not read API key");

        clientApiKey = TextUtils.isEmpty(providedClientApiKey) ? bundle.getString("attachApiKey") : providedClientApiKey;
        baseUrl = BuildConfig.BASE_URL;
        apiKey = BuildConfig.API_KEY;
        platformUrl = BuildConfig.PLATFORM_URL;
        loggingUrl = BuildConfig.LOGGING_URL;
        deviceId = extractDeviceId();
        acceptLanguage = Locale.getDefault().getLanguage();

        if (bundle != null) {
            boolean attachErrorReporting = bundle.getBoolean("attachErrorReporting", false);
            AttachLoggerFactory.setErrorReporting(attachErrorReporting);
        }
    }

    @Override
    public String getBaseUrl() {
        return baseUrl;
    }

    @Override
    public String getApiKey() {
        return apiKey;
    }

    @Override
    public String getPlatformUrl() {
        return platformUrl;
    }

    @Override
    public String getLoggingUrl() {
        return loggingUrl;
    }

    @Override
    public Map<String, String> getLoggingHeaders() {
        Map<String, String> headers = new HashMap<>();
        headers.put("X-Attach-ApiKey", apiKey);
        headers.put("X-Attach-Client-ApiKey", clientApiKey);
        return headers;
    }

    @Override
    public Map<String, String> getPlatformHeaders() {
        return new HashMap<>();
    }

    @Override
    public String extractSessionCookie(String setCookie) {
        List<HttpCookie> cookies = HttpCookie.parse(setCookie);
        for (HttpCookie cookie : cookies) {
            if (cookie.getName().equals("session")) return cookie.getValue();
        }
        return null;
    }

    @Override
    public String getPackageName() {
        return packageName;
    }

    @Override
    public String createSessionCookie(String cookie) {
        if (TextUtils.isEmpty(cookie)) return null;
        return "session=" + cookie;
    }

    @Override
    public Map<String, String> getOctopusHeaders() {
        Map<String, String> headers = new HashMap<>();
        headers.put("X-Attach-PackageName", packageName);
        headers.put("X-Attach-ApiKey", apiKey);
        headers.put("X-Attach-Client-ApiKey", clientApiKey);
        headers.put("X-Attach-DeviceId", deviceId);
        headers.put("Accept-Language", acceptLanguage);
        return headers;
    }

    private String extractDeviceId() {
        String deviceId = cache.getString("deviceId");
        if (!TextUtils.isEmpty(deviceId)) return deviceId;

        BigInteger bigInteger = new BigInteger(
            UUID.randomUUID().toString().replace("-", "").toLowerCase(),
            16
        );
        deviceId = Base64.encodeToString(bigInteger.toByteArray(), Base64.NO_PADDING).trim();
        cache.putString("deviceId", deviceId);
        return deviceId;
    }


    public static AttachEnvironment createEnvironment(Context context, String apiKey) {
        try {
            String env = BuildConfig.ATTACH_ENVIRONMENT;
            if ("production".equals(env)) {
                return new ProductionEnvironment(context, apiKey);
            } else {
                return new DevelopmentEnvironment(context, apiKey);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
