package live.attach.application;

import java.util.concurrent.TimeUnit;

import live.attach.repackaged.io.reactivex.Observable;
import live.attach.repackaged.io.reactivex.Scheduler;
import live.attach.repackaged.io.reactivex.subjects.BehaviorSubject;
import live.attach.domain.model.session.Session;
import live.attach.domain.model.session.User;

public class AttachSessionInteractor {
    private final BehaviorSubject<Session> sessionStream;
    private final BehaviorSubject<User> userStream;

    public AttachSessionInteractor(
        SessionResolver sessionResolver,
        AttachEnvironment environment,
        Scheduler scheduler
    ) {
        userStream = BehaviorSubject.createDefault(User.initial);
        sessionStream = BehaviorSubject.createDefault(Session.initial);

        getUserStream().debounce(500, TimeUnit.MILLISECONDS, scheduler)
            .observeOn(scheduler)
            .switchMap(sessionResolver::resolveSession)
            .subscribe(sessionStream);
    }

    private Observable<User> getUserStream() {
        return userStream.filter(user -> user != User.initial);
    }

    public Observable<Session> getSessionStream() {
        return sessionStream
            .filter(session -> session != Session.initial)
            .distinctUntilChanged();
    }

    public void signIn(User user) {
        if (user == User.initial) return;
        userStream.onNext(user == null ? User.guest : user);
    }

    public void signOut() {
        userStream.onNext(User.guest);
    }

    public boolean validSessionExist() {
        return sessionStream.getValue().isValid();
    }


    public void refreshSession() {
        User user = userStream.getValue();
        userStream.onNext(user);
    }

    public void refreshOrUseGuest() {
        User user = userStream.getValue();
        if (user == User.initial) {
            userStream.onNext(User.guest);
        } else {
            userStream.onNext(user);
        }
    }

    public void expireSession() {
        sessionStream.onNext(Session.expired());
    }
}
