package live.attach.application;

import live.attach.repackaged.com.google.gson.Gson;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import live.attach.repackaged.io.reactivex.Observable;
import live.attach.repackaged.io.reactivex.schedulers.Schedulers;
import live.attach.repackaged.io.reactivex.subjects.BehaviorSubject;
import live.attach.domain.model.session.RoomEvent;
import live.attach.domain.model.application.AttachProperties;
import live.attach.domain.model.chat.ChatHistory;
import live.attach.domain.model.chat.Message;
import live.attach.domain.model.exception.AttachException;
import live.attach.domain.model.exception.error.AttachError;
import live.attach.domain.model.exception.error.AttachInternalException;
import live.attach.domain.model.exception.warning.AttachDetailedWarning;
import live.attach.domain.model.exception.warning.AttachWarning;
import live.attach.domain.model.exception.warning.DebugVersionException;
import live.attach.domain.model.exception.warning.NonGenuineVersionException;
import live.attach.domain.model.item.ItemId;
import live.attach.domain.model.participant.Participant;
import live.attach.domain.model.session.RoomPresence;
import live.attach.domain.model.session.RoomSession;
import live.attach.domain.model.session.Session;
import live.attach.domain.model.session.SessionExceptionInfo;
import live.attach.domain.model.video.Candidate;
import live.attach.domain.model.video.IceServer;
import live.attach.domain.model.video.VideoCall;
import live.attach.infrastructure.logger.AttachLogger;
import live.attach.infrastructure.logger.AttachLoggerFactory;
import live.attach.infrastructure.octopus.socket.dto.response.Mapper;
import live.attach.infrastructure.octopus.socket.dto.response.MessageResponsePayload;
import live.attach.infrastructure.octopus.socket.dto.response.MessagesResponsePayload;
import live.attach.infrastructure.octopus.socket.dto.response.ParticipantAnswerResponsePayload;
import live.attach.infrastructure.octopus.socket.dto.response.ParticipantAwayResponsePayload;
import live.attach.infrastructure.octopus.socket.dto.response.ParticipantByeResponsePayload;
import live.attach.infrastructure.octopus.socket.dto.response.ParticipantCandidateResponsePayload;
import live.attach.infrastructure.octopus.socket.dto.response.ParticipantHereResponsePayload;
import live.attach.infrastructure.octopus.socket.dto.response.ParticipantInviteResponsePayload;
import live.attach.infrastructure.octopus.socket.dto.response.ParticipantOfferResponsePayload;
import live.attach.infrastructure.octopus.socket.dto.response.ParticipantsResponsePayload;
import live.attach.infrastructure.octopus.socket.dto.response.ServerStatusResponsePayload;
import live.attach.infrastructure.octopus.socket.video.VideoConnectionClient;
import live.attach.infrastructure.octopus.socket.video.WebRtcVideoController;
import live.attach.lib.EmptySlotsDistinctList;
import live.attach.lib.Pair;
import live.attach.lib.SimpleObserver;

public class AttachRoomInteractor implements WebRtcVideoController {
    private static final AttachLogger log = AttachLoggerFactory.getLogger();

    private final BehaviorSubject<ItemId> itemStream;
    private final BehaviorSubject<EmptySlotsDistinctList<Participant>> participantsStream;
    private final BehaviorSubject<ChatHistory> messagesStream;
    private final BehaviorSubject<VideoCall> videoCallStream;
    private final BehaviorSubject<SessionExceptionInfo> exceptionsStream;
    private final BehaviorSubject<RoomPresence> roomPresenceStream;

    private final RoomLiveConnection roomLiveConnection;
    private AttachProperties properties = new AttachProperties();
    private Gson gson = new Gson();
    private VideoConnectionClient videoConnectionClient;

    public AttachRoomInteractor(
        RoomSessionResolver roomSessionResolver,
        RoomLiveConnection roomLiveConnection,
        Observable<Session> sessionStream
    ) {
        this.roomLiveConnection = roomLiveConnection;

        this.exceptionsStream = BehaviorSubject.createDefault(SessionExceptionInfo.empty);
        this.itemStream = BehaviorSubject.createDefault(ItemId.empty);
        this.messagesStream = BehaviorSubject.createDefault(ChatHistory.empty());
        this.participantsStream = BehaviorSubject.createDefault(new EmptySlotsDistinctList<>());
        this.videoCallStream = BehaviorSubject.createDefault(VideoCall.DEFAULT);
        this.roomPresenceStream = BehaviorSubject.createDefault(RoomPresence.initial);

        roomLiveConnection.getRoomPresence().subscribe(roomPresenceStream);

        Observable
            .combineLatest(
                getItemStream().debounce(750, TimeUnit.MILLISECONDS),
                sessionStream,
                Pair::new
            )
            .observeOn(Schedulers.io())
            .switchMap(itemIdSessionPair -> {
                ItemId itemId = itemIdSessionPair.one;
                Session session = itemIdSessionPair.two;
                if (session.hasApplication() && session.isDevelopment()) {
                    publishException(new DebugVersionException());
                }
                if (session.hasApplication() && !session.isGenuine()) {
                    publishException(new NonGenuineVersionException());
                }
                if (!session.isValid())
                    return Observable.just(RoomSession.error(session, null));
                return roomSessionResolver.resolveRoomSession(session, itemId);
            })
            .subscribe(new SimpleObserver<RoomSession>() {
                @Override
                public void onNext(RoomSession roomSession) {
                    if (roomSession.hasError()) {
                        roomPresenceStream.onNext(RoomPresence.error(roomSession, null));
                    } else {
                        roomLiveConnection.connect(roomSession);
                    }
                }
            });

        roomLiveConnection.getRoomEvents().subscribe(new SimpleObserver<RoomEvent>() {
            @Override
            public void onNext(RoomEvent roomEvent) {
                String data = roomEvent.payload;
                switch (roomEvent.event) {
                    case "participants":
                        ParticipantsResponsePayload participants = gson.fromJson(data, ParticipantsResponsePayload.class);
                        EmptySlotsDistinctList<Participant> participantsList = new EmptySlotsDistinctList<>();
                        for (ParticipantsResponsePayload.Participant p : participants.data) {
                            // ignore current user
                            if (p.socketId.equals(getCurrentParticipant().getSocketId()))
                                continue;
                            participantsList.add(new Participant(
                                p.socketId,
                                p.user.id,
                                p.user.avatar,
                                p.user.username
                            ));
                        }
                        participantsStream.onNext(participantsList);
                        break;
                    case "participant-away":
                        EmptySlotsDistinctList<Participant> currentParticipants = participantsStream.getValue();
                        ParticipantAwayResponsePayload participantAway = gson.fromJson(data, ParticipantAwayResponsePayload.class);
                        String socketId = participantAway.data.socketId;
                        currentParticipants.removeFirst(participant -> participant.getSocketId().equals(socketId));
                        participantsStream.onNext(currentParticipants);

                        VideoCall videoCall = videoCallStream.getValue();
                        if (videoCall.isActive() && getVideoParticipant().getSocketId().equals(socketId)) {
                            onParticipantBye(VideoCall.ByeReason.CALL_FINISH);
                        }
                        break;
                    case "participant-here":
                        currentParticipants = participantsStream.getValue();
                        ParticipantHereResponsePayload participantHere = gson.fromJson(data, ParticipantHereResponsePayload.class);
                        currentParticipants.add(new Participant(
                            participantHere.data.socketId,
                            participantHere.data.user.id,
                            participantHere.data.user.avatar,
                            participantHere.data.user.username
                        ));
                        participantsStream.onNext(currentParticipants);
                        break;
                    case "participant-bye":
                        if (!videoCallStream.getValue().isActive()) return;

                        ParticipantByeResponsePayload participantByeResponsePayload = gson.fromJson(data, ParticipantByeResponsePayload.class);
                        VideoCall.ByeReason reason = VideoCall.ByeReason.CALL_FINISH;
                        try {
                            reason = VideoCall.ByeReason.valueOf(participantByeResponsePayload.meta.reason);
                        } catch (Exception ignored) {
                        }
                        onParticipantBye(reason);
                        break;
                    case "messages":
                        MessagesResponsePayload messagesResponsePayload = gson.fromJson(data, MessagesResponsePayload.class);
                        ChatHistory existingMessages = messagesStream.getValue();
                        existingMessages.addMessages(Mapper.getMessages(messagesResponsePayload));
                        messagesStream.onNext(existingMessages);
                        break;
                    case "message":
                        MessageResponsePayload messageResponsePayload = gson.fromJson(data, MessageResponsePayload.class);
                        existingMessages = messagesStream.getValue();
                        existingMessages.addMessage(Mapper.getMessage(messageResponsePayload));
                        messagesStream.onNext(existingMessages);
                        break;
                    case "participant-invite":
                        ParticipantInviteResponsePayload participantInviteResponsePayload = gson.fromJson(data, ParticipantInviteResponsePayload.class);
                        List<IceServer> iceServers = Mapper.getIceServers(participantInviteResponsePayload);
                        Participant participant = Mapper.getParticipant(participantInviteResponsePayload);

                        if (properties.isVideocallEnabled()) {
                            if (videoCallStream.getValue().isActive()) {
                                roomLiveConnection.sendParticipantBye(
                                    getVideoParticipant().getSocketId(),
                                    VideoCall.ByeReason.INVITE_BUSY
                                );
                            } else {
                                videoCallStream.onNext(new VideoCall(
                                    VideoCall.State.INCOMING,
                                    participant,
                                    null,
                                    iceServers
                                ));
                            }
                        } else {
                            roomLiveConnection.sendParticipantBye(
                                getVideoParticipant().getSocketId(),
                                VideoCall.ByeReason.INVITE_REJECT
                            );
                        }
                        break;
                    case "participant-offer":
                        ParticipantOfferResponsePayload participantOfferResponsePayload = gson.fromJson(data, ParticipantOfferResponsePayload.class);
                        if (videoConnectionClient != null) {
                            videoConnectionClient.onParticipantOffer(
                                participantOfferResponsePayload.data.socketId,
                                participantOfferResponsePayload.data.sdp,
                                participantOfferResponsePayload.meta.options.iceServers
                            );
                        }
                        break;
                    case "participant-answer":
                        ParticipantAnswerResponsePayload participantAnswerResponsePayload = gson.fromJson(data, ParticipantAnswerResponsePayload.class);
                        if (videoConnectionClient != null) {
                            videoConnectionClient.onParticipantAnswer(
                                participantAnswerResponsePayload.data.socketId,
                                participantAnswerResponsePayload.data.sdp
                            );
                            videoCallStream.onNext(videoCallStream.getValue().setState(VideoCall.State.IN_PROGRESS));
                        }
                        break;
                    case "participant-candidate":
                        ParticipantCandidateResponsePayload participantCandidateResponsePayload = gson.fromJson(data, ParticipantCandidateResponsePayload.class);
                        if (videoConnectionClient != null) {
                            videoConnectionClient.onParticipantCandidate(
                                participantCandidateResponsePayload.data.socketId,
                                participantCandidateResponsePayload.data.candidate
                            );
                        }
                        break;
                    case "error":
                        AttachException error = AttachException.exception(data);
                        onSocketException(error);
                        break;
                    case "server-status":
                        ServerStatusResponsePayload serverStatusResponsePayload = gson.fromJson(data, ServerStatusResponsePayload.class);
                        if (serverStatusResponsePayload.data.state.equals("DOWN")) {
                            onSocketException(new AttachInternalException());
                        }
                        break;
                }
            }
        });
    }

    private void publishException(AttachException exception) {
        if (exception == null) return;

        if (exception instanceof AttachError) {
            AttachError attachError = (AttachError) exception;
            log.error(attachError.getAttachMessage(), attachError);
            log.remote(attachError.getAttachMessage());
        }
        if (exception instanceof AttachWarning) {
            if (exception instanceof AttachDetailedWarning) {
                log.warning("Warning: " + ((AttachDetailedWarning) exception).getAttachMessageRes());
            } else {
                log.warning("Warning: " + exception.getClass().getSimpleName());
            }
        }
        exceptionsStream.onNext(SessionExceptionInfo.of(exception));
    }

    private Participant getCurrentParticipant() {
        RoomPresence roomPresence = roomPresenceStream.getValue();
        return roomPresence.getCurrentParticipant();
    }

    private Participant getVideoParticipant() {
        VideoCall videoCall = videoCallStream.getValue();
        return videoCall.participant;
    }

    private Observable<ItemId> getItemStream() {
        return itemStream.filter(itemId -> itemId != ItemId.empty);
    }

    public Observable<SessionExceptionInfo> getExceptionsStream() {
        return exceptionsStream;
    }

    public void enterRoom(ItemId itemId, AttachProperties properties) {
        pauseRoom();
        leaveRoom();
        log.debug("EnteringRoom: " + itemId.toString());
        if (properties != null) this.properties = properties;
        itemStream.onNext(itemId);
    }

    public boolean restartRoom() {
        if (itemStream.getValue() != ItemId.empty) {
            log.debug("Restarting room " + itemStream.getValue());
            exceptionsStream.onNext(SessionExceptionInfo.empty);
            itemStream.onNext(itemStream.getValue());
            return true;
        }
        return false;
    }

    public void pauseRoom() {
        log.debug("Pausing room...");
        stopCall(VideoCall.ByeReason.CALL_FINISH);
        messagesStream.onNext(ChatHistory.empty());
        participantsStream.onNext(new EmptySlotsDistinctList<>());
        roomPresenceStream.onNext(RoomPresence.initial);
        exceptionsStream.onNext(SessionExceptionInfo.empty);
        roomLiveConnection.disconnect();
    }

    public void leaveRoom() {
        log.debug("Leaving room...");
        itemStream.onNext(ItemId.empty);
    }

    private void onSocketException(AttachException exception) {
        if (exception instanceof AttachError) {
            stopCall(VideoCall.ByeReason.CALL_FINISH);
            messagesStream.onNext(ChatHistory.empty());
            participantsStream.onNext(new EmptySlotsDistinctList<>());
            roomLiveConnection.disconnect();
            RoomSession roomSession = roomPresenceStream.getValue().getRoomSession();
            roomPresenceStream.onNext(RoomPresence.error(roomSession, exception));
        }
        if (exception instanceof AttachWarning) {
            publishException(exception);
        }
    }

    public void requestMessages() {
        ChatHistory currentHistory = messagesStream.getValue();
        roomLiveConnection.requestMessages(currentHistory.getOldestMessageDate());
    }

    public void stopCall(VideoCall.ByeReason reason) {
        if (videoCallStream.getValue().state != VideoCall.State.NONE) {
            roomLiveConnection.sendParticipantBye(getVideoParticipant().getSocketId(), reason);
            videoCallStream.onNext(VideoCall.DEFAULT);
        }
    }

    public void callParticipant(Participant participant) {
        videoCallStream.onNext(new VideoCall(
            VideoCall.State.OUTGOING,
            participant,
            null,
            new ArrayList<>()
        ));
        roomLiveConnection.inviteParticipant(participant.getSocketId());
    }

    public void acceptCall() {
        VideoCall currentCall = videoCallStream.getValue();
        videoCallStream.onNext(currentCall.setState(VideoCall.State.IN_PROGRESS));
    }

    public void setVideoConnectionClient(VideoConnectionClient videoConnectionClient) {
        this.videoConnectionClient = videoConnectionClient;
    }

    public Observable<List<Participant>> getParticipantsStream() {
        return participantsStream.map(EmptySlotsDistinctList::asList);
    }

    public Observable<List<Message>> getMessagesStream() {
        return messagesStream.map(chatHistory -> new ArrayList<>(chatHistory.getMessages()));
    }

    public Observable<VideoCall> getVideoCallStream() {
        return videoCallStream;
    }

    public Observable<RoomPresence> getRoomPresenceStream() {
        return roomPresenceStream
            .distinctUntilChanged()
            .doOnNext(roomPresence -> {
                if (roomPresence.hasError()) {
                    publishException(roomPresence.getError());
                }
                if (roomPresence.isActive()) {
                    requestMessages();
                    roomLiveConnection.requestParticipants();
                }
            });
    }

    private void onParticipantBye(VideoCall.ByeReason reason) {
        Observable.concat(
            Observable.just(
                videoCallStream.getValue().setState(VideoCall.State.STOPPED).setByeReason(reason)
            ),
            Observable.just(VideoCall.DEFAULT).delay(1500, TimeUnit.MILLISECONDS)
        ).subscribe(new SimpleObserver<VideoCall>() {
            @Override
            public void onNext(VideoCall videoCall) {
                videoCallStream.onNext(videoCall);
            }
        });
    }

    @Override
    public void sendIceCandidate(Candidate candidate) {
        roomLiveConnection.sendIceCandidate(getVideoParticipant().getSocketId(), candidate);
    }

    @Override
    public void sendAnswerCreated(String patchedSessionDescription) {
        roomLiveConnection.sendAnswerCreated(
            getVideoParticipant().getSocketId(),
            patchedSessionDescription
        );
        VideoCall currentCall = videoCallStream.getValue();
        videoCallStream.onNext(currentCall.setState(VideoCall.State.IN_PROGRESS));
    }

    @Override
    public void sendOfferCreated(String patchedSessionDescription) {
        roomLiveConnection.sendOfferCreated(
            getVideoParticipant().getSocketId(),
            patchedSessionDescription
        );
    }

    public void sendMessage(String message) {
        RoomPresence currentPresence = roomPresenceStream.getValue();
        if (currentPresence.isActive()) {
            roomLiveConnection.sendMessage(message);

            ChatHistory chatHistory = messagesStream.getValue();
            chatHistory.addMessage(Message.localMessage(message, getCurrentParticipant()));
            messagesStream.onNext(chatHistory);
        }
    }
}
