package com.suncode.upgrader.change.task;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.ImmutableList;
import com.suncode.upgrader.change.Change;
import com.suncode.upgrader.change.ChangeResult;
import com.suncode.upgrader.change.ExecutionStatus;
import com.suncode.upgrader.database.SupportedDatabase;

/**
 * Zmiana, której akcje definiowane są poprzez zadania {@link Task}. Zadania te wykonywane są w kolejności zdefiniowaj w
 * pliku XML. Niepowodzenie wykonywania jednego z zadań powoduje przerwanie wykonywania zmiany z wynikiem
 * {@link ExecutionStatus#FAILED}.
 * <p>
 * Wszystkie zadania zostaną wykonone w jednej transakcji. W przypadku wycofania zmiany zadania wycofywane są w
 * odwrotnej kolejności.
 * 
 * @author Łukasz Mocek
 */
public class TaskChange
    extends Change
{
    private static final Logger log = LoggerFactory.getLogger( TaskChange.class );

    private final List<Task> tasks;

    public TaskChange( String id, String version, String project, boolean mandatory, Set<SupportedDatabase> target, String comment, List<Task> tasks )
    {
        super( id, version, project, mandatory, target, comment, true );
        this.tasks = ImmutableList.copyOf( tasks );
    }

    @Override
    public ChangeResult run()
    {
        for ( Task task : tasks )
        {
            try
            {
                log.debug( "Executing task {}", task );
                task.run();
            }
            catch ( Exception e )
            {
                return resultFailed( new TaskExecutionException( "Error during task execution: " + task, e ) );
            }
        }
        return result( ExecutionStatus.EXECUTED );
    }

    @Override
    public void rollback()
    {
        Collections.reverse( tasks );
        for ( Task task : tasks )
        {
            try
            {
                log.debug( "Rolling back task {}", task );
                task.rollback();
            }
            catch ( Exception e )
            {
                throw new TaskExecutionException( "Error during task rollback: " + task, e );
            }
        }
    }

    public List<Task> getTasks()
    {
        return tasks;
    }

    @Override
    public String toString()
    {
        return "TaskChange[" + getId() + "]";
    }
}
