package com.suncode.upgrader.change.liquibase;

import java.util.Set;

import liquibase.changelog.ChangeSet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.suncode.upgrader.change.Change;
import com.suncode.upgrader.change.ChangeResult;
import com.suncode.upgrader.change.ExecutionStatus;
import com.suncode.upgrader.database.SupportedDatabase;

/**
 * Zmiana wykonywany przez narzędzie Liquibase. Bazuje na pojedynczym {@link ChangeSet} zdefiniowanym w pliku XML.
 * Właściwości tj. id, są pobierane z właściwości {@link ChangeSet}
 * <p>
 * Pojedyncza zmiana {@link ChangeSet} jest transakcyjna, ponieważ Liquibase wykonuje każdego {@link ChangeSet} w
 * osobnej transakcji.
 * 
 * @author Łukasz Mocek
 */
public class LiquibaseChange
    extends Change
{
    private static final Logger log = LoggerFactory.getLogger( LiquibaseChange.class );

    private String changeLogPath;

    private ChangeSet changeSet;

    public LiquibaseChange( String changeLogPath, ChangeSet changeSet, String version, String project, boolean mandatory,
                            Set<SupportedDatabase> target, String comment, boolean failOnError )
    {
        super( changeSet.getId(), version, project, mandatory, target, comment, failOnError);

        this.changeLogPath = changeLogPath;
        this.changeSet = changeSet;
    }


    @Override
    public ChangeResult run()
    {
        try
        {
            log.debug( "Executing changeSet [{}] from file: {}", changeSet.getId(), changeLogPath );
            ExecutionStatus executionStatus = LiquibaseHelper.executeChangeSet( changeSet );
            return result( executionStatus );
        }
        catch ( Exception e )
        {
            return resultFailed( new LiquibaseExecutionException( "Error during changeSet execution: " + changeSet.getId(), e ) );
        }

    }

    @Override
    public void rollback()
    {
        try
        {
            log.debug( "Rolling back changeSet [{}] from file: {}", changeSet.getId(), changeLogPath );
            LiquibaseHelper.rollbackChangeSet( changeSet );
        }
        catch ( Exception e )
        {
            throw new LiquibaseExecutionException( "Error during changeSet rollback: " + changeSet.getId(), e );
        }
    }

    public ChangeSet getChangeSet()
    {
        return changeSet;
    }

    @Override
    public String toString()
    {
        return "LiquibaseChange[" + changeLogPath + ", id: " + changeSet.getId() + "]";
    }
}
