package com.suncode.upgrader;

import com.suncode.upgrader.change.Change;
import com.suncode.upgrader.change.ChangeExecutor;
import com.suncode.upgrader.change.ChangeResource;
import com.suncode.upgrader.change.ChangeResult;
import com.suncode.upgrader.change.Changes;
import com.suncode.upgrader.change.DefaultVersionComparator;
import com.suncode.upgrader.database.ChangeLogRepository;
import com.suncode.upgrader.database.DbVersion;
import com.suncode.upgrader.database.NoDbVersionException;
import com.suncode.upgrader.xml.ChangesParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ResourceLoader;
import org.springframework.util.Assert;

import javax.sql.DataSource;
import java.util.List;

/**
 * Klasa udostepnia metody do aktualizacji bazy danych.
 * 
 * @author Łukasz Mocek
 */
public class Upgrader
{
    private static final Logger log = LoggerFactory.getLogger( Upgrader.class );

    private final String project;

    private final ChangeResource changeResource;

    private final ChangesParser changesParser;

    private final ChangeExecutor changeExecutor;

    private final ChangeLogRepository changeLogRepository;

    private boolean requiredDbVersion;

    public static Builder create()
    {
        return new Builder();
    }

    private Upgrader( Builder builder )
    {
        Assert.notNull( builder.dataSource, "[Assertion failed] - this argument is required; it must not be null" );
        Assert.notNull( builder.project, "[Assertion failed] - this argument is required; it must not be null" );

        ClassLoader previousContextClassLoader = Thread.currentThread().getContextClassLoader();

        try
        {
            Thread.currentThread().setContextClassLoader( builder.resourceLoader.getClassLoader() );

            this.project = builder.project;
            this.changeResource = new ChangeResource( builder.changeFile, builder.resourceLoader );

            this.changesParser = (builder.changesParser == null) ? new ChangesParser() : builder.changesParser;
            this.changeLogRepository =
                (builder.changeLogRepository == null) ? new ChangeLogRepository( builder.dataSource ) : builder.changeLogRepository;
            this.changeExecutor =
                (builder.changeExecutor == null) ?
                    new ChangeExecutor( builder.dataSource, changeLogRepository, changeResource ) :
                    builder.changeExecutor;
        }
        finally
        {
            Thread.currentThread().setContextClassLoader( previousContextClassLoader );
        }
    }

    /**
     * Aktualizacja do najnowszej wersji
     * 
     * @return {@link UpgradeStatus} wynik aktualizacji
     */
    public UpgradeStatus upgrade()
    {
        return upgrade( null );
    }

    /**
     * Aktualizacja od zadanej wersji do aktualnej
     * 
     * @param fromVersion wersja od jakiej ma zostac wykonana aktualizacja
     * @return {@link UpgradeStatus} wynik aktualizacji
     */
    public UpgradeStatus upgrade( String fromVersion )
    {
        ClassLoader previousContextClassLoader = Thread.currentThread().getContextClassLoader();

        UpgradeStatus upgradeStatus = new UpgradeStatus();
        try
        {
            Thread.currentThread().setContextClassLoader( changeResource.getResourceLoader().getClassLoader() );

            initDbVersionForProject( project );
            if ( fromVersion == null )
            {
                fromVersion = changeLogRepository.getDbVersion( project );
            }

            List<Change> changesList = changesParser.parse( changeResource, project );
            Changes changes = new Changes( new DefaultVersionComparator(), changesList );

            // TODO dodać tagowanie wersji bazy dla projektu (GitLab - Issue #20)
            return executeChanges( changes, fromVersion, upgradeStatus, false );
        }
        catch ( UpgradeFailedException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new UpgradeFailedException( project, upgradeStatus, e );
        }
        finally
        {
            Thread.currentThread().setContextClassLoader( previousContextClassLoader );
        }
    }

    /**
     * Aktualizuje do zadanej wersji
     *
     * @param toVersion wersja do jakiej ma zostac wykonana aktualizacja
     * @return {@link UpgradeStatus} wynik aktualizacji
     */
    public UpgradeStatus upgradeToVersion( String toVersion )
    {
        ClassLoader previousContextClassLoader = Thread.currentThread().getContextClassLoader();

        UpgradeStatus upgradeStatus = new UpgradeStatus();
        try
        {
            Thread.currentThread().setContextClassLoader( changeResource.getResourceLoader().getClassLoader() );

            initDbVersionForProject( project );

            List<Change> changesList = changesParser.parse( changeResource, project );
            Changes changes = new Changes( new DefaultVersionComparator(), changesList );

            return executeChanges( changes, toVersion, upgradeStatus, true );
        }
        catch ( UpgradeFailedException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new UpgradeFailedException( project, upgradeStatus, e );
        }
        finally
        {
            Thread.currentThread().setContextClassLoader( previousContextClassLoader );
        }
    }

    private void initDbVersionForProject( String project )
    {
        try
        {
            changeLogRepository.getDbVersion( project );
        }
        catch ( NoDbVersionException e )
        {
            if ( isRequiredDbVersion() )
            {
                throw e;
            }
            changeLogRepository.addDbVersion( new DbVersion( project, null ) );
        }

    }

    private UpgradeStatus executeChanges( Changes changes, String version, UpgradeStatus upgradeStatus, boolean toVersion )
    {
        if ( toVersion )
        {
            log.info( "Upgrading project '{}' to version {}. ", project, (version != null ? version : "<none> (all changes will be executed)") );
        }
        else
        {
            log.info( "Upgrading project '{}' to current version starting from {}. ", project, (version != null ? version
                : "<none> (all changes will be executed)") );
        }

        List<Change> changesToExecute = toVersion ? getChangesToVersionToExecute( changes, version ) : getChangesToExecute( changes, version );
        for ( Change change : changesToExecute )
        {
            ChangeResult changeResults = changeExecutor.execute( change );
            upgradeStatus.addChangeResult( changeResults );
            if ( changeResults.isFailed() && change.isFailOnError() )
            {
                if ( change.isMandatory() )
                {
                    log.info( "Execution od changes was interrupted , because execution of mandatory change {} failed. Caused: ",
                              change,
                              changeResults.getException() );
                    throw new UpgradeFailedException( project, upgradeStatus );
                }

                log.info( "Execution of not mandatory change {} failed. Continuing execution of rest changes. Caused: ", change,
                          changeResults.getException() );
            }
            log.info( "Change {} executed with status: {}", change, changeResults.getExecutionStatus() );
        }
        return upgradeStatus;
    }

    private List<Change> getChangesToExecute( Changes changes, String fromVersion )
    {
        if ( fromVersion != null )
        {
            return changes.getChangesFromVersion( fromVersion );
        }
        return changes.getChanges();
    }

    private List<Change> getChangesToVersionToExecute( Changes changes, String toVersion )
    {
        if ( toVersion != null )
        {
            return changes.getChangesToVersion( toVersion );
        }
        return changes.getChanges();
    }

    public boolean isRequiredDbVersion()
    {
        return requiredDbVersion;
    }

    public void setRequiredDbVersion( boolean requiredDbVersion )
    {
        this.requiredDbVersion = requiredDbVersion;
    }

    public static class Builder
    {
        private DataSource dataSource;

        private String project;

        private String changeFile;

        private ResourceLoader resourceLoader;

        private ChangesParser changesParser;

        private ChangeExecutor changeExecutor;

        private ChangeLogRepository changeLogRepository;

        public Builder dataSource( DataSource dataSource )
        {
            this.dataSource = dataSource;
            return this;
        }

        public Builder project( String project )
        {
            this.project = project;
            return this;
        }

        public Builder changeFile( String changeFile )
        {
            this.changeFile = changeFile;
            return this;
        }

        public Builder resourceLoader( ResourceLoader resourceLoader )
        {
            this.resourceLoader = resourceLoader;
            return this;
        }

        Builder changesParser( ChangesParser changesParser )
        {
            this.changesParser = changesParser;
            return this;
        }

        Builder changeExecutor( ChangeExecutor changeExecutor )
        {
            this.changeExecutor = changeExecutor;
            return this;
        }

        Builder changeLogRepository( ChangeLogRepository changeLogRepository )
        {
            this.changeLogRepository = changeLogRepository;
            return this;
        }

        public Upgrader build()
        {
            return new Upgrader( this );
        }
    }
}
