package com.suncode.upgrader.database.transaction;

import java.lang.reflect.UndeclaredThrowableException;
import java.sql.SQLException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Transaction
{

    protected static final Logger log = LoggerFactory.getLogger( Transaction.class );

    private TransactionManager transactionManager;

    public Transaction( TransactionManager tm )
    {
        this.transactionManager = tm;
    }

    public <T> T execute( TransactionCallback<T> action )
        throws RuntimeException
    {
        T result = null;

        try
        {
            transactionManager.setUp();
            result = action.doInTransaction( transactionManager.getConnection() );
            transactionManager.commit();
        }
        catch ( RuntimeException re )
        {
            rollbackOnException( re );
            throw re;
        }
        catch ( Error err )
        {
            rollbackOnException( err );
            throw err;
        }
        catch ( Exception ex )
        {
            rollbackOnException( ex );
            throw new UndeclaredThrowableException( ex, "TransactionCallback threw undeclared checked exception" );
        }
        finally
        {
            try
            {
                transactionManager.finalization();
            }
            catch ( SQLException e )
            {
                throw new DbExecutionException( "Error during transaction finalization" );
            }
        }
        return result;
    }

    private void rollbackOnException( Throwable ex )
        throws RuntimeException
    {
        try
        {
            transactionManager.rollback();
        }
        catch ( SQLException e )
        {
            log.error( "Application exception overridden by rollback error", ex );
            throw new DbExecutionException( "Error during rollback" );
        }
        catch ( RuntimeException re )
        {
            log.error( "Application exception overridden by rollback error", ex );
            throw re;
        }
        catch ( Error err )
        {
            log.error( "Application exception overridden by rollback error", ex );
            throw err;
        }

    }

}
