package com.suncode.upgrader.change;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Set;

import javax.sql.DataSource;

import com.google.common.collect.Lists;
import com.suncode.upgrader.database.DbChangeLog;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.suncode.upgrader.change.liquibase.LiquibaseHelper;
import com.suncode.upgrader.change.task.TaskExecutionException;
import com.suncode.upgrader.database.ChangeLogRepository;
import com.suncode.upgrader.database.DataAccessChangeLogException;
import com.suncode.upgrader.database.SupportedDatabase;
import com.suncode.upgrader.database.transaction.Transaction;
import com.suncode.upgrader.database.transaction.TransactionCallback;
import com.suncode.upgrader.database.transaction.TransactionFactory;
import com.suncode.upgrader.database.transaction.TransactionVoidCallback;

import liquibase.database.Database;
import liquibase.exception.DatabaseException;

/**
 * Komponent odpowiedzialny za wykonywanie zmian
 * 
 * @author Łukasz Mocek
 */
public class ChangeExecutor
{
    private static final Logger log = LoggerFactory.getLogger( ChangeExecutor.class );

    private DataSource dataSource;

    private ChangeLogRepository changeLogRepository;

    private final ChangeResource changeResource;

    public ChangeExecutor( DataSource dataSource, ChangeLogRepository changeLogRepository,
                           ChangeResource changeResource )
    {
        this.dataSource = dataSource;
        this.changeLogRepository = changeLogRepository;
        this.changeResource = changeResource;
    }

    /**
     * Wykonuje podaną zmianę
     * 
     * @param change {@link Change} zmiana do wykonania
     * @return {@link ChangeResult} wynik wykonania zmiany
     */
    public ChangeResult execute( Change change )
    {
        log.info( "Executing change [{}]", change );
        DbChangeLog dbChangeLog = changeLogRepository.getChangeByPk( change.getId(), change.getProject() );
        if ( changeLogRepository.isChangeExecuted( dbChangeLog ) )
        {
            log.info( "Change [{}] was already executed with status [{}]", change,
                      dbChangeLog.getResult() );
            return new ChangeResult( change, ExecutionStatus.ALREADY_EXECUTED );
        }

        ChangeResult changeResult;
        try
        {
            if ( changeLogRepository.isChangeFailed( dbChangeLog ) )
            {
                log.info( "Change [{}] was already executed with status [{}]", change,
                          ExecutionStatus.FAILED );
            }
            changeResult = executeChangeInTransaction( change );
        }
        catch ( Exception e )
        {
            changeResult = new ChangeResult( change, ExecutionStatus.FAILED, e );
        }

        try
        {
            log.debug( "Saving change result [{}] in database", changeResult );
            changeLogRepository.saveChangeResult( changeResult );
        }
        catch ( DataAccessChangeLogException e )
        {
            if ( changeResult.isFailed() )
            {
                // Logujemy wyjątek, który przerwawł wykonywanie zmiany, ponieważ dalej przekazany
                // zostanie wyjątek
                // rzucony w trakcie zapisu ChangeResult
                log.error( "Original exception will be override by exception during saving change result. Original exception: ",
                           changeResult.getException() );
            }
            // Jeżeli zmiana się nie wykonała to nie ma potrzeby wykonania rollbacku,
            // bo i tak transakcja ze zmianą nie została zatwierdzona
            if ( changeResult.isExecuted() )
            {
                try
                {
                    rollbackChangeInTrasaction( change );
                }
                catch ( Exception e1 )
                {
                    log.error( "FATAL ERROR! Can't rollback change!", e1 );
                }
            }
            changeResult = new ChangeResult( change, ExecutionStatus.FAILED, e );
        }
        log.info( "Change [{}] executed with status [{}]", changeResult.getChange().getId(), changeResult.getExecutionStatus() );
        return changeResult;
    }

    private ChangeResult executeChangeInTransaction( final Change change )
        throws TaskExecutionException
    {
        Transaction transaction = TransactionFactory.getTransaction( dataSource );
        ChangeResult changeResult = transaction.execute( new TransactionCallback<ChangeResult>() {
            @Override
            public ChangeResult doInTransaction( Connection connection )
                throws TaskExecutionException, SQLException
            {
                try
                {
                    ChangeContext.init( change, connection, changeResource );
                    if ( !runOnThisDb( connection, change.getTarget() ) )
                    {
                        log.debug( "Change {} skipped, because current database wasn't definied in attribute 'target'.",
                                   change );
                        return new ChangeResult( change, ExecutionStatus.SKIPPED );
                    }
                    return change.run();
                }
                finally
                {
                    ChangeContext.clear();
                }
            }
        } );
        return changeResult;
    }

    private boolean runOnThisDb( Connection connection, Set<SupportedDatabase> target )
    {
        try
        {
            if ( target.isEmpty() )
            {
                return true;
            }

            Database database = LiquibaseHelper.initDataBase( connection );
            if ( target.contains( SupportedDatabase.valueOf( database.getShortName().toUpperCase() ) ) )
            {
                return true;
            }
            return false;
        }
        catch ( DatabaseException e )
        {
            return false;
        }
    }

    private void rollbackChangeInTrasaction( final Change change )
        throws TaskExecutionException
    {
        Transaction transaction = TransactionFactory.getTransaction( dataSource );
        transaction.execute( new TransactionVoidCallback() {

            @Override
            protected void doInTransactionVoid( Connection connection )
            {
                try
                {
                    ChangeContext.init( change, connection, changeResource );
                    change.rollback();
                }
                finally
                {
                    ChangeContext.clear();
                }
            }
        } );
    }
}
