package com.suncode.upgrader.database;

import java.util.List;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;

import com.google.common.collect.Lists;
import com.suncode.upgrader.UpgraderSchema;
import com.suncode.upgrader.change.Change;
import com.suncode.upgrader.change.ChangeResult;
import com.suncode.upgrader.change.ExecutionStatus;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

public class ChangeLogRepository
{
    private static final Logger log = LoggerFactory.getLogger( ChangeLogRepository.class );

    public static final String db_version = "db_version";

    public static final String db_changelog = "db_changelog";

    private final DBChangeLogMapper mapper = new DBChangeLogMapper();

    private final JdbcTemplate jdbcTemplate;

    private final NamedParameterJdbcTemplate namedParameterJdbcTemplate;

    private final PlatformTransactionManager transactionManager;

    public ChangeLogRepository( DataSource dataSource )
    {
        this.jdbcTemplate = new JdbcTemplate( dataSource );
        this.namedParameterJdbcTemplate = new NamedParameterJdbcTemplate( jdbcTemplate );
        this.transactionManager = new DataSourceTransactionManager( dataSource );

        UpgraderSchema.init( dataSource );
    }

    public String getDbVersion( String project )
        throws NoDbVersionException
    {
        String select = "SELECT version FROM " + db_version + " WHERE project = ?";
        try
        {
            return jdbcTemplate.queryForObject( select, String.class, project );
        }
        catch ( DataAccessException e )
        {
            throw new NoDbVersionException( "For a given project: " + project + " does not set version." );
        }
    }

    public void addDbVersion( DbVersion dbVersion )
    {
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        TransactionStatus transactionStatus = transactionManager.getTransaction( transactionDefinition );

        try
        {
            String insert = "INSERT INTO " + db_version + " (project, version) VALUES (:project, :version)";
            namedParameterJdbcTemplate.update( insert, dbVersion.mapFieldsToMap() );
            transactionManager.commit( transactionStatus );

            log.debug( "To table '{}' added row ({}, {})", db_version, dbVersion.getProject(), dbVersion.getVersion() );
        }
        catch ( Exception ex )
        {
            transactionManager.rollback( transactionStatus );
        }
    }

    public void updateDbVersion( DbVersion dbVersion )
    {
        String update = "UPDATE " + db_version + " SET project=:project, version=:version";
        namedParameterJdbcTemplate.update( update, dbVersion.mapFieldsToMap() );
        log.debug( "In table '{}' update row to value ({}, {})", db_version, dbVersion.getProject(), dbVersion.getVersion() );
    }

    public List<DbChangeLog> getAllChanges( String id, String project )
    {
        String select = "SELECT id, project, change_date, change_result, message, version FROM " + db_changelog
            + " WHERE id = ? AND project = ?";
        Object[] parameters = new Object[] { id, project };
        return jdbcTemplate.query( select, parameters, new DBChangeLogMapper() );
    }

    public DbChangeLog getChangeByPk( String id, String project )
    {
        String select = "SELECT * FROM " + db_changelog + " WHERE id = ? AND project = ?";
        Object[] parameters = new Object[] { id, project };
        List<DbChangeLog> result = jdbcTemplate.query( select, parameters, new DBChangeLogMapper() );
        return result.isEmpty() ? null : result.get( 0 );
    }

    public int addChangeLog( DbChangeLog changeLog )
    {
        String insert =
            "INSERT INTO " + db_changelog
                + "(id, project, change_date, change_result, message, version) VALUES (:id, :project, :date, :result, :message, :version)";
        return namedParameterJdbcTemplate.update( insert, changeLog.mapFieldsToMap() );
    }

    public void updateChangeLog( DbChangeLog changeLog )
    {
        String update = "UPDATE " + db_changelog
            + " SET change_date=:date, change_result=:result, message=:message, version=:version WHERE id=:id AND project=:project";
        namedParameterJdbcTemplate.update( update, changeLog.mapFieldsToMap() );
    }

    public void saveChangeResult( ChangeResult changeResult )
        throws DataAccessChangeLogException
    {
        DbChangeLog dbChangeLog = mapper.mapChangeResult( changeResult );
        try
        {
            if ( isChangeInChangeLog( changeResult.getChange() ) )
            {
                updateChangeLog( dbChangeLog );
            }
            else
            {
                addChangeLog( dbChangeLog );
            }
        }
        catch ( DataAccessException e )
        {
            throw new DataAccessChangeLogException( "Error during saving changeLog", e );
        }
    }

    private boolean isChangeInChangeLog( Change change )
    {
        DbChangeLog result = getChangeByPk( change.getId(), change.getProject() );
        return result != null;
    }

    public boolean isChangeExecuted( Change change )
    {
        DbChangeLog result = getChangeByPk( change.getId(), change.getProject() );
        return result != null && result.hasStatus( Lists.newArrayList( ExecutionStatus.EXECUTED, ExecutionStatus.SKIPPED ) ) ;
    }

    public boolean isChangeExecuted( DbChangeLog dbChangeLog )
    {
        return dbChangeLog != null && dbChangeLog.hasStatus( Lists.newArrayList( ExecutionStatus.EXECUTED, ExecutionStatus.SKIPPED ) );
    }

    public boolean isChangeFailed( DbChangeLog dbChangeLog )
    {
        return dbChangeLog != null && dbChangeLog.hasStatus( Lists.newArrayList( ExecutionStatus.FAILED ) );
    }
}
