package com.suncode.upgrader.xml;

import java.io.InputStream;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.io.IOUtils;
import org.springframework.util.Assert;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.suncode.upgrader.change.Change;
import com.suncode.upgrader.change.ChangeResource;
import com.suncode.upgrader.xml.liquibase.LiquibaseElementParser;
import com.suncode.upgrader.xml.task.ChangeElementParser;

/**
 * Parser zmian
 * 
 * @author Marcin Macias 17 wrz 2015
 */
public class ChangesParser
{
    private Map<String, ElementParser> elementParsers = Maps.newHashMap();

    public ChangesParser()
    {
        LiquibaseElementParser liquibase = new LiquibaseElementParser();
        ChangeElementParser change = new ChangeElementParser();
        IncludeElementParser include = new IncludeElementParser( this );

        elementParsers.put( liquibase.supportedElementTag(), liquibase );
        elementParsers.put( change.supportedElementTag(), change );
        elementParsers.put( include.supportedElementTag(), include );
    }

    /**
     * Odczytuje zmiany z podanego pliku XML.
     * 
     * @param changeResource plik XML ze zmianami
     * @param project
     * @return
     */
    public List<Change> parse( ChangeResource changeResource, String project )
        throws ChangesParseException
    {
        Assert.notNull( changeResource, "[Assertion failed] - this argument is required; it must not be null" );
        InputStream stream = null;
        
        try
        {
            stream = changeResource.get().getInputStream();
            if ( stream == null )
            {
                throw new IllegalArgumentException( "Classpath resource [" + changeResource + "] not found" );
            }
            DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
            DocumentBuilder db = dbf.newDocumentBuilder();
            Document doc = db.parse( stream );
            return parseChanges( doc, changeResource, project );
        }
        catch ( Exception e )
        {
            throw new ChangesParseException( "Error while parsing changes xml [" + changeResource + "]", e );
        }
        finally
        {
            IOUtils.closeQuietly( stream );
        }
    }

    private List<Change> parseChanges( Document doc, ChangeResource changeResource, String project )
        throws Exception
    {
        Element changesEl = doc.getDocumentElement();

        List<Change> changes = Lists.newArrayList();
        NodeList changesList = changesEl.getChildNodes();
        for ( int i = 0; i < changesList.getLength(); i++ )
        {
            Node node = changesList.item( i );
            if ( node.getNodeType() == Node.ELEMENT_NODE )
            {
                List<Change> parsedChanges = readChange( (Element) node, changeResource, project );
                changes.addAll( parsedChanges );
            }
        }
        return changes;
    }

    private List<Change> readChange( Element changeNode, ChangeResource changeResource, String project )
        throws Exception
    {
        ElementParser parser = elementParsers.get( changeNode.getNodeName() );
        if ( parser == null )
        {
            throw new IllegalArgumentException( "ElementParser for tag '" + changeNode.getNodeName()
                + "' doesn't exist" );
        }
        return parser.parse( changeNode, project, changeResource );
    }
}
