package com.suncode.upgrader.change.liquibase;

import com.google.common.collect.ImmutableMap;
import com.suncode.upgrader.change.ChangeContext;
import com.suncode.upgrader.change.ChangeResource;
import com.suncode.upgrader.change.ExecutionStatus;
import liquibase.Scope;
import liquibase.changelog.ChangeSet;
import liquibase.changelog.ChangeSet.ExecType;
import liquibase.changelog.DatabaseChangeLog;
import liquibase.database.Database;
import liquibase.database.DatabaseConnection;
import liquibase.database.DatabaseFactory;
import liquibase.database.core.UnsupportedDatabase;
import liquibase.database.jvm.JdbcConnection;
import liquibase.exception.ChangeLogParseException;
import liquibase.exception.DatabaseException;
import liquibase.exception.MigrationFailedException;
import liquibase.exception.RollbackFailedException;
import liquibase.parser.core.xml.XMLChangeLogSAXParser;
import liquibase.resource.ResourceAccessor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;

import java.sql.Connection;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Komponent zarządzajacy metodami Liquibase
 * 
 * @author Marcin Macias 15 wrz 2015
 */
public class LiquibaseHelper
{
    private static final Logger log = LoggerFactory.getLogger( LiquibaseHelper.class );

    public static void executeChangeLog( String path, ResourceLoader resourceLoader, Connection connection )
        throws ChangeLogParseException, DatabaseException, MigrationFailedException
    {
        executeChangeLog( path, null, resourceLoader, connection );
    }

    public static void executeChangeLog( String path, Resource relativeTo, ResourceLoader resourceLoader,
                                         Connection connection )
        throws ChangeLogParseException, DatabaseException, MigrationFailedException
    {
        LiquibaseTaskResourceAccessor accessor = new LiquibaseTaskResourceAccessor( relativeTo, resourceLoader );

        try
        {
            Scope.child(
                ImmutableMap.of( Scope.Attr.resourceAccessor.name(), accessor ),
                () -> {
                    DatabaseChangeLog changeLog = getChangeLog( path, relativeTo, accessor );
                    Database db = initDataBase( connection );
                    executeChangeSets( changeLog, db );
                }
            );
        }
        catch ( ChangeLogParseException | DatabaseException | MigrationFailedException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RuntimeException( "Failed to execute change sets", e );
        }
    }

    private static DatabaseChangeLog getChangeLog( String path, final Resource relativeTo,
                                                   ResourceAccessor accessor )
        throws ChangeLogParseException
    {
        if ( relativeTo == null )
        {
            if ( path.startsWith( "." ) )
            {
                throw new IllegalArgumentException( "Path can not be relative" );
            }
        }

        XMLChangeLogSAXParser xmlChangeLogSAXParser = new XMLChangeLogSAXParser();
        return xmlChangeLogSAXParser.parse( path, null, accessor);
    }

    public static Database initDataBase( Connection connection )
        throws DatabaseException
    {
        DatabaseConnection dBConection = new JdbcConnection( connection );
        Database db = DatabaseFactory.getInstance().findCorrectDatabaseImplementation( dBConection );
        if ( db instanceof UnsupportedDatabase )
        {
            throw new DatabaseException( "Unsupported database." );
        }
        return db;
    }

    private static void executeChangeSets( DatabaseChangeLog changeLog, Database db )
        throws MigrationFailedException
    {
        for ( ChangeSet changeSet : changeLog.getChangeSets() )
        {
            changeSet.setFailOnError( true );
            ExecType execType = changeSet.execute( changeLog, db );
            assertExecutionResult( executionStatusOf( execType ) );
        }
    }

    public static ExecutionStatus executeChangeSet( ChangeSet changeSet )
        throws DatabaseException, MigrationFailedException
    {
        try
        {
            ChangeResource changeResource = ChangeContext.changeResource();
            LiquibaseTaskResourceAccessor accessor = new LiquibaseTaskResourceAccessor( changeResource.get(),
                                                                                        changeResource.getResourceLoader() );
            return Scope.child(
                ImmutableMap.of( Scope.Attr.resourceAccessor.name(), accessor ),
                () -> {
                    ExecType execType = changeSet.execute( changeSet.getChangeLog(), getDatabase() );
                    ExecutionStatus executionStatus = executionStatusOf( execType );
                    log.debug( "Changed ExecType [{}] to ExecutionStatus [{}]", execType, executionStatus );
                    assertExecutionResult( executionStatus );
                    return executionStatus;
                }
            );
        }
        catch ( DatabaseException | MigrationFailedException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new MigrationFailedException( changeSet, "Failed to execute change set", e );
        }
    }

    public static void rollbackChangeSet( ChangeSet changeSet )
        throws RollbackFailedException, DatabaseException
    {
        changeSet.rollback( getDatabase() );
    }

    private static Database getDatabase()
        throws DatabaseException
    {
        Connection connection = ChangeContext.get().getConnection();
        return initDataBase( connection );
    }

    private static void assertExecutionResult( ExecutionStatus executionStatus )
        throws LiquibaseExecutionException
    {
        if ( executionStatus == ExecutionStatus.FAILED )
        {
            // Nie powinno się zdarzyć.
            // Mimo ustawienia failOnError na true wynikiem wywołania metody execute jest FAILED,
            // a metoda nie rzuciła wyjątku. Prawdopodobnie błąd w kodzie.
            throw new LiquibaseExecutionException( "Execution of change didn't throw exception though return FAILED status. Check logs." );
        }
    }

    private static ExecutionStatus executionStatusOf( ExecType result )
    {
        switch ( result )
        {
            case EXECUTED:
                return ExecutionStatus.EXECUTED;
            case FAILED:
                return ExecutionStatus.FAILED;
            case SKIPPED:
                return ExecutionStatus.SKIPPED;
            default:
                return ExecutionStatus.FAILED;
        }
    }
}
