package com.suncode.upgrader.change;

import java.util.regex.Pattern;

import org.springframework.util.Assert;

/**
 * Domyslna implementacja interfejsu komparatora wersji Change. Komparator wykorzystuje następujący schemat zapisu
 * wersji: trzy segmenty numeryczne rozdzielone '.' i opcjonalny segment 'SRX', gdzie X to liczba. Przykładowo:
 * "3.1.12", "3.1.14.SR1"
 */
public class DefaultVersionComparator
    implements VersionComparator
{

    private static final String VERSION_PATTERN = "^\\d+(\\.\\d+){2}(\\.SR\\d+)?$";

    private static final String SPLIT_PARAM = "\\.";

    private static final int MAX_NUMBER_OF_SECTION = 4;

    public int compare( String firstVersion, String secondVersion )
    {
        Assert.notNull( firstVersion, "[Assertion failed] - this argument is required; it must not be null" );
        Assert.notNull( secondVersion, "[Assertion failed] - this argument is required; it must not be null" );

        if ( !checkVersionFormat( firstVersion ) || !checkVersionFormat( secondVersion ) )
        {
            throw new IllegalArgumentException( "One of following versions has invalid format: [" + firstVersion
                + "] or [" + secondVersion + "]" );
        }

        String[] splitedVersion0 = firstVersion.split( SPLIT_PARAM );
        String[] splitedVersion1 = secondVersion.split( SPLIT_PARAM );

        int nonEqualSectionIndex = searchNonEqualSectionIndex( splitedVersion0, splitedVersion1 );

        if ( isNonEqualSection( splitedVersion0, splitedVersion1, nonEqualSectionIndex ) )
        {
            int diff = compareNonEqualSection( splitedVersion0, splitedVersion1, nonEqualSectionIndex );
            return Integer.signum( diff );
        }
        else
        {
            // ciagi są równe lub jeden jest podciagiem drugiego
            // np. "3.1.12" = "3.1.12" lub "3.1.12" < "3.1.12.SR1"
            return Integer.signum( splitedVersion0.length - splitedVersion1.length );
        }
    }

    private boolean isNonEqualSection( String[] splitedVersion0, String[] splitedVersion1, int nonEqualSectionIndex )
    {
        return nonEqualSectionIndex < splitedVersion0.length && nonEqualSectionIndex < splitedVersion1.length;
    }

    private int compareNonEqualSection( String[] splitedVersion0, String[] splitedVersion1, int nonEqualSectionIndex )
    {
        String version0Section = splitedVersion0[nonEqualSectionIndex];
        String version1Section = splitedVersion1[nonEqualSectionIndex];

        if ( isLastSection( nonEqualSectionIndex ) )
        {
            // ucięcie "SR"
            version0Section = version0Section.substring( 2 );
            version1Section = version1Section.substring( 2 );
        }

        return Integer.valueOf( version0Section ).compareTo( Integer.valueOf( version1Section ) );
    }

    private boolean isLastSection( int nonEqualSectionIndex )
    {
        return nonEqualSectionIndex == MAX_NUMBER_OF_SECTION - 1;
    }

    private int searchNonEqualSectionIndex( String[] splitedVersion0, String[] splitedVersion1 )
    {
        int index = 0;

        while ( isNonEqualSection( splitedVersion0, splitedVersion1, index )
            && splitedVersion0[index].equals( splitedVersion1[index] ) )
        {
            index++;
        }

        return index;
    }

    private boolean checkVersionFormat( String version )
    {
        Assert.notNull( version, "[Assertion failed] - this argument is required; it must not be null" );
        return Pattern.matches( VERSION_PATTERN, version );
    }
}
