package com.suncode.upgrader.change;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import org.springframework.util.Assert;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public class Changes
{
    private VersionComparator comparator;

    private final Map<String, Change> changes;

    public Changes( VersionComparator comparator, List<Change> changes )
    {
        Assert.notNull( comparator, "[Assertion failed] - this argument is required; it must not be null" );
        Assert.notNull( changes, "[Assertion failed] - this argument is required; it must not be null" );
        this.comparator = comparator;
        this.changes = setChanges( changes );
    }

    private Map<String, Change> setChanges( List<Change> changeList )
    {
        Map<String, Change> changeMap = new LinkedHashMap<String, Change>();
        for ( Change change : changeList )
        {
            if ( changeMap.put( change.getId(), change ) != null )
            {
                throw new IllegalArgumentException( "Change id is not unique: " + change.getId() );
            }
        }
        return changeMap;
    }

    /**
     * Pobiera zmianę dla danego identyfikatora
     */
    public Change getChange( String id )
    {
        Assert.hasText( id, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );
        return changes.get( id );
    }

    public List<Change> getChanges()
    {
        return ImmutableList.copyOf( changes.values() );
    }

    /**
     * Pobiera listę zmian dla danej wersji
     */
    public List<Change> getChanges( String version )
    {
        List<Change> result = Lists.newArrayList();
        for ( Change change : changes.values() )
        {
            if ( comparator.compare( change.getVersion(), version ) == 0 )
            {
                result.add( change );
            }
        }
        return result;
    }

    /**
     * Pobiera listę zmian należących do danego przedziału
     */
    public List<Change> getChanges( String sourceVersion, String targetVersion )
    {
        List<Change> result = Lists.newArrayList();
        for ( Change change : changes.values() )
        {
            if ( comparator.compare( change.getVersion(), sourceVersion ) >= 0
                && comparator.compare( change.getVersion(), targetVersion ) < 1 )
            {
                result.add( change );
            }
        }
        return result;
    }

    /**
     * Pobiera listę zmian od danej wersji włączając bądź wykluczając podaną wersję
     */
    public List<Change> getChangesFromVersion( String sourceVersion, boolean includeSourceVersion )
    {
        List<Change> result = Lists.newArrayList();
        for ( Change change : changes.values() )
        {
            int difference = comparator.compare( change.getVersion(), sourceVersion );
            if ( includeSourceVersion && difference >= 0 )
            {
                result.add( change );
            }
            else if ( difference == 1 )
            {
                result.add( change );
            }
        }
        return result;
    }

    /**
     * Pobiera listę wszystkich zmian od podanej wersji
     */
    public List<Change> getChangesFromVersion( String sourceVersion )
    {
        return getChangesFromVersion( sourceVersion, true );
    }

    /**
     * Pobiera listę wszystkich zmian do podanej wersji
     */
    public List<Change> getChangesToVersion( String targetVersion )
    {
        return getChanges( "0.0.0", targetVersion );
    }
}
