package com.suncode.upgrader.change;

import java.sql.Connection;

import org.springframework.core.io.ResourceLoader;

/**
 * Klasa przechowująca informacje kontekstowe zmian {@linkplain Change}.
 * <p>
 * Kontekst jest aktywny całe wykonywanie zmiany przez {@linkplain ChangeExecutor#execute(Change)}.
 * Inicjowany jest przed rozpoczęciem wykonywania zmiany i czyszczony jest po jej wykonaniu.
 * <p>
 * Obiekty przechowywane w kontekscie:
 * <ol>
 * <li>{@link Change}
 * <li>{@link Connection}
 * <li>{@link ChangeResource}
 * </ol>
 * <p>
 * Jeżeli podczas wykonywania zadania potrzebne jest połączenie to powinno ono byc pobierane z
 * {@linkplain ChangeContext} w następujący sposób:
 * <pre>
 * <code>
 * Connection connection = ChangeContext.getConnection();
 * </code>
 * </pre>
 * 
 * @author Łukasz Mocek
 */
public class ChangeContext
{
    private static final ThreadLocal<ChangeContext> threadContext = new ThreadLocal<ChangeContext>();

    private Change change;

    private Connection connection;

    private ChangeResource changeResource;

    private ChangeContext( Change change, Connection connection, ChangeResource changeResource )
    {
        this.connection = connection;
        this.changeResource = changeResource;
    }

    public static ChangeContext get()
    {
        ChangeContext context = threadContext.get();
        if ( context == null )
        {
            throw new IllegalStateException( "No active context" );
        }
        return context;
    }

    public static boolean isActive()
    {
        return threadContext.get() != null;
    }

    public static void init( Change change, Connection connection, ChangeResource changeResource )
    {
        threadContext.set( new ChangeContext( change, connection, changeResource ) );
    }

    public static void clear()
    {
        threadContext.remove();
    }

    public static Connection connection()
    {
        return get().getConnection();
    }

    public static ChangeResource changeResource()
    {
        return get().getChangeResource();
    }
    
    public static ResourceLoader resourceLoader()
    {
        return changeResource().getResourceLoader();
    }
    
    public static ClassLoader classLoader()
    {
        return resourceLoader().getClassLoader();
    }

    public Change getChange()
    {
        return change;
    }

    public Connection getConnection()
    {
        return connection;
    }

    public ChangeResource getChangeResource()
    {
        return changeResource;
    }
}
