Ext.namespace('Ext.ux.plusmpm.scheduledtasks');

/**
 * Okno edycji istniejącego zadania
 */
Ext.ux.plusmpm.scheduledtasks.EditTaskWindow = function (config) {
  config = Ext.apply(
    {
      width: 1000,
      y: 50,
      autoHeight: true,
      shadow: false,
      modal: true,
      layout: 'fit',
      title: PW.t('editScheduledTaskWizard'),
      items: [
        {
          xtype: 'form',
          autoHeight: true,
          border: false,
          ref: 'wizard',
          layout: 'card',
          activeItem: 0,
          bbar: [
            '->',
            {
              text: '&laquo; ' + PW.t('back'),
              handler: this.onPreviousHandler,
              disabled: true,
              scope: this,
              ref: '../../prevButton',
            },
            {
              text: PW.t('next') + ' &raquo;',
              handler: this.onNextHandler,
              scope: this,
              ref: '../../nextButton',
            },
            {
              text: PW.t('save'),
              handler: this.onSaveHandler,
              hidden: true,
              scope: this,
              iconCls: 'dvnt-icon-save',
              ref: '../../addButton',
            },
          ],

          items: [
            {
              xtype: 'st_edit_wizard_step1',
              ref: 'step1',
              taskRecord: config.taskRecord,
            },
            {
              xtype: 'st_edit_wizard_step2',
              taskRecord: config.taskRecord,
              editable: true,
            },
          ],
        },
      ],
    },
    config
  );

  Ext.ux.plusmpm.scheduledtasks.EditTaskWindow.superclass.constructor.call(this, config);
}; // end of Ext.ux.plusmpm.scheduledtasks.EditTaskWindow constructor

Ext.extend(Ext.ux.plusmpm.scheduledtasks.EditTaskWindow, Ext.Window, {
  /**
   * Confnięcie na poprzednią kartę
   */
  onPreviousHandler: function () {
    this.wizard.getLayout().setActiveItem(0);
    this.prevButton.setDisabled(true);
    this.nextButton.setVisible(true);
    this.addButton.setVisible(false);
  },

  /**
   * Przełączenie na następną kartę
   */
  onNextHandler: function (record) {
    if (this.validateParameters()) {
      this.wizard.getLayout().setActiveItem(1);
      this.prevButton.setDisabled(false);
      this.nextButton.setVisible(false);
      this.addButton.setVisible(true);
    }
  },

  validateParameters: function () {
    var hasErrors = false;

    var allFields = this.getAllParameterFields();
    allFields.forEach(function (item) {
      if (!item.validate()) {
        hasErrors = true;
      }
    });

    return !hasErrors;
  },

  getAllParameterFields: function () {
    var paramsFieldSet = this.wizard.step1.paramsFieldSet;
    if (!paramsFieldSet) {
      return [];
    }

    return paramsFieldSet.parametersContainer.findBy(function (cmp) {
      return cmp.isFormField;
    }, this);
  },

  onSaveHandler: async function () {
    var form = this.wizard.getForm();

    var params = await this._buildParams();

    if (form.isValid()) {
      form.submit({
        url: 'api/scheduledtasks/edit',
        method: 'POST',
        params: {
          params: Ext.encode(params),
        },
        scope: this,
        waitMsg: PW.t('editingScheduledTask'),
        success: function (form, action) {
          Ext.StoreMgr.get('tasksCategoryStore').load();
          this.close();
        },
        failure: function (form, action) {
          Ext.Msg.show({
            title: PW.t('error'),
            msg: action?.result?.error ?? PW.t('failedToEditTask'),
            buttons: Ext.Msg.OK,
            icon: Ext.MessageBox.ERROR,
          });
        },
      });
    }
  },

  _buildParams: async function () {
    var definitions = this.taskRecord.data.parameters;
    var fields = this.wizard.findBy(function (comp) {
      return comp.isFormField && !comp.disabled && comp.isScheduledTaskParameter;
    }, this);

    var params = [];

    for (var definition of definitions) {
      var arrayValue = [];
      var isArray = definition.type.indexOf('[') !== -1;

      for (var field of fields) {
        var fieldMatchesDefinition = this._hasMatchingComponentParameterIdOrPosition(field, definition);

        if (fieldMatchesDefinition) {
          var singleValue;
          if (field.xtype === 'checkbox') {
            singleValue = {
              value: String(field.checked),
            };
          } else if (field.xtype === 'fileuploadfield') {
            var files = field.fileInput.dom.files;
            var file = files[0];
            if (files.length > 0) {
              singleValue = {
                value: await this._toBase64(file),
                metadata: {
                  fileName: file.name,
                },
              };
            } else {
              singleValue = {
                value: '',
                metadata: {
                  fileName: 'empty.txt',
                },
              };
            }

            if (field.fileId) {
              Ext.apply(singleValue.metadata, {
                fileId: field.fileId,
              });
            }
          } else if (field.xtype === 'combo') {
            singleValue = {
              value: field.getValue(),
            };
          } else {
            singleValue = {
              value: field.getRawValue(),
            };
          }

          singleValue = Ext.apply(
            {
              valueType: 'single',
            },
            singleValue
          );
          arrayValue.push(singleValue);
        }
      }

      var value;
      if (isArray) {
        value = {
          valueType: 'array',
          value: arrayValue,
        };
      } else if (arrayValue.length > 0) {
        value = arrayValue[0];
      } else {
        value = {
          valueType: 'single',
        };
      }

      params.push({
        position: definition.position,
        value: value,
        type: definition.type,
        componentParameterId: definition.componentParameterId,
      });
    }
    return params;
  },

  _toBase64: function (file) {
    return new Promise(resolve => {
      var reader = new FileReader();
      reader.readAsDataURL(file);
      reader.onload = function () {
        resolve(reader.result);
      };
      reader.onerror = function (error) {
        resolve('');
      };
    });
  },

  _hasMatchingComponentParameterIdOrPosition: function (field, definition) {
    if (definition.componentParameterId) {
      return definition.componentParameterId == field.componentParameterId;
    }

    return definition.position == field.parameterPosition;
  },

  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.EditTaskWindow.superclass.initComponent.call(this);
  },
}); // end of Ext.extend

/**
 * Krok 1 - wybranie klasy zadania, wyszukiwanie zadań
 */
Ext.ux.plusmpm.scheduledtasks.TaskEditStep1 = function (config) {
  var task = config.taskRecord.data;

  config = Ext.apply(
    {
      autoHeight: true,
      layout: 'form',
      style: 'padding: 10px',
      items: [
        {
          xtype: 'hidden',
          name: 'id',
          value: task.id,
        },
        {
          xtype: 'textfield',
          name: 'staticId',
          value: task.staticId,
          fieldLabel: PW.t('staticId'),
          disabled: true,
          anchor: '95%',
        },
        {
          xtype: 'compositefield',
          fieldLabel: PW.t('className'),
          anchor: '95%',
          msgTarget: 'side',
          ref: 'classNameField',
          buildCombinedErrorMessage: function (errors) {
            return errors[0].error;
          },
          items: [
            {
              xtype: 'textfield',
              name: 'classname',
              allowBlank: false,
              flex: 1,
              disabled: true,
              value: task.className,
            },
          ],
        },
        {
          xtype: 'combo',
          name: 'method',
          fieldLabel: PW.t('selectTheMethod'),
          anchor: '95%',
          emptyText: PW.t('selectTheMethod'),
          value: task.methodName,
          mode: 'local',
          store: [task.methodName],
          triggerAction: 'all',
          forceSelection: true,
          disabled: true,
          valueField: 'name',
          displayField: 'name',
          ref: 'methodComboBox',
        },
      ],
    },
    config
  );

  Ext.ux.plusmpm.scheduledtasks.TaskEditStep1.superclass.constructor.call(this, config);
}; // end of Ext.ux.plusmpm.scheduledtasks.TaskEditStep1 constructor

Ext.extend(Ext.ux.plusmpm.scheduledtasks.TaskEditStep1, Ext.Container, {
  descImg: '<img style="padding-left: 5px" src="pages/images/tooltip.png"/>',

  /**
   * Przejscie do następnego kroku, walidacja danych...
   * @param handler zakonczono walidację
   */
  onStepAccept: function (handler) {},

  addParametersFieldSet: function (data) {
    if (this.paramsFieldSet !== undefined) {
      this.remove(this.paramsFieldSet);
      this.paramsFieldSet = undefined;
    }

    if (data.parameters.length > 0) {
      var parametersDefinition = new Ext.util.MixedCollection();
      Ext.each(
        this.taskRecord.data.parameters,
        function (parameter) {
          var definition = {
            componentParamId: parameter.componentParameterId,
            name: parameter.annotationName,
            description: parameter.annotationDesc,
            optional: parameter.isOptional,
            position: parameter.position,
            type: parameter.type,
            fieldType: parameter.fieldType,
            value: parameter.id ? parameter.value : parameter.defaultValue,
            valueMetadata: parameter.valueMetadata,
          };
          parametersDefinition.add(parameter.componentParameterId || parameter.position, definition);
        },
        this
      );

      var parametersBuilder = new Ext.ux.plusmpm.scheduledtasks.ParametersBuilder({
        parametersDefinition: parametersDefinition,
      });

      this.paramsFieldSet = Ext.create({
        xtype: 'fieldset',
        title: PW.t('parameters'),
        collapsible: true,
        items: [
          {
            xtype: 'st_parameterscontainer',
            id: 'st-parameterscontainer',
            style: 'padding: 5px;',
            ref: 'parametersContainer',

            window: this,
            parametersBuilder: parametersBuilder,
            isInnerContainer: false,
            editable: true,
          },
        ],
      });

      this.paramsFieldSet.parametersContainer.buildParameters(this.taskRecord.data.componentTaskId);

      if (this.paramsFieldSet.parametersContainer.items?.getCount()) {
        this.add(this.paramsFieldSet);
      }
    }

    this.doLayout();
    this.ownerCt.setHeight(this.getHeight() + 20);
  },

  onActivate: function () {
    this.ownerCt.setHeight(this.getHeight() + 20);
  },

  onAfterRender: function () {
    this.addParametersFieldSet(this.taskRecord.data);
  },

  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.TaskEditStep1.superclass.initComponent.call(this);
    this.on('activate', this.onActivate, this);
    this.on('afterrender', this.onAfterRender, this);
  },
}); // end of Ext.extend
Ext.reg('st_edit_wizard_step1', Ext.ux.plusmpm.scheduledtasks.TaskEditStep1);

/**
 * Krok 1 - wybranie klasy zadania, wyszukiwanie zadań
 */
Ext.ux.plusmpm.scheduledtasks.TaskEditStep2 = function (config) {
  var task = config.taskRecord.data;
  var isEditable = config.editable;

  var me = this;

  config = Ext.apply(
    {
      layout: 'form',
      style: 'padding: 10px',
      autoHeight: true,
      listeners: {
        show: function () {
          //robimy to, bo na IE nie do końca dobrze renderują się wszystkie pola
          this.doLayout();
        },
      },
      items: [
        {
          xtype: 'fieldset',
          title: PW.t('taskInformation'),
          labelWidth: 150,
          collapsible: true,
          collapsed: false,
          defaults: {
            xtype: 'textfield',
            anchor: '95%',
            disabled: !isEditable,
          },
          items: [
            {
              name: 'staticId',
              value: task.staticId,
              fieldLabel: PW.t('staticId'),
              disabled: true,
            },
            {
              name: 'name',
              value: task.name,
              fieldLabel: PW.t('taskName'),
              allowBlank: false,
              ref: '../nameField',
            },
            {
              name: 'description',
              value: task.description,
              fieldLabel: PW.t('taskDescription'),
              ref: '../descField',
            },
            {
              xtype: 'combo',
              name: 'planMode',
              fieldLabel: PW.t('planMode'),
              editable: false,
              forceSelection: true,
              allowBlank: false,
              mode: 'local',
              displayField: 'text',
              valueField: 'planMode',
              hiddenName: 'planMode',
              flex: 1,
              triggerAction: 'all',
              store: new Ext.data.ArrayStore({
                fields: ['text', 'planMode'],
                data: [
                  [PW.t('runOnceMode'), 'RUN_ONCE'],
                  [PW.t('periodicMode'), 'PERIODIC'],
                  [PW.t('atTheEndOfMonthMode'), 'AT_THE_END_OF_MONTH'],
                ],
              }),
              listeners: {
                afterrender: function () {
                  this.setValue(task.planMode);
                  me.onPlanModeSelect.call(me, task.planMode)
                },
                select: function (combo, record) {
                  me.onPlanModeSelect.call(me, record.data.planMode)
                },
              },
            },
            {
              xtype: 'fieldset',
              title: PW.t('planning'),
              id: "planningFieldSet",
              ref: "../planningFieldSet",
              labelWidth: 150,
              collapsible: false,
              collapsed: false,
              disabled: false,
              defaults: {
                xtype: 'textfield',
                anchor: '95%',
                disabled: !isEditable,
              },
              items: [],
            },
            {
              xtype: "numberfield",
              name: "maxWorkingMinutes",
              fieldLabel: PW.t('maxWorkingMinutes'),
              width: 100,
              allowNegative: false,
              allowDecimals: false,
              allowBlank: true,
              minValue: 1,
              value: task.maxWorkingMinutes,
            },
            {
              xtype: 'checkbox',
              name: 'active',
              fieldLabel: PW.t('activeTask'),
              checked: task.active,
            },
            {
              xtype: 'checkbox',
              name: 'saveHistory',
              fieldLabel: PW.t('saveHistoryExecution'),
              checked: task.saveHistory,
            },
            {
              xtype: 'arrayfield',

              id: 'userNameNotifications',
              fieldLabel: PW.t('userNameNotifications'),
              values: task.userNameNotifications,

              fieldConfig: {
                xtype: 'combo',

                name: 'userNameNotifications',
                hiddenName: 'userNameNotifications',
                allowBlank: false,
                height: 40,

                mode: 'remote',
                pageSize: 25,
                valueField: 'userName',
                displayField: 'displayName',
                store: {
                  xtype: 'jsonstore',
                  fields: [
                    {
                      name: 'userName',
                      type: 'string',
                    },
                    {
                      name: 'displayName',
                      type: 'string',
                    },
                  ],
                  url: Suncode.getAbsolutePath('api/scheduledtasks/notifications/users'),
                  restful: true,
                  root: 'data',
                  totalProperty: 'total',
                  remoteSort: true,
                  sortInfo: {
                    field: "userName",
                    direction: "ASC",
                  },
                  autoLoad: true,
                  listeners: {
                    load: function () {
                      var arrayCombobox = Ext.getCmp('userNameNotifications');
                      var value = arrayCombobox.getValue();
                      if (value && !this.firstLoadExecuted) {
                        arrayCombobox.setValue(value);
                        this.firstLoadExecuted = true;
                      }
                    },
                  },
                },

                forceSelection: true,
                triggerAction: 'all',
                minChars: 3,
                invalidClass: 'avoid-border-non-existent-class',
                listeners: {
                  invalid: function (field) {
                    field.wrap.addClass('x-form-invalid');
                  },
                  valid: function (field) {
                    field.wrap.removeClass('x-form-invalid');
                  },
                },
              },
            },
            {
              xtype: 'arrayfield',
              fieldLabel: PW.t('emailNotifications'),
              values: task.emailNotifications,

              fieldConfig: {
                xtype: 'textfield',

                name: 'emailNotifications',
                allowBlank: false,
                height: 40,
              }
            },
          ],
        },
      ],
    },
    config
  );

  Ext.ux.plusmpm.scheduledtasks.TaskEditStep2.superclass.constructor.call(this, config);
}; // end of Ext.ux.plusmpm.scheduledtasks.TaskEditStep2 constructor

Ext.extend(Ext.ux.plusmpm.scheduledtasks.TaskEditStep2, Ext.Container, {
  onActivate: function () {
    this.ownerCt.setHeight(this.getHeight() + 20);

    this.nameField.focus();

    //sprawia, że będziemy poruszac się tabulatorem tylko po polach wewnątrz okna
    var firstElement = this.nameField;
    var lastElement = this.ownerCt.ownerCt.addButton;

    Ext.EventManager.addListener(firstElement.el, 'keydown', function (e) {
      var key = e.getKey();
      var shiftKey = e.shiftKey;
      if (shiftKey && key == e.TAB) {
        e.stopEvent();
        lastElement.focus(false, 100);
      }
    });

    Ext.EventManager.addListener(lastElement.el, 'keydown', function (e) {
      var key = e.getKey();
      var shiftKey = e.shiftKey;
      if (!shiftKey && key == e.TAB) {
        e.stopEvent();
        firstElement.focus(false, 100);
      }
    });
  },

  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.TaskEditStep2.superclass.initComponent.call(this);
    this.on('activate', this.onActivate, this);
  },

  onPlanModeSelect: function (planMode) {
    var planningFieldSet = Ext.getCmp("planningFieldSet");
    planningFieldSet.removeAll();

    var items;
    if (planMode === "RUN_ONCE") {
      items = this.createRunOnceModeItems();
    }
    else if (planMode === "PERIODIC") {
      items = this.createPeriodicModeItems();
    }
    else {
      items = this.createAtTheEndOfMonthModeItems();
    }

    Ext.each(
      items,
      function (item) {
        planningFieldSet.add(item);
      }
    );

    planningFieldSet.doLayout();
  },

  createRunOnceModeItems: function() {
    var task = this.taskRecord.data;
    var isEditable = this.editable;

    return [
      {
        xtype: "datefield",
        fieldLabel: PW.t("firstRunDate"),
        id: "first_run_field",

        disabled: true,

        format: "Y-m-d H:i:s",
        value: Ext.util.Format.date(task.firstRun, "Y-m-d H:i:s")
      },
      {
        xtype: "datefield",
        fieldLabel: PW.t("nextRunDate"),
        id: "next_run_field",
        name: "nextRun",

        minValue: Ext.util.Format.date(task.firstRun, "Y-m-d H:i:s"),
        value: Ext.util.Format.date(task.nextRun, "Y-m-d H:i:s"),
        format: "Y-m-d H:i:s",
        allowBlank: false,

        triggerClass: "dvnt-icon-time",
        onTriggerClick: function() {
        },
        listeners: {
          afterrender: function(field) {
            if (!isEditable) {
              return;
            }

            field.trigger.on("mouseover", function() {
              var x = field.trigger.getX();
              var y = field.trigger.getY();

              Calendar.setup({
                inputField: "next_run_field",
                ifFormat: "%Y-%m-%d %H:%M:%S",
                button: field.trigger.id,
                singleClick: true,
                showsTime: true,
                timeFormat: "24",
                position: [x + 70, y],
                onUpdate: function(cal) {
                  field.validate();
                }
              });
            });
          }
        }
      }
    ];
  },

  createPeriodicModeItems: function() {
    var task = this.taskRecord.data;

    return [
      ...this.createRunOnceModeItems(),
      {
        xtype: "compositefield",
        fieldLabel: PW.t("period"),
        id: "periodCompositeField",
        items: [
          {
            xtype: "numberfield",
            name: "periodValue",
            disabled: task.runOnce,
            width: 80,
            allowNegative: false,
            allowDecimals: false,
            allowBlank: false,
            minValue: 1,
            value: task.period
          },
          {
            xtype: "combo",
            name: "periodUnit",
            disabled: task.runOnce,
            editable: false,
            forceSelection: true,
            allowBlank: false,
            mode: "local",
            displayField: "text",
            valueField: "unit",
            hiddenName: "periodUnit",
            flex: 1,
            triggerAction: "all",
            store: new Ext.data.ArrayStore({
              fields: ["text", "unit"],
              data: [
                [PW.t("minutes"), "MINUTES"],
                [PW.t("hours"), "HOURS"],
                [PW.t("days"), "DAYS"],
                [PW.t("months"), "MONTHS"],
                [PW.t("years"), "YEARS"]
              ]
            }),
            listeners: {
              afterrender: function() {
                this.setValue(task.periodUnit);
              }
            }
          }
        ]
      },
      {
        xtype: "compositefield",
        id: "workingHours",
        fieldLabel: PW.t("workingHours"),

        items: [{
          xtype: "st_bound_numberfield",
          boundComponentIds: ["startingHour", "endingHour"],

          id: "startingHour",
          name: "startingHour",
          width: 50,
          allowNegative: false,
          allowDecimals: false,
          allowBlank: true,
          minValue: 0,
          maxValue: 23,
          disabled: task.runOnce,
          value: task.startingHour
        }, {
          xtype: "label",
          text: "-",
          margins: "13"
        }, {
          xtype: "st_bound_numberfield",
          boundComponentIds: ["startingHour", "endingHour"],

          id: "endingHour",
          name: "endingHour",
          width: 50,
          allowNegative: false,
          allowDecimals: false,
          allowBlank: true,
          minValue: 0,
          maxValue: 23,
          disabled: task.runOnce,
          value: task.endingHour
        }]
      }
    ];
  },

  createAtTheEndOfMonthModeItems: function() {
    var task = this.taskRecord.data;

    return [
      {
        xtype: "numberfield",
        name: "daysFromMonthEnd",
        fieldLabel: PW.t('daysFromMonthEnd'),
        width: 100,
        allowNegative: false,
        allowDecimals: false,
        allowBlank: false,
        minValue: 0,
        maxValue: 27,
        value: task.daysFromMonthEnd,
      },
      {
        xtype: "compositefield",
        id: "targetHours",
        fieldLabel: PW.t("targetHours"),

        items: [{
          xtype: "st_bound_numberfield",
          boundComponentIds: ["targetHour", "targetMinute"],

          id: "targetHour",
          name: "targetHour",
          width: 50,
          allowNegative: false,
          allowDecimals: false,
          allowBlank: false,
          minValue: 0,
          maxValue: 23,
          value: task.targetHour
        }, {
          xtype: "label",
          text: ":",
          margins: "13"
        }, {
          xtype: "st_bound_numberfield",
          boundComponentIds: ["targetHour", "targetMinute"],

          id: "targetMinute",
          name: "targetMinute",
          width: 50,
          allowNegative: false,
          allowDecimals: false,
          allowBlank: false,
          minValue: 0,
          maxValue: 59,
          value: task.targetMinute
        }]
      }
    ];
  }
}); // end of Ext.extend
Ext.reg('st_edit_wizard_step2', Ext.ux.plusmpm.scheduledtasks.TaskEditStep2);
