(function(Ext) {
    
    /**
     * Api to build data source's parameters form.
     *
     * @class PW.DataSources.DynamicFormApi
     */
    Ext.define('Suncode.datasourcemanager.api.DynamicFormApi', {
        
        alternateClassName: 'PW.DataSources.DynamicFormApi',
            
        constructor: function( form ) {
            this.form = form;
            this.configurationPanel = this.form.up( '#data-source-configuration' );
        },
        
        /**
         * Adds field to the form.
         *
         * @method addField
         * 
         * @param {String / Object} definition Id of field or a definition. If specified id points to the component parameter,
         * then properties such as name, description, optional, type are taken from parameter definition, but you can overwrite them here.
         * @param {String} definition.id Id of field
         * @param {String} definition.name Field name (label)
         * @param {String} definition.description Field description (tooltip)
         * @param {String} definition.type Field type. You should set it only if field doesn't point on component parameter.
         * Available types: string, integer, float, date, datetime, boolean.
         * @param {Boolean} definition.optional Field requirement. If true then field can be empty.
         * @param {Boolean} definition.readOnly Field editability. If true then field value cannot be changed by user.
         * @param {Boolean} definition.hidden Field visibility. If true then field is hidden.
         * @param {Object} definition.listeners Listeners of a field
         * @param {function} definition.listeners.change(field,newValue,oldValue) Fires when the value of a field is changed.<br>
         *          <h4>Parameters:</h4>
         *              <ul class="params-list">
         *                  <li class="param">
         *                      <code class="param-name">field</code>
         *                      <span class="type">Object</span>
         *                      <div class="param-description">Field element. Its for internal use.</div>
         *                  </li>
         *                  <li class="param">
         *                      <code class="param-name">newValue</code>
         *                      <span class="type">Object</span>
         *                      <div class="param-description">New field value</div>
         *                  </li>
         *                  <li class="param">
         *                      <code class="param-name">oldValue</code>
         *                      <span class="type">Object</span>
         *                      <div class="param-description">Old field value</div>
         *                  </li>
         *              </ul>
         * @param {function} definition.listeners.blur Fires when this field loses focus.
         * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
         */
        addField: function( config, position ) {
            this.form.insertParameter(position || this.form.getParametersCount(), this.form._readDefinition(config) );
        },
        
        /**
         * Adds field to the form as combobox field.
         * 
         * @method addCombobox
         * 
         * @param {String / Object} definition Id of field or a definition. If specified id points to the component parameter,
         * then properties such as name, description, optional, type are taken from parameter definition, but you can overwrite them here.
         * @param {String} definition.id Id of field
         * @param {String} definition.valueField Field name that sets the field value
         * @param {String} definition.displayField Field name that value will be displayed
         * @param {Integer} definition.minChars The minimum number of characters entered,
         * after which Combobox starts filtering. Default: 0.
         * @param {Boolean} definition.queryCaching When true, this prevents the combo from re-querying when the current query is the same as the previous query. Defaults true.
         * @param {Object[]} definition.values Only if 'remote' is unspecified. Values to display.
         * @param {String} definition.values.id Value of a field.
         * @param {String} definition.values.display Display value of a field.
         * @param {String} definition.values.description Description of a value.
         * @param {Object} definition.remote Object containing definition of remote data retrieval.
         * @param {String} definition.remote.url Url address
         * @param {Object[]} definition.remote.fields Definition of fields returned from server
     					<ul class="params-list">
         *                  <li class="param">
         *                      <code class="param-name">name</code>
         *                      <span class="type">String</span>
         *                      <div class="param-description">Field name.</div>
         *                  </li>
         *                  <li class="param">
         *                      <code class="param-name">type</code>
         *                      <span class="type">String</span>
         *                      <div class="param-description">Field type.</div>
         *                  </li>
         *              </ul>
         * @param {Boolean} definition.remote.remoteSort Determines if the data should be sorted on the server side (true)
         * or on the browser side (false). By default, false.
         * @param {Integer} definition.remote.pageSize Number of displaying results on page. By default, 25.
         * @param {Object[]} definition.sort Array of objects defining the sorting method.
         * @param {String} definition.sort.property Field after which we want to sort.
         * @param {String} definition.sort.direction Direction after which we want to sort.
         * ASC for ascending direction, DESC for descending direction.
         * @param {Object} definition.listeners Listeners of a field
         * @param {function} definition.listeners.change(field,newValue,oldValue) Fires when the value of a field is changed.<br>
         *          <h4>Parameters:</h4>
         *              <ul class="params-list">
         *                  <li class="param">
         *                      <code class="param-name">field</code>
         *                      <span class="type">Object</span>
         *                      <div class="param-description">Field element. Its for internal use.</div>
         *                  </li>
         *                  <li class="param">
         *                      <code class="param-name">newValue</code>
         *                      <span class="type">Object</span>
         *                      <div class="param-description">New field value</div>
         *                  </li>
         *                  <li class="param">
         *                      <code class="param-name">oldValue</code>
         *                      <span class="type">Object</span>
         *                      <div class="param-description">Old field value</div>
         *                  </li>
         *              </ul>
         * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
         */
        addCombobox: function( config, position ) {
            config = this.form._readDefinition( config );
            
            var combo = {
                xtype: 'combo',
                hideLabel: config.hideLabel,
                forceSelection: config.forceSelection !== false,
                triggerAction: 'all',
                minChars: (config.minChars) ? config.minChars : 0,
                valueField: (config.valueField) ? config.valueField : 'id',
                displayField: (config.displayField) ? config.displayField : 'display',
        		queryCaching: (config.queryCaching === false) ? false : true 
            };

            if (Array.isArray(config.template)) {
              const templateItems = config.template
                .map(
                  item =>
                    `<div>
                        <b>${item.label}:</b>
                        <span>{${item.field}}</span>
                     </div>`
                )
                .join('');

              combo = Ext.apply(combo, {
                tpl: Ext.create(
                  'Ext.XTemplate',
                  `<tpl for=".">
                       <div class="x4-boundlist-item">
                           ${templateItems}
                       </div>
                   </tpl>`
                ),
              });
            }

            Ext.apply(combo, (config.remote) ? this._remoteComboBox(config) : this._localComboBox(config));

            this.form.insertParameter( position || this.form.getParametersCount(), config, combo );
        },
        
        _localComboBox: function(config){
            return {
                queryMode: 'local',
                store: Ext.create( 'Ext.data.Store', {
                    data: config.values,
                    fields: [{
                        name: 'id',
                        type: config.type
                    }, {
                        name: 'display',
                        type: 'string'
                    }, {
                        name: 'description',
                        type: 'string'
                    } ]
                } ),
            }
        },
        
        _remoteComboBox: function(config){
            var remote = config.remote;

            return {
                queryMode: 'remote',
                store: {
                    xtype: 'store',
                    fields: (remote.fields) ? remote.fields : [{
                        name: 'id',
                        type: config.type
                    }, {
                        name: 'display',
                        type: 'string'
                    }, {
                        name: 'description',
                        type: 'string'
                    }],
                    proxy: {
                        type: 'ajax',
                        url: remote.url,
                        extraParams: {
                            query: ''
                        },
                        reader: {
                            type: 'json',
                            root: 'data',
                            totalProperty: 'total',
                        }
                    },
                    pageSize: (remote.pageSize) ? remote.pageSize : 0,
                    remoteSort: (remote.remoteSort) ? remote.remoteSort : false,
                    sorters: (config.sort) ? config.sort : [],
                    autoLoad: true
                },
                pageSize: (remote.pageSize) ? remote.pageSize : 25,
                setValue: function(value, injection) {
                    if(value && injection) {
                        var me = this,
                            store = me.getStore();
                        if( store.findExact(this.valueField, value) < 0 ) {
                            store.load({
                                params: {
                                    query: value,
                                    unique: true
                                },
                                callback: function() {
                                    if (typeof value == 'object'){
                                        Ext.form.ComboBox.prototype.setValue.call(me, value.data.name);
                                    }
                                    else{
                                        Ext.form.ComboBox.prototype.setValue.call(me, value);
                                    }
                                }
                            });
                        }
                        else {
                            Ext.form.ComboBox.prototype.setValue.call(me, value);
                        }
                    }
                    else {
                        Ext.form.ComboBox.prototype.setValue.call(this, value);
                    }
                }
            }
        },
        
        /**
         * Adds field to the form as TextArea field.
         * <b>Warning!</b> Only for String parameter.
         *
         * @method addTextArea
         * 
         * @param {Object} config Configuration of field. (See {{#crossLink "PW.DataSources.DynamicFormApi/addField"}}{{/crossLink}} definition)
         * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
         */
        addTextArea: function( config, position ){
            config = this.form._readDefinition( config );
            
            if( this.form.isComponentParameter( config.id ) && ( config.array || config.type != 'string' ) ) {
                return;
            }
            if ( !Ext.isEmpty( config ) ) {
                this.form.insertParameter( position || this.form.getParametersCount(), config, {
                    xtype: 'textareafield',
                    cls: 'custom-resize-handle',
                    listeners: {
                        afterrender: textarea => {
                            textarea.resizeObserver = new ResizeObserver(entries => {
                                const newHeight = Math.ceil(entries[0].borderBoxSize[0].blockSize)
                                textarea.setHeight(newHeight)
                                textarea.up().updateLayout()
                            })

                            textarea.resizeObserver.observe(textarea.inputEl.dom);
                        }
                    }
                });
            }
        },
        
        /**
         * Adds field to the form as password field.
         * <b>Warning!</b> Only for String parameter.
         * 
         * @method addPassword
         * 
         * @param {Object} config Configuration of field. (See {{#crossLink "PW.DataSources.DynamicFormApi/addField"}}{{/crossLink}} definition)
         * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
         */
        addPassword: function(config, position){
            config = this.form._readDefinition( config );
            
            if( this.form.isComponentParameter( config.id ) && ( config.array || config.type != 'string' ) ) {
                return;
            }
            if ( !Ext.isEmpty( config ) ) {
                this.form.insertParameter( position || this.form.getParametersCount(), config, {
                    xtype: 'textfield',
                    inputType: 'password'
                } );
            }
        },
        
        /**
         * Adds fields to the form as checkbox field.
         * 
         * @method addCheckbox
         * 
         * @param {Object} config Configuration of field. (See {{#crossLink "PW.DataSources.DynamicFormApi/addField"}}{{/crossLink}} definition)
         * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
         */
        addCheckbox: function(config, position) {
            config = this.form._readDefinition( config );
            
            if( this.form.isComponentParameter( config.id ) && ( config.array || config.type != 'boolean' ) ) {
                return;
            }
            
            this.form.insertParameter( position || this.form.getParametersCount(), config, {
                xtype: 'checkbox'
            } );
        },
        
        /**
         * Adds button to the form.
         * 
         * @method addButton
         * 
         * @param {Object} definition Button definition.
         * @param {String} definition.id Button id.
         * @param {String} definition.text Displayed button text.
         * @param {function} definition.handler Function that is triggered by clicking a button.
         * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
         */
        addButton: function(definition, position) {
            Ext.apply(definition, {
                itemId: definition.id,
                flex: 0,
                height: '100%'
            });
            delete definition.id;
            this.form.insert(position || this.form.getParametersCount(), Ext.create('Ext.button.Button', definition));
        },
        
        /**
         * Hides element by id.
         * 
         * @method hide
         * 
         * @param {String} elementId Id of element to hide.
         */
        hide: function(elementId) {
            var element = this.form.query('#' + elementId)[0];
            if(element && element.hide) {
                element.hide();
            }
        },
        
        /**
         * Shows element by id.
         * 
         * @method show
         * 
         * @param {String} elementId Id of element to show.
         */
        show: function(elementId) {
            var element = this.form.query('#' + elementId)[0];
            if(element && element.show) {
                element.show();
            }
        },
        
        /**
         * Disables element by id.
         * 
         * @method disable
         * 
         * @param {String} elementId Id of element to disable.
         */
        disable: function( elementId ) {
            var element = this.form.query('#' + elementId)[0];
            if(element && element.setReadOnly) {
                element.setReadOnly( true );
            }
        },
        
        /**
         * Enables element by id.
         * 
         * @method enable
         * 
         * @param {String} elementId Id of element to enable.
         */
        enable: function( elementId ) {
            var element = this.form.query('#' + elementId)[0];
            if(element && element.setReadOnly) {
                element.setReadOnly( false );
            }
        },
        
        /**
         * Gets element value by id.
         * 
         * @method getValue
         * 
         * @param {String} elementId Id of element to get the value.
         */
        getValue: function(elementId) {
            var element = this.form.query('#' + elementId)[0];
            if (element && element.getValue) {
                if (element.xtype == "multivaluefield") {
                    return element.getValue(elementId);
                } else {
                    return element.getValue();
                }
            }

            //If elementId = elements in the table
            //var table = form.addTable();
            //table.addField( elementId );
            var values = this.form.query('[name=' + elementId + ']');
            values = jQuery.map(values, function(item, index) {
                return item.value;
            });
            return values;
        },

        /**
         * Sets element value by id.
         *
         * @method setValue
         *
         * @param {String} elementId Id of element to set the value.
         * @param value Value to set.
         */
        setValue: function(elementId, value) {
            var element = this.form.query('#' + elementId)[0];
            if (element && element.getValue) {
                if (element.xtype != "multivaluefield") {
                    element.setValue(value);
                }
            }
        },
        
        /**
         * Adds table to the form. The table grouped parameters in one component. To add a field to a table, use the API function.
         * @example 
         * 		var table = form.addTable();
         * 		table('param-1');
         * 		table('param-2');
         * 		table(...);
         * 
         * @method addTable
         * 
         * @param {Object} config Configuration of table.
         * @param {String} config.id Id of table.
         * @param {String} config.name name of table.
         * @param {String} config.description Description of table.
         */
        addTable: function( config ){
            var table = Ext.create('Suncode.datasourcemanager.component.GroupedMultiValueField', config || {});
            return new Suncode.datasourcemanager.api.DynamicFormApi( this.form.add(table).getParametersForm() );
        },
        
        /**
         * Adds row to the form. The row may have other fields arranged horizontally. To add a field to a row, use the API function.
         * @example 
         * 		var row = form.addRow();
         * 		row.addField('param-1');
         * 		row.addField('param-2');
         * 		row.addCombobox(...);
         * 
         * @method addRow
         * 
         * @param {Object} config Configuration of row.
         * @param {String} config.id Id of row.
         * @param {Integer} config.fieldSpace Space between fields.
         */
        addRow: function( customConfig ) {
            var config = Ext.applyIf({}, {
                id: undefined,
                fieldsSpace: 5
            });
            config = Ext.apply( config, customConfig );
            var row = Ext.create('Suncode.datasourcemanager.component.ParametersForm', {
                id: config.id,
                layout: 'hbox',
                margin: '0 0 5px 0',
                isRow: true,
                fieldsSpace: config.fieldsSpace,
                defaults: {
                    flex: 1,
                    labelWidth: 200,
                    labelPad: 10,
                    allowBlank: false
                }
            });
            
            return new Suncode.datasourcemanager.api.DynamicFormApi( this.form.add(row) );
        },
        
        /**
         * Puts mask in the view.
         * 
         * @method mask
         * 
         * @param {Boolean} onlyForm If true then puts mask in form otherwise in whole panel.
         */
        mask: function( onlyForm ) {
            var panel = onlyForm ? this.form : this.configurationPanel;
            panel.mask();
        },
        
        /**
         * Removes mask from the view.
         * 
         * @method unmask
         * 
         * @param {Boolean} onlyForm If true then removes mask from form otherwise from whole panel.
         */
        unmask: function( onlyForm ) {
            var panel = onlyForm ? this.form : this.configurationPanel;
            panel.unmask();
        }
        
    });
    
})(Ext4);