(function(Ext, messages) {

	Ext.define('Suncode.pluginmanager.Manager', {
	    extend: 'Ext.app.Controller',

	    stores: ['Suncode.pluginmanager.store.Plugins'],
	    models: ['Plugin'],
	    views: ['Viewport'],

	    startUrlTemplate: Suncode.getAbsolutePath('/api/plugins/{0}/start'),
	    stopUrlTemplate: Suncode.getAbsolutePath('/api/plugins/{0}/stop'),
	    deleteUrlTemplate: Suncode.getAbsolutePath('/api/plugins/{0}'),
	    updateUrlTemplate: Suncode.getAbsolutePath('/api/plugins/{0}/update'),

	    refs: [{
	        ref: 'installForm',
	        selector: 'pluginmanager-installer'
	    }, {
	        ref: 'pluginList',
	        selector: 'pluginmanager-pluginlist'
	    }, {
	        ref: 'pluginDetails',
	        selector: 'pluginmanager-details'
	    }, {
	        ref: 'pluginStartupError',
	        selector: 'pluginmanager-systemstartuperror'
		}],

		onLaunch: function(){
			this.getPluginsStore().load();
		},

		getPluginsStore: function(){
		    return this.getStore('Suncode.pluginmanager.store.Plugins');
		},

		init: function(){
	        var me = this;

	        me.eventbus.unlisten(me.id);
	        me.control({
	        	'pluginmanager-installer': {
					installed: me.initPluginOrLicense,
					forceUpdate: me.updatePlugin
            	},
            	'pluginmanager-updater': {
            		update: me.updatePlugin
            	},
            	'pluginmanager-details': {
            		start: me.startPlugin,
            		stop: me.stopPlugin,
            		uninstall: me.uninstallPlugin
            	},
            	'pluginmanager-pluginlist': {
            		selectionchange: me.pluginSelected,
            		beforeselect: me.beforePluginSelected,
            		beforedeselect: me.beforePluginSelected
            	}
	        });
	    },

	    initPluginOrLicense: function(result){
	        var me = this,
				store = me.getPluginsStore(),
				list = me.getPluginList();

	        if( !result.entity.licenseInstallation ) {
				var plugin = me.getPluginModel().create(result.entity);

				// dodajemy do store'a wtyczkę i zaznaczamy ją
				store.add(plugin);

				list.getView().refresh();
				list.getSelectionModel().select(plugin);
			}
	        else {
				store.reload();
	        	PW.ui.Message.success(messages('plugins.view.licenseInstalled' ));
			}
	    },

	    beforeAction: function(){
	        var me = this;

	        if(me.currentAction){
	        	return false;
	        }
	        me.currentAction = true;
	        return true;

	    },

	    afterAction: function(){
	        var me = this,
	            store = me.getPluginsStore(),
	            details = me.getPluginDetails();

	        try {
                // odświezenie listy wtyczek, ponieważ ich stany mogły się zmienić
	        	store.load({
	        		callback: function(){
			        	details.unmask();
	        	    }
	        	});
    		}
	        finally{
	        	delete me.currentAction;
	        }
	    },

	    updatePlugin: function(plugin, form, filename, pluginKey, force) {
            var me = this,
                store = me.getPluginsStore(),
                details = me.getPluginDetails(),
                url;

            if((!plugin&&!pluginKey) || !form || me.beforeAction() == false){
                return;
            }

            details.mask(messages('view.details.update.progress', filename ));

			pluginKey = pluginKey ? pluginKey : plugin.get('key');
            url = Ext.String.format(me.updateUrlTemplate, pluginKey);
            form.submit({
				url: url,
				params: {
					force: !!force
				},
				timeout: 120000,
                success: function(){
                    me.afterAction();
                },
                failure: function(f, action){
                    me.afterAction();

					var result = action.result;
					if(result["missing"]) {
						Suncode.pluginmanager.view.Requirements.update(result, function() {
							me.updatePlugin(plugin, form, filename, pluginKey, true);
						});
					}
					else {
						var details = (result && result.message) ? result.message : messages('error.unexpected');
						me.showError(messages('error'), messages('view.details.update.error.unexpected', details) );
					}
                }
            });

	    },

	    startPlugin: function(plugin, force){
	        var me = this,
	            msg = messages('view.details.start.progress');

            me.performSimpleAction({
				plugin: plugin,
				force: force,
            	msg: msg,
            	url: Ext.String.format(me.startUrlTemplate, plugin.get('key'))
            }, function(result, response){
				
				if(response.status == 412) {
                    Suncode.pluginmanager.view.Requirements.start(result, function() {
						me.startPlugin(plugin, true);
					});
				}
                else {
                    var details = (result && result.message)
                        ? result.message
                        : messages('error.unexpected');
                    me.showError(messages('error'), messages('view.details.start.error.unexpected', details) );
                }
            });
	    },

		stopPlugin: function(plugin, force){
		    var me = this,
		    msg = messages('view.details.stop.progress');

            me.performSimpleAction({
				plugin: plugin,
				force: force,
            	msg: msg,
            	url: Ext.String.format(me.stopUrlTemplate, plugin.get('key'))
            }, function(result, response){
				if(response.status == 412) {
                    Suncode.pluginmanager.view.Requirements.stop(result, function() {
						me.stopPlugin(plugin, true);
					});
				}
			});
		},

		uninstallPlugin: function(plugin, force){
		    var me = this,
		            msg = messages('view.details.delete.progress');

            me.performSimpleAction({
				plugin: plugin,
				force: force,
            	msg: msg,
            	url: Ext.String.format(me.deleteUrlTemplate, plugin.get('key')),
            	method: 'DELETE'
            }, function(result, response) {
				if(response.status == 412) {
                    Suncode.pluginmanager.view.Requirements.uninstall(result, function() {
						me.uninstallPlugin(plugin, true);
					});
				}
			});
		},

		performSimpleAction: function(config, failure){
		    var me = this,
				plugin = config.plugin,
				force = config.force,
		        url = config.url,
		        msg = config.msg,
		        details = me.getPluginDetails();

		    if(!plugin || !url || me.beforeAction() == false){
		    	return;
		    }

	        details.mask(msg);
	        Ext.Ajax.request({
				url: url + (force ? "?force=true" : ""),
				timeout: 120000,
	            method: config.method || 'POST',
	            callback: function(opts, success, response){
	        	    me.afterAction();

	        	    var result = Ext.decode(response.responseText, true);
	        	    if(Ext.isFunction(failure)){
	        	        if(!success || (result && !result.success)){
	        	            failure.call(me, result, response);
	        	        }
	        	    }
	            }
	        });

		},

		showError: function(title, message){
		    Ext.Msg.show({
		       title: title,
		       msg: message,
		       buttons: Ext.MessageBox.OK,
		       icon: Ext.MessageBox.ERROR
		    });
		},

		beforePluginSelected: function(grid, plugin){
	        // nie pozwól na wybranie wtyczki, jeżeli inna akcja jest właśnie wykonywana
		    return this.currentAction? false : true;
		},

		pluginSelected: function(grid, selection){
            var details = this.getPluginDetails();
            var pluginStartupError = this.getPluginStartupError();

            // jeżeli wybralismy wtyczkę, zbinduj formę i pokaż ją
            if (Ext.isEmpty(selection)) {
                details.setVisible(false);
            }
            else {
                var selected = selection[0]
                details.bind(selected);

                if (selected.hasErrorWhileSystemStartup()) {
                    pluginStartupError.setMessage(selected.get('systemStartupError'));
                    pluginStartupError.setVisible(true);
                }
                else {
                    pluginStartupError.setVisible(false);
                }
            }
		}
	});

}(Ext4, Suncode.pluginmanager.i18n.MessageSource.getMessage));
