(function (Ext) {
  var messages = Suncode.context('dashboard').messages;

  Ext.define('Suncode.dashboard.administration.model.User', {
    extend: 'Ext.data.Model',
    fields: [
      {
        name: 'id',
        mapping: 'objectId',
        type: 'int',
      },
      {
        name: 'username',
        mapping: 'userName',
        type: 'string',
      },
      {
        name: 'fullName',
        type: 'string',
      },
      {
        name: 'name',
        convert: function (value, record) {
          return record.data.fullName + ' (' + record.data.username + ')';
        },
      },
    ],
  });

  Ext.define('Suncode.dashboard.administration.model.Group', {
    extend: 'Ext.data.Model',
    fields: [
      {
        name: 'id',
        mapping: 'objectId',
        type: 'int',
      },
      {
        name: 'name',
        type: 'string',
      },
    ],
  });

  var shareTypeStore = Ext.create('Ext.data.Store', {
    fields: ['type', 'display'],
    data: [
      { type: 'ALL', display: messages('share.all') },
      { type: 'USER', display: messages('share.user') },
      { type: 'GROUP', display: messages('share.group') },
    ],
  });

  Ext.define('Suncode.dashboard.administration.view.EditDashboardForm', {
    extend: 'Suncode.dashboard.administration.view.StatusForm',
    alias: 'widget.edit-dashboard-form',
    hidden: true,
    bodyStyle: {
      border: '0',
    },
    initComponent: function () {
      Ext.apply(this, {
        style: {
          margin: '5px',
        },
        items: [
          {
            xtype: 'textfield',
            name: 'name',
            fieldLabel: messages('dashboard.name'),
            allowBlank: false,
          },
          {
            xtype: 'textarea',
            name: 'description',
            fieldLabel: messages('dashboard.description'),
          },
          {
            xtype: 'fieldcontainer',
            fieldLabel: messages('dashboard.shared'),
            items: [
              {
                xtype: 'grid',
                hideHeaders: true,
                id: 'sharesContainer',
                itemId: 'sharesGrid',
                maxHeight: 150,
                viewConfig: {
                  emptyText: messages('form.share.msg'),
                  deferEmptyText: false,
                },
                listeners: {
                  beforeselect: function () {
                    return false;
                  },
                },
                columns: [
                  {
                    dataIndex: 'shareType',
                    itemId: 'sharesShareType',
                    width: 25,
                    renderer: function (value, metadata, record) {
                      var css, qtip;
                      if (value === 'ALL') {
                        css = 'dvnt-icon-globe';
                        qtip = messages('all');
                      } else if (value === 'USER') {
                        css = 'dvnt-icon-user';
                        qtip = messages('user');
                      } else if (value === 'GROUP') {
                        css = 'dvnt-icon-users';
                        qtip = messages('group');
                      }
                      return '<span data-qtip="' + qtip + '" style="width: 16px; height:16px; display: block" class="' + css + '"></span>';
                    },
                  },
                  {
                    dataIndex: 'resourceName',
                    itemId: 'sharesResourceName',
                    flex: 1,
                    renderer: function (value, metadata, record) {
                      if (record.get('shareType') === 'ALL') {
                        return messages('all');
                      }
                      return value;
                    },
                  },
                  {
                    xtype: 'actioncolumn',
                    itemId: 'sharesActionColumn',
                    width: 25,
                    iconCls: 'dvnt-icon-x-symbol',
                    tooltip: messages('delete'),
                    handler: this.removeDashboardShare,
                  },
                ],
              },
            ],
          },
          {
            xtype: 'fieldcontainer',
            fieldLabel: messages('dashboard.shared'),
            layout: {
              type: 'hbox',
              align: 'middle',
            },
            defaults: {
              margin: {
                right: 5,
              },
              msgTarget: 'side',
            },
            items: [
              {
                xtype: 'combo',
                flex: 2,
                isFormField: false,
                store: shareTypeStore,
                itemId: 'shareType',
                value: 'ALL',
                displayField: 'display',
                valueField: 'type',
                queryMode: 'local',
                editable: false,
                listeners: {
                  scope: this,
                  change: this.handleShareTypeChange,
                },
              },
              {
                xtype: 'combo',
                flex: 3,
                itemId: 'shareResource',
                isFormField: false,
                disabled: true,
                valueField: 'id',
                displayField: 'name',
                queryMode: 'remote',
                forceSelection: true,
                allowBlank: false,
                minChars: 1,
                pageSize: 20,
              },
              {
                xtype: 'button',
                iconCls: 'dvnt-icon-plus-circle',
                itemId: 'addShareButton',
                height: 36,
                margin: 0,
                handler: this.addDashboardShare,
                scope: this,
              },
            ],
          },
            {
                xtype: 'textfield',
                fieldLabel: messages('TranslatedFieldType_ABBREVIATION'),
                maxLength: 3,
                name: 'abbreviation',
                fieldStyle: 'text-transform:uppercase;',
            },
            {
                xtype: 'combobox',
                store: Ext.create('Suncode.dashboard.administration.store.Colors', {autoLoad: true}),
                fieldLabel: messages('Kolor'),
                name: 'color',
                queryMode: 'remote',
                displayField: 'name',
                valueField: 'value',
                itemId: 'color',
                editable: false,
            },
        ],
        dockedItems: [
          {
            xtype: 'toolbar',
            dock: 'bottom',
            ui: 'footer',
            defaults: {
              minWidth: 75,
            },
            items: [
              {
                xtype: 'tbtext',
                style: 'text-align: left;',
                flex: 1,
              },
              {
                xtype: 'button',
                cls: 'toolbar-button-overrides-secondary',
                width: '150px',
                text: messages('form.cancel'),
                scope: this,
                handler: function () {
                  // hide this form
                  this.hide();
                },
              },
              {
                xtype: 'button',
                cls: 'toolbar-button-overrides-primary',
                width: '180px',
                text: messages('form.edit.save'),
                scope: this,
                handler: this.saveChanges,
              },
            ],
          },
        ],
      });

      this.callParent();
      this.initRefs();
      this.initStores();
    },

    /**
     * Cache components so we don't have to query them all the time
     */
    initRefs: function () {
      this.refs = {
        'share.type': this.down('#shareType'),
        'share.resource': this.down('#shareResource'),
        'share.grid': this.down('#sharesGrid'),
      };
    },

    /**
     * Init users and groups stores. Save them in refs
     */
    initStores: function () {
      this.refs['store.user'] = new Ext.data.Store({
        model: 'Suncode.dashboard.administration.model.User',
        proxy: {
          type: 'ajax',
          url: Suncode.getAbsolutePath('/api/users/query'),
          reader: {
            type: 'json',
            root: 'data',
          },
        },
      });

      this.refs['store.group'] = new Ext.data.Store({
        model: 'Suncode.dashboard.administration.model.Group',
        proxy: {
          type: 'ajax',
          url: Suncode.getAbsolutePath('/api/groups/query'),
          reader: {
            type: 'json',
            root: 'data',
          },
        },
      });
    },

    /**
     * Collects changes and sends them to server
     */
    saveChanges: function () {
      var me = this,
        form = me.getForm(),
        store = me.refs['share.grid'].getStore(),
        submitData = {},
        type,
        resource;

      if (!form.isValid()) {
        return;
      }

      // collect data "by hand"
      Ext.apply(submitData, {
        name: form.findField('name').getValue(),
        description: form.findField('description').getValue(),
          abbreviation: form.findField('abbreviation').getValue(),
          color: form.findField('color').getValue(),
      });

      store.each(function (share, index) {
        type = share.get('shareType');

        submitData['shares[' + index + '].type'] = type;
        if (type != 'ALL') {
          submitData['shares[' + index + '].resource'] = share.get('resource');
        }
      });

      // submit form with custom params
      me.body.mask();
      me.setStatusLoading(messages('form.save.loading.msg'));
      var mainPanel = me.up().up();
      form.submit({
        url: Suncode.context('dashboard').base + '/api/dashboards/' + me.dashboard.get('id'),
        params: submitData,
        success: function () {
          me.body.unmask();
          me.setStatusSuccess(messages('form.save.success.msg'), true);
          Ext.StoreManager.lookup('MyDashboards').load({
            params: {
              user: mainPanel.chosenUsername,
            },
          });
        },
        failure: function () {
          me.body.unmask();
          me.setStatusFailure(messages('form.save.failure.msg'), true);
        },
      });
    },

    /**
     * Share type combo changed. Setup resource combo
     */
    handleShareTypeChange: function (combo, type) {
      var me = this,
        resource = me.refs['share.resource'];

      // reset combo
      resource.emptyText = ' ';
      delete resource.lastQuery;

      // disable if ALL
      resource.setDisabled(type == 'ALL');

      if (type == 'USER') {
        resource.emptyText = messages('form.share.user.empty');
        resource.bindStore(me.refs['store.user']);
      } else if (type == 'GROUP') {
        resource.emptyText = messages('form.share.group.empty');
        resource.bindStore(me.refs['store.group']);
      }
      // reset it now, to apply empty text
      resource.reset();
    },

    /**
     * Binds this dashboard to edit form
     */
    bind: function (dashboard) {
        var me = this,
            type = me.refs['share.type'],
            resource = me.refs['share.resource'],
            grid = me.refs['share.grid'];

        me.dashboard = dashboard;

        // set title and bind form standard fields
        me.clearStatus();
        me.setTitle(messages('form.edit.title', Ext.util.Format.htmlEncode(dashboard.get('translatedName'))));
        me.getForm().setValues(dashboard.data);

        // reset custom form fields
        type.reset();
        resource.reset();

        // shares
        var store = Ext.create('Ext.data.Store', {
            autoDestroy: true,
            model: 'Suncode.dashboard.administration.model.DashboardShare',
        });

        store.add(dashboard.shares().getRange());
        grid.bindStore(store);
    },

    /**
     * Finds share entry with following type and resource (type != ALL)
     */
    findShare: function (store, type, resource) {
      if (type == 'ALL') {
        return store.find('shareType', type);
      }
      return store.findBy(function (record) {
        return record.get('resource') == resource && record.get('shareType') == type;
      });
    },

    hightlightRow: function (grid, index) {},

    addDashboardShare: function () {
      var me = this,
        resourceCombo = me.refs['share.resource'],
        typeCombo = me.refs['share.type'],
        grid = me.refs['share.grid'],
        resource = resourceCombo.getValue(),
        type = typeCombo.getValue(),
        store = grid.getStore();

      if (!resourceCombo.isValid() || me.findShare(store, type, resource) > -1) {
        return;
      }

      if (type == 'ALL' && store.getCount() > 0) {
        Ext.Msg.show({
          title: messages('share.override.title'),
          msg: messages('share.override.msg'),
          buttons: Ext.Msg.OKCANCEL,
          icon: Ext.Msg.INFO,
          fn: function (button) {
            if (button === 'ok') {
              me.doAddDashboardShare(store, 'ALL');
            }
          },
        });
      } else {
        me.doAddDashboardShare(store, type, resourceCombo);
      }
    },

    doAddDashboardShare: function (store, type, resourceCombo) {
      var share = {
        shareType: type,
      };

      // suspend layouts
      Ext.suspendLayouts();
      if (type != 'ALL') {
        var record = resourceCombo.findRecordByValue(resourceCombo.getValue());
        Ext.apply(share, {
          resource: record.get('id'),
          resourceName: record.get('name'),
        });

        var allShareIndex = store.find('shareType', 'ALL');
        if (allShareIndex >= 0) {
          store.removeAt(allShareIndex);
        }

        resourceCombo.reset();
      } else {
        store.removeAll();
      }

      // add share to store
      store.add(share);

      // resume layouts flushing changes
      Ext.resumeLayouts(true);
    },

    removeDashboardShare: function (grid, rowIndex) {
      grid.getStore().removeAt(rowIndex);
    },

    onDestroy: function () {
      this.callParent();

      this.refs['store.user'].destroyStore();
      this.refs['store.group'].destroyStore();

      delete this.refs;
    },
  });
})(Ext4);
