(function(Ext) {
/**
 * Add basic filtering to Ext.tree.Panel. Add as a mixin:
 *  mixins: {
 *      treeFilter: 'MyApp.lib.TreeFilter'
 *  }
 */
Ext.define('Suncode.datasourcemanager.mixins.TreeFilter', {

    filter: function (query, category) {
        const clonedRoot = Ext.clone(this.store.originalRoot)
        this.assignParents(clonedRoot);

        this.filterByCategory(category, clonedRoot);
        this.filterByQuery(query, clonedRoot);

        this.store.setRootNode(clonedRoot)
    },

    assignParents(node) {
        if (!node.children) {
            return
        }

        node.children.forEach(childNode => {
            childNode.parentNode = node
            this.assignParents(childNode)
        })
    },

    filterByCategory(category, rootNode) {
        if (!category) {
            return;
        }


        const customCategoryNode = rootNode.children.find(node => node.id.split('#')[1] === "custom")
        if (!customCategoryNode) {
            return;
        }

        const categoryNode = customCategoryNode.children.find(node => node.id.split('#')[1] == category)
        if (!categoryNode) {
            return;
        }

        categoryNode.expanded = true;
        rootNode.children = [categoryNode]
    },

    filterByQuery(query, rootNode) {
        if (!query) {
            return
        }

        const idSet = new Set();
        this.cascadeDown(rootNode, node => {
            if (this.nodeMatchesQuery(node, query)) {
                idSet.add(node.id)
                this.cascadeDown(node, childNode => {
                    idSet.add(childNode.id)
                })
                this.cascadeUp(node, (parentNode) => {
                    idSet.add(parentNode.id)
                    parentNode.expanded = true;
                })
            }
        })

        this.filterById(rootNode, idSet);
    },

    nodeMatchesQuery(node, query) {
        if (node.name && this.matchesQuery(node.name, query)) {
            return true;
        }

        if (node.description && this.matchesQuery(node.description, query)) {
            return true;
        }

        if (node.datasourceId && this.matchesQuery(node.datasourceId, query)) {
            return true;
        }

        if (node.parameters && node.parameters.some(parameter => this.matchesQuery(parameter, query))) {
            return true;
        }

        return false;
    },

    matchesQuery(prop, query) {
        return prop.toLowerCase().indexOf(query.toLowerCase()) > -1
    },

    filterById(node, idSet) {
        if (!node.children) {
            return
        }

        node.children = node.children.filter(childNode => {
            return idSet.has(childNode.id)
        })

        node.children.forEach(childNode => {
            this.filterById(childNode, idSet)
        })
    },

    cascadeDown(node, callback) {
        if (!node.children) {
            return
        }

        node.children.forEach(childNode => {
            callback(childNode)
            this.cascadeDown(childNode, callback)
        })
    },

    cascadeUp(node, callback) {
        const parentNode = node.parentNode
        if (!parentNode) {
            return
        }

        callback(parentNode)

        this.cascadeUp(parentNode, callback)
    },
});

})(Ext4);