Ext.namespace('Ext.ux.plusmpm.scheduledtasks');

/**
 * Constructs dynamic form API object for scheduled task parameters.
 *
 * @method Ext.ux.plusmpm.scheduledtasks.DynamicFormApi
 *
 * @param {Ext.form.FormPanel} config.form Form on which building parameters occurs.
 * @param {Ext.Window} config.window Window to which form belongs.
 */
Ext.ux.plusmpm.scheduledtasks.DynamicFormApi = function (config) {
  Ext.ux.plusmpm.scheduledtasks.DynamicFormApi.superclass.constructor.call(this, config);
};

Ext.extend(Ext.ux.plusmpm.scheduledtasks.DynamicFormApi, Ext.Component, {
  initComponent: function () {
    Ext.ux.plusmpm.scheduledtasks.DynamicFormApi.superclass.initComponent.call(this);
  },

  /**
   * Adds field to the form.
   *
   * @method addField
   *
   * @param config
   * @param {String / Object} config Id of field or a definition. If specified id points to the component parameter,
   * then properties such as name, description, optional, type are taken from parameter definition, but you can overwrite them here.
   * @param {String} config.id Id of field
   * @param {String} config.name Field name (label)
   * @param {String} config.description Field description (tooltip)
   * @param {Boolean} config.optional Field requirement. If true then field can be empty.
   * @param {Boolean} config.readOnly Field editability. If true then field value cannot be changed by user.
   * @param {Boolean} config.hidden Field visibility. If true then field is hidden.
   * @param {Object} config.listeners Listeners of a field
   * @param {function} config.listeners.change(field,newValue,oldValue) Fires when the value of a field is changed.<br>
   *          <h4>Parameters:</h4>
   *              <ul class="params-list">
   *                  <li class="param">
   *                      <code class="param-name">field</code>
   *                      <span class="type">Object</span>
   *                      <div class="param-description">Field element. Its for internal use.</div>
   *                  </li>
   *                  <li class="param">
   *                      <code class="param-name">newValue</code>
   *                      <span class="type">Object</span>
   *                      <div class="param-description">New field value</div>
   *                  </li>
   *                  <li class="param">
   *                      <code class="param-name">oldValue</code>
   *                      <span class="type">Object</span>
   *                      <div class="param-description">Old field value</div>
   *                  </li>
   *              </ul>
   * @param {function} config.listeners.blur Fires when this field loses focus.
   * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
   */
  addField: function (config, position) {
    var paramDefinitionWithConfig = this.parametersBuilder._readDefinition(config);
    var paramCmpConfig = this.parametersBuilder.createParameter(paramDefinitionWithConfig);

    this.form.insertParameterCmp(
      position !== undefined ? position : this.form.getParametersCount(),
      paramCmpConfig,
      paramDefinitionWithConfig
    );
  },

  /**
   * Adds field to the form as combobox field.
   *
   * @method addCombobox
   *
   * @param {String / Object} config Id of field or a definition. If specified id points to the component parameter,
   * then properties such as name, description, optional, type are taken from parameter definition, but you can overwrite them here.
   * @param {String} config.id Id of field
   * @param {String} config.valueField Field name that sets the field value
   * @param {String} config.displayField Field name that value will be displayed
   * @param {Integer} config.minChars The minimum number of characters entered,
   * after which Combobox starts filtering. Default: 0.
   * @param {Object[]} config.values Only if 'remote' is unspecified. Values to display.
   * Uses valueField and displayField from config to display data.
   * Record can be represented:
   * <ul>
   *     <li>by object with property name valueField and displayField from config</li>
   *     <li>2-element array [valueField, displayField]</li>
   * </ul>
   * @param {Object} config.remote Object containing definition of remote data retrieval.
   * Uses valueField and displayField from config to display data.
   * @param {String} config.remote.url Url address
   * @param {Object} config.remote.params Object with parameters to append to request
   * @param {Boolean} config.remote.remoteSort Determines if the data should be sorted on the server side (true)
   * or on the browser side (false). By default, false.
   * @param {Integer} config.remote.pageSize Number of displaying results on page. By default, 25.
   * @param {Object[]} config.sort Array of objects defining the sorting method.
   * @param {String} config.sort.field Field after which we want to sort.
   * @param {String} config.sort.direction Direction after which we want to sort.
   * ASC for ascending direction, DESC for descending direction.
   * @param {Object} config.listeners Listeners of a field
   * @param {function} config.listeners.change(field,newValue,oldValue) Fires when the value of a field is changed.<br>
   *          <h4>Parameters:</h4>
   *              <ul class="params-list">
   *                  <li class="param">
   *                      <code class="param-name">field</code>
   *                      <span class="type">Object</span>
   *                      <div class="param-description">Field element. Its for internal use.</div>
   *                  </li>
   *                  <li class="param">
   *                      <code class="param-name">newValue</code>
   *                      <span class="type">Object</span>
   *                      <div class="param-description">New field value</div>
   *                  </li>
   *                  <li class="param">
   *                      <code class="param-name">oldValue</code>
   *                      <span class="type">Object</span>
   *                      <div class="param-description">Old field value</div>
   *                  </li>
   *              </ul>
   * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
   */
  addCombobox: function (config, position) {
    var paramDefinitionWithConfig = this.parametersBuilder._readDefinition(config);

    Ext.apply(paramDefinitionWithConfig, {
      valueField: paramDefinitionWithConfig.valueField ? paramDefinitionWithConfig.valueField : 'id',
      displayField: paramDefinitionWithConfig.displayField ? paramDefinitionWithConfig.displayField : 'display',
    });

    var combo = {
      xtype: 'combo',
      forceSelection: true,
      triggerAction: 'all',
      minChars: paramDefinitionWithConfig.minChars ? paramDefinitionWithConfig.minChars : 0,
      valueField: paramDefinitionWithConfig.valueField,
      displayField: paramDefinitionWithConfig.displayField,
    };

    Ext.apply(
      combo,
      paramDefinitionWithConfig.remote ? this._remoteComboBox(paramDefinitionWithConfig) : this._localComboBox(paramDefinitionWithConfig)
    );

    var paramCmpConfig = this.parametersBuilder.createParameter(paramDefinitionWithConfig, combo);

    this.form.insertParameterCmp(
      position !== undefined ? position : this.form.getParametersCount(),
      paramCmpConfig,
      paramDefinitionWithConfig
    );
  },

  _localComboBox: function (paramDefinitionWithConfig) {
    return {
      mode: 'local',
      store: new Ext.data.ArrayStore({
        data: paramDefinitionWithConfig.values.map(function (value) {
          if (Ext.isArray(value)) {
            return value;
          }
          return [value[paramDefinitionWithConfig.valueField], value[paramDefinitionWithConfig.displayField]];
        }),
        fields: [
          {
            name: paramDefinitionWithConfig.valueField,
            type: 'string',
          },
          {
            name: paramDefinitionWithConfig.displayField,
            type: 'string',
          },
        ],
        sortInfo: paramDefinitionWithConfig.sort,
      }),
      invalidClass: 'avoid-border-non-existent-class',
      listeners: {
        invalid: function (field) {
          field.wrap.addClass('x-form-invalid');
        },
        valid: function (field) {
          field.wrap.removeClass('x-form-invalid');
        },
      },
    };
  },

  _remoteComboBox: function (paramDefinitionWithConfig) {
    var remote = paramDefinitionWithConfig.remote;

    var me = this;
    return {
      mode: 'remote',
      store: {
        xtype: 'jsonstore',
        fields: [
          {
            name: paramDefinitionWithConfig.valueField,
            type: 'string',
          },
          {
            name: paramDefinitionWithConfig.displayField,
            type: 'string',
          },
        ],
        url: remote.url,
        baseParams: remote.params || {},
        restful: true,
        root: 'data',
        totalProperty: 'total',
        remoteSort: remote.remoteSort ? remote.remoteSort : false,
        sortInfo: paramDefinitionWithConfig.sort,
        autoLoad: true,
        listeners: {
          load: function () {
            var combobox = me._find(paramDefinitionWithConfig.id);
            var value = combobox.getValue();
            if (value && !this.firstLoadExecuted) {
              combobox.setValue(value);
              this.firstLoadExecuted = true;
            }
          },
        },
      },
      pageSize: remote.pageSize ? remote.pageSize : 25,
      invalidClass: 'avoid-border-non-existent-class',
      listeners: {
        invalid: function (field) {
          field.wrap.addClass('x-form-invalid');
        },
        valid: function (field) {
          field.wrap.removeClass('x-form-invalid');
        },
      },
    };
  },

  /**
   * Changes combobox remote url.
   *
   * @param {String} elementId Id of element to set configuration.
   * @param {String} url url.
   *
   */
  setComboboxRemoteUrl: function (elementId, url) {
    var combobox = Ext.getCmp(elementId);

    if (combobox.setComboboxRemoteUrl) {
      combobox.setComboboxRemoteUrl(url);
      return;
    }

    if (combobox.mode != 'remote') {
      return;
    }

    var jsonStore = combobox.getStore();
    jsonStore.removeAll();
    jsonStore.url = url;
    jsonStore.proxy.setUrl(url, true);
    jsonStore.load();

    combobox.setValue('');
  },

  /**
   * Changes combobox remote parameters configuration.
   *
   * @param {String} elementId Id of element to set configuration.
   * @param {Object} params Object with parameters to append to request.
   *
   */
  setComboboxRemoteParams: function (elementId, params) {
    var combobox = Ext.getCmp(elementId);

    if (combobox.setComboboxRemoteParams) {
      combobox.setComboboxRemoteParams(params);
      return;
    }

    if (combobox.mode != 'remote') {
      return;
    }

    var jsonStore = combobox.getStore();
    jsonStore.removeAll();
    jsonStore.baseParams = params || {};
    jsonStore.load();

    combobox.setValue('');
  },

  /**
   * Adds field to the form as TextArea field.
   * <b>Warning!</b> Only for String parameter.
   *
   * @method addTextArea
   *
   * @param {Object} config Configuration of field. (See {{#crossLink "Ext.ux.plusmpm.scheduledtasks.DynamicFormApi/addField"}}{{/crossLink}} definition)
   * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
   */
  addTextArea: function (config, position) {
    var paramDefinitionWithConfig = this.parametersBuilder._readDefinition(config);
    var paramCmpConfig = this.parametersBuilder.createParameter(paramDefinitionWithConfig, {
      xtype: 'textarea',
      height: 60,
    });

    if (this.parametersBuilder.isComponentParameter(paramDefinitionWithConfig.id) && paramDefinitionWithConfig.type != 'string') {
      return;
    }
    if (!Ext.isEmpty(config)) {
      this.form.insertParameterCmp(
        position !== undefined ? position : this.form.getParametersCount(),
        paramCmpConfig,
        paramDefinitionWithConfig
      );
    }
  },

  /**
   * Adds field to the form as password field.
   * <b>Warning!</b> Only for String parameter.
   *
   * @method addPassword
   *
   * @param {Object} config Configuration of field. (See {{#crossLink "Ext.ux.plusmpm.scheduledtasks.DynamicFormApi/addField"}}{{/crossLink}} definition)
   * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
   */
  addPassword: function (config, position) {
    var paramDefinitionWithConfig = this.parametersBuilder._readDefinition(config);
    var paramCmpConfig = this.parametersBuilder.createParameter(paramDefinitionWithConfig, {
      xtype: 'textfield',
      inputType: 'password',
    });

    if (this.parametersBuilder.isComponentParameter(paramDefinitionWithConfig.id) && paramDefinitionWithConfig.type != 'string') {
      return;
    }
    if (!Ext.isEmpty(config)) {
      this.form.insertParameterCmp(
        position !== undefined ? position : this.form.getParametersCount(),
        paramCmpConfig,
        paramDefinitionWithConfig
      );
    }
  },

  /**
   * Adds fields to the form as checkbox field.
   *
   * @method addCheckbox
   *
   * @param {Object} config Configuration of field. (See {{#crossLink "Ext.ux.plusmpm.scheduledtasks.DynamicFormApi/addField"}}{{/crossLink}} definition)
   * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
   */
  addCheckbox: function (config, position) {
    var paramDefinitionWithConfig = this.parametersBuilder._readDefinition(config);
    var paramCmpConfig = this.parametersBuilder.createParameter(paramDefinitionWithConfig, {
      xtype: 'checkbox',
    });

    if (this.parametersBuilder.isComponentParameter(paramDefinitionWithConfig.id) && paramDefinitionWithConfig.type != 'boolean') {
      return;
    }

    this.form.insertParameterCmp(
      position !== undefined ? position : this.form.getParametersCount(),
      paramCmpConfig,
      paramDefinitionWithConfig
    );
  },

  /**
   * Adds button to the form.
   *
   * @method addButton
   *
   * @param {Object} definition Button definition.
   * @param {String} definition.id Button id.
   * @param {String} definition.text Displayed button text.
   * @param {function} definition.handler Function that is triggered by clicking a button.
   * @param {Integer} position Position of a field in a form. If not specified, then field will be added on last position.
   */
  addButton: function (definition, position) {
    Ext.apply(definition, {
      itemId: definition.id,
      flex: 0,
      style: {
        marginTop: '0px',
      },
      height: '100%',
      width: 'auto',
    });

    this.form.insertSimpleCmp(position !== undefined ? position : this.form.getParametersCount(), new Ext.Button(definition));
  },

  /**
   * Hides element by id.
   *
   * @method hide
   *
   * @param {String} elementId Id of element to hide.
   */
  hide: function (elementId) {
    var element = this._find(elementId);
    if (element && element.hide) {
      element.hide();
    }
  },

  /**
   * Shows element by id.
   *
   * @method show
   *
   * @param {String} elementId Id of element to show.
   */
  show: function (elementId) {
    var element = this._find(elementId);
    if (element && element.show) {
      element.show();
    }
  },

  /**
   * Disables element by id.
   *
   * @method disable
   *
   * @param {String} elementId Id of element to disable.
   */
  disable: function (elementId) {
    var element = this._find(elementId);
    if (element && element.setReadOnly) {
      element.setReadOnly(true);
    }
  },

  /**
   * Enables element by id.
   *
   * @method enable
   *
   * @param {String} elementId Id of element to enable.
   */
  enable: function (elementId) {
    var element = this._find(elementId);
    if (element && element.setReadOnly) {
      element.setReadOnly(false);
    }
  },

  /**
   * Gets element value by id.
   * If element represents an array, then values array will be returned.
   *
   * @method getValue
   *
   * @param {String} elementId Id of element to get the value.
   */
  getValue: function (elementId) {
    var element = this._find(elementId);
    if (element && element.getValue) {
      return element.getValue();
    }
  },

  /**
   * Sets element value by id.
   * If element represents an array, then value should be an array.
   *
   * @method setValue
   *
   * @param {String} elementId Id of element to set the value.
   * @param value Value to set.
   */
  setValue: function (elementId, value) {
    var element = this._find(elementId);
    if (element && element.setValue) {
      element.setValue(value);
    }
  },

  /**
   * Changes requirement of parameter with provided id.
   * Only parameter declared as optional can have changed requirement.
   *
   * @param {String} elementId Id of element to set requirement.
   * @param value true to make parameter required, false otherwise.
   */
  setRequired: function (elementId, value) {
    var element = this._find(elementId);
    if (!element) {
      return;
    }

    var paramDefinition = this.parametersBuilder._readDefinition(elementId);
    if (!paramDefinition.optional) {
      return;
    }

    paramDefinition.optional = !value;
    this.parametersBuilder.refreshLabel(element.ownerCt, element, paramDefinition);

    if (element.setRequired) {
      element.setRequired(value);
    } else {
      element.allowBlank = !value;
    }
  },

  /**
   * Adds row to the form. The row may have other fields arranged horizontally. To add a field to a row, use the API function.
   * @example
   * 		var row = form.addRow();
   * 		row.addField('param-1');
   * 		row.addField('param-2');
   * 		row.addCombobox(...);
   *
   * @method addRow
   *
   * @param {Object} customConfig Configuration of row.
   * @param {String} customConfig.id Id of row.
   * @param {String} customConfig.fieldLabel Row label.
   * @param {Integer} customConfig.fieldsSpace Space between fields.
   */
  addRow: function (customConfig) {
    var config = Ext.applyIf(
      {},
      {
        id: undefined,
        fieldsSpace: 5,
      }
    );
    config = Ext.apply(config, customConfig);

    var me = this;
    var row = new Ext.ux.plusmpm.scheduledtasks.ParametersContainer({
      window: me.window,
      parametersBuilder: me.parametersBuilder,
      isRow: true,
      editable: this.form.editable,
      fieldsSpace: config.fieldsSpace,

      initialLabel: config.fieldLabel,
      fieldLabel: config.fieldLabel || ' ',
      labelSeparator: config.fieldLabel ? ':' : '',

      id: config.id,
      layout: 'hbox',
      items: [],
      defaults: {
        flex: 1,
        labelWidth: 250,
        allowBlank: false,
      },
    });

    return new Ext.ux.plusmpm.scheduledtasks.DynamicFormApi({
      form: me.form.add(row),
      parametersBuilder: me.parametersBuilder,
      window: me.window,
    });
  },

  /**
   * Adds fieldset to the form. The fieldset may have other fields grouped in named section.
   * To add a field to a fieldset, use the API function.
   *
   * @example
   * 		var fieldSet = form.addFieldSet('Form parameters');
   * 		fieldSet.addField('param-1');
   * 		fieldSet.addField('param-2');
   * 		fieldSet.addCombobox(...);
   *
   * @method addFieldSet
   *
   * @param {String} title Name of the fieldset.
   */
  addFieldSet: function (title) {
    var me = this;
    var fieldSet = new Ext.ux.plusmpm.scheduledtasks.ParametersFieldSet({
      window: me.window,
      parametersBuilder: me.parametersBuilder,
      editable: this.form.editable,

      title: title,
      collapsible: true,
      items: [],
    });

    return new Ext.ux.plusmpm.scheduledtasks.DynamicFormApi({
      form: this.form.add(fieldSet),
      parametersBuilder: me.parametersBuilder,
      window: me.window,
    });
  },

  _find: function (elementId) {
    return this.form.find('id', elementId)[0] || Ext.getCmp(elementId);
  },

  /**
   * Puts mask in the view.
   *
   * @method mask
   *
   * @param {Boolean} onlyForm If true then puts mask in form otherwise in whole panel.
   */
  mask: function (onlyForm) {
    var panel = onlyForm ? this.form : this.window;
    panel.el.mask(Ext.LoadMask.prototype.msg);
  },

  /**
   * Removes mask from the view.
   *
   * @method unmask
   *
   * @param {Boolean} onlyForm If true then removes mask from form otherwise from whole panel.
   */
  unmask: function (onlyForm) {
    var panel = onlyForm ? this.form : this.window;
    panel.el.unmask();
  },
});

Ext.reg('st_dynamicformapi', Ext.ux.plusmpm.scheduledtasks.DynamicFormApi);
