/*
 * Decompiled with CFR 0.152.
 */
package com.suncode.pwfl.upgrade;

import com.plusmpm.database.hibernate.DatabaseVendor;
import com.suncode.pwfl.config.Environment;
import com.suncode.pwfl.upgrade.UpgraderLockDao;
import com.suncode.pwfl.upgrade.UpgraderLockDaoImpl;
import com.suncode.pwfl.upgrade.UpgraderLockService;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.UUID;
import javax.sql.DataSource;
import org.glowroot.agent.api.Instrumentation;
import org.hibernate.TransactionException;
import org.hibernate.dialect.Dialect;
import org.hibernate.engine.jdbc.dialect.internal.StandardDialectResolver;
import org.hibernate.engine.jdbc.dialect.spi.DatabaseMetaDataDialectResolutionInfoAdapter;
import org.hibernate.engine.jdbc.dialect.spi.DialectResolutionInfo;
import org.hibernate.exception.ConstraintViolationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class UpgraderLockServiceImpl
implements UpgraderLockService {
    private static final Logger log = LoggerFactory.getLogger(UpgraderLockServiceImpl.class);
    private static final String UPGRADER_LOCK_RESOURCE_ID = "dummyResourceId";
    private static final long LOCK_WAIT_TIME_MS = Duration.ofSeconds(1L).toMillis();
    public static final long PERIOD_BETWEEN_MONITOR_INVOCATIONS_MS = Duration.ofSeconds(30L).toMillis();
    public static final Duration POSTPONE_LOCK_DEADLINE_DURATION = Duration.of(1L, ChronoUnit.MINUTES);
    private final String nodeId = Environment.getNodeId();
    private final UpgraderLockDao dao;

    public UpgraderLockServiceImpl(DataSource dataSource) {
        Dialect dialect = this.resolveDialect(dataSource);
        DatabaseVendor databaseType = DatabaseVendor.getInstance((Dialect)dialect);
        this.createLockTableIfNotExists(dataSource, databaseType);
        this.dao = new UpgraderLockDaoImpl(dataSource, dialect);
    }

    private Dialect resolveDialect(DataSource dataSource) {
        Dialect dialect;
        block8: {
            Connection connection = dataSource.getConnection();
            try {
                Dialect dialect2 = new StandardDialectResolver().resolveDialect((DialectResolutionInfo)new DatabaseMetaDataDialectResolutionInfoAdapter(connection.getMetaData()));
                log.info("Using hibernate dialect {}", (Object)dialect2);
                dialect = dialect2;
                if (connection == null) break block8;
            }
            catch (Throwable throwable) {
                if (connection != null) {
                    try {
                        connection.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
            connection.close();
        }
        return dialect;
    }

    private void createLockTableIfNotExists(DataSource dataSource, DatabaseVendor databaseType) {
        switch (databaseType) {
            case PostgreSQL: {
                this.createTablePostgresIfNotExists(dataSource);
                break;
            }
            case MicrosoftSQLServer: 
            case MicrosoftSQLServer2005: 
            case MicrosoftSQLServer2008: {
                this.createTableMssqlIfNotExists(dataSource);
                break;
            }
            case Oracle9i: 
            case Oracle10g: {
                this.createTableOracleIfNotExists(dataSource);
                break;
            }
            default: {
                throw new IllegalStateException(String.format("Unsupported database type: %s", databaseType.name()));
            }
        }
    }

    private void createTablePostgresIfNotExists(DataSource dataSource) {
        try (Connection connection = dataSource.getConnection();
             Statement statement = connection.createStatement();){
            statement.execute("    CREATE TABLE IF NOT EXISTS pm_upgrader_lock (\n        id BIGINT,\n        nodeid VARCHAR(255),\n        transactionid VARCHAR(255),\n        resourceid VARCHAR(255),\n        deadline TIMESTAMP,\n        CONSTRAINT pm_upgrader_lock_pk PRIMARY KEY (id),\n        CONSTRAINT pm_upgrader_lock_unique_res UNIQUE (resourceid)\n    );\n\n    CREATE SEQUENCE IF NOT EXISTS pm_upgrader_lock_id_seq\n        START WITH 1\n        INCREMENT BY 1\n        NO MINVALUE\n        NO MAXVALUE\n        CACHE 1;\n");
        }
        catch (SQLException e) {
            log.trace("Error occurred while creating upgrader lock table", (Throwable)e);
            log.info("Upgrader lock table probably already exists, skipping its creation");
        }
    }

    private void createTableMssqlIfNotExists(DataSource dataSource) {
        try (Connection connection = dataSource.getConnection();
             Statement statement = connection.createStatement();){
            statement.execute("    CREATE TABLE pm_upgrader_lock (\n        id BIGINT IDENTITY(1,1),\n        nodeid VARCHAR(255),\n        transactionid VARCHAR(255),\n        resourceid VARCHAR(255),\n        deadline DATETIME,\n        CONSTRAINT pm_upgrader_lock_pk PRIMARY KEY (id),\n        CONSTRAINT pm_upgrader_lock_unique_res UNIQUE (resourceid)\n    )\n");
        }
        catch (SQLException e) {
            log.trace("Error occurred while creating upgrader lock table", (Throwable)e);
            log.info("Upgrader lock table probably already exists, skipping its creation");
        }
    }

    private void createTableOracleIfNotExists(DataSource dataSource) {
        try (Connection connection = dataSource.getConnection();
             Statement statement = connection.createStatement();){
            statement.execute("    CREATE TABLE pm_upgrader_lock (\n        id NUMBER(19),\n        nodeid VARCHAR2(255),\n        transactionid VARCHAR2(255),\n        resourceid VARCHAR2(255),\n        deadline TIMESTAMP,\n        CONSTRAINT pm_upgrader_lock_pk PRIMARY KEY (id),\n        CONSTRAINT pm_upgrader_lock_unique_res UNIQUE (resourceid)\n    )\n");
            statement.execute("    CREATE SEQUENCE pm_upgrader_lock_id_seq\n        START WITH 1\n        INCREMENT BY 1\n        NOCACHE\n        NOCYCLE\n");
        }
        catch (SQLException e) {
            log.trace("Error occurred while creating upgrader lock table", (Throwable)e);
            log.info("Upgrader lock table probably already exists, skipping its creation");
        }
    }

    @Override
    public void doWithinUpgraderLock(Runnable runnable) {
        LockAttempt lockAttempt = new LockAttempt(this.nodeId);
        this.tracedDoWithinUpgraderLock(lockAttempt, runnable);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Instrumentation.Transaction(transactionType="Upgrader", transactionName="Synchronized upgrade execution", traceHeadline="Executing {{0.toString}}", timer="executing synchronized upgrade", alreadyInTransactionBehavior=Instrumentation.AlreadyInTransactionBehavior.CAPTURE_NEW_TRANSACTION)
    private void tracedDoWithinUpgraderLock(LockAttempt lockAttempt, Runnable runnable) {
        try {
            this.dao.clearPossiblyAbandonedLocksFromPreviousRun(this.nodeId);
        }
        catch (Exception e) {
            log.warn("Failed to clear possibly abandoned locks, skipping it anyway", (Throwable)e);
        }
        log.info("Start upgrader lock monitor");
        Thread thread = new Thread(() -> this.monitorUpgraderLock(lockAttempt), "upgrader-lock-monitor");
        thread.start();
        try {
            log.info("Locking upgrader lock");
            this.lock(lockAttempt);
            runnable.run();
        }
        finally {
            log.info("Stopping upgrader lock monitor");
            thread.interrupt();
            log.info("Unlocking upgrader lock");
            this.unlock(new UnlockAttempt(lockAttempt));
        }
    }

    void monitorUpgraderLock(LockAttempt lockAttempt) {
        try {
            while (true) {
                this.postponeLockDeadline(lockAttempt);
                this.clearAbandonedLocks();
                Thread.sleep(PERIOD_BETWEEN_MONITOR_INVOCATIONS_MS);
            }
        }
        catch (InterruptedException e) {
            log.info("Stopped upgrader lock monitor");
            return;
        }
    }

    private void postponeLockDeadline(LockAttempt lockAttempt) {
        try {
            this.dao.postponeLockDeadline(lockAttempt.nodeId(), lockAttempt.transactionId(), lockAttempt.resourceId(), Date.from(Instant.now().plus(POSTPONE_LOCK_DEADLINE_DURATION)));
        }
        catch (Exception e) {
            log.warn("Failed to perform upgrader locks postponing, skipping doing it", (Throwable)e);
        }
    }

    private void clearAbandonedLocks() {
        try {
            int removedAbandonedProcessLocks = this.dao.deleteAbandonedLocks();
            if (removedAbandonedProcessLocks > 0) {
                log.warn("Removed {} expired upgrader locks (most likely due to server node crash)", (Object)removedAbandonedProcessLocks);
            }
        }
        catch (Exception e) {
            log.warn("Failed to perform clean up of expired upgrader locks, skipping doing it", (Throwable)e);
        }
    }

    @Instrumentation.TraceEntry(message="Performing {{0.toString}}", timer="Performing resource lock")
    private void lock(LockAttempt lockAttempt) {
        while (this.tryLock(lockAttempt) == LockResult.AWAITING) {
            try {
                Thread.sleep(LOCK_WAIT_TIME_MS);
            }
            catch (InterruptedException e) {
                throw new TransactionException(String.format("Received interrupt signal during lock awaiting on %s", lockAttempt), (Throwable)e);
            }
        }
    }

    @Instrumentation.Timer(value="checking resource lock")
    private synchronized LockResult tryLock(LockAttempt lockAttempt) {
        boolean lockedSuccessfully = this.tryAcquireLock(lockAttempt);
        if (lockedSuccessfully) {
            return LockResult.ACQUIRED;
        }
        boolean isResourceLockOwner = this.tryCheckIsResourceLockOwning(lockAttempt);
        if (isResourceLockOwner) {
            return LockResult.ALREADY_LOCKED;
        }
        return LockResult.AWAITING;
    }

    private boolean tryAcquireLock(LockAttempt lockAttempt) {
        try {
            this.dao.insertLock(lockAttempt.nodeId(), lockAttempt.transactionId(), lockAttempt.resourceId(), Date.from(Instant.now().plus(1L, ChronoUnit.MINUTES)));
            return true;
        }
        catch (TransactionException e) {
            Throwable throwable = e.getCause();
            if (throwable instanceof ConstraintViolationException) {
                ConstraintViolationException constraintViolationException = (ConstraintViolationException)throwable;
                String jdbcErrorText = constraintViolationException.getCause() != null ? constraintViolationException.getCause().getMessage() : constraintViolationException.getMessage();
                log.trace("Constraint violation on insert by {}, most likely lock is held by other node: {}", (Object)lockAttempt, (Object)jdbcErrorText);
                return false;
            }
            throw new TransactionException(String.format("Error occurred while inserting resource lock by %s", lockAttempt), (Throwable)e);
        }
        catch (Exception e) {
            throw new TransactionException(String.format("Unknown error occurred while inserting resource lock by %s", lockAttempt), (Throwable)e);
        }
    }

    private boolean tryCheckIsResourceLockOwning(LockAttempt lockAttempt) {
        try {
            return this.dao.isResourceLockOwning(lockAttempt.nodeId(), lockAttempt.transactionId(), lockAttempt.resourceId());
        }
        catch (Exception e) {
            throw new TransactionException(String.format("Unknown error occurred while validating ownership of resource lock by %s", lockAttempt), (Throwable)e);
        }
    }

    @Instrumentation.TraceEntry(message="Performing {{0.toString}}", timer="Performing resource unlock")
    private void unlock(UnlockAttempt unlockAttempt) {
        try {
            this.dao.deleteLock(unlockAttempt.nodeId(), unlockAttempt.transactionId(), unlockAttempt.resourceId());
        }
        catch (Exception e) {
            log.warn("Failed to delete resource lock at {}", (Object)unlockAttempt, (Object)e);
        }
    }

    private record LockAttempt(String nodeId, String transactionId, String resourceId) {
        public LockAttempt(String nodeId) {
            this(nodeId, UUID.randomUUID().toString(), UpgraderLockServiceImpl.UPGRADER_LOCK_RESOURCE_ID);
        }
    }

    private record UnlockAttempt(String nodeId, String transactionId, String resourceId) {
        public UnlockAttempt(LockAttempt lockAttempt) {
            this(lockAttempt.nodeId(), lockAttempt.transactionId(), lockAttempt.resourceId());
        }
    }

    private static enum LockResult {
        ACQUIRED,
        ALREADY_LOCKED,
        AWAITING;

    }
}

