/**
 * JIRA: PWFL-2943
 * Hotfix podglądu dokumentów przy użyciu wtyczki Adobe Acrobat Reader w przeglądarce Firefox.
 *
 * W przypadku przeglądarki Firefox oraz wtyczki Adobe Acrobat Reader za każdym razem kiedy klikniemy w obszar wtyczki,
 * focus zostanie przekazany do wtyczki i jedynym sposobem odzyskania focusu jest manualne sfocusowanie dowolnego,
 * wcześniej nie sfocusowanego elementu.
 * 
 * Hotfix polega na przechwyceniu wszystkich eventów kliknięcia na formularzu. Gdy przechwycimy taki event, resetujemy focus elementu:
 * - jeżeli element jest focusowalny (może otrzymać focus) - focusujemy go jeszcze raz (blur i focus)
 * - jeżeli element nie jest focusowalny - focusujemy pomocniczy niewidoczny obiekt, po czym focusujemy docelowy element
 *
 * Refs:
 * https://bugzilla.mozilla.org/show_bug.cgi?id=93149
 * https://bugzilla.mozilla.org/show_bug.cgi?id=78414
 */
(function (Ext, $) {
    var focusLost = false;
    
    SuncodeFormService.addOnReadyEvent(function () {
        if (applyFix()) {
            var $focusEl = createFocusEl();
            var documents = Ext.ComponentQuery.query('documentviewpanel');
 
            var resetFocus = function (target) {
                if(focusLost){
                    var $target = target ? $(target) : $(document.activeElement);
                    if ($target.is(':focus')) {
                        $target.blur().focus();
                    } else {
                        $focusEl.focus();
                        $target.focus();
                    }
                    focusLost = false;
                }
            };
            
            // przechwytujemy każdy event click i resetujemy focus
            $(document).mousedown(function (event) {
                resetFocus(event.target);
            });
            
            // po załadowaniu dokumentu reset focus'u
            for(var i=0; i<documents.length; i++){
                documents[i].on('load', function(){
                    focusLost = true;
                    setTimeout(resetFocus, 100);
                });
            }
            
            // Po każdym wyjściu z pola widoku dokumentów możliwa jest strata focusu
            var region = documents[0].up('tabpanel').body.dom;
            $(region).mouseleave(function(){
                 focusLost = true;
            });
        }
    });

    function applyFix() {
        if (Ext.isGecko && navigator.plugins.namedItem("Adobe Acrobat")) {
            var documents = Ext.ComponentQuery.query('documentviewpanel');
            return !Ext.isEmpty(documents);
        }
        return false;
    };

    function createFocusEl() {
        var $focusEl = $('<div/>').appendTo('body');
        $focusEl.attr('tabindex', '-1');
        $focusEl.attr('class', 'focus-reset');
        $focusEl.css({
            'position': 'fixed',
            'top': '-1px',
            'left': '-1px'
        });
        return $focusEl;
    };
})(Ext4, jQuery);