PW.FormActions.create('tools.screenshot', {
    defaultActions: {
        button: async function () {
            const expandComments = this.get("expandComments");
            const expandTables = this.get("expandTables");
            const includeTabBar = this.get('includeTabBar');
            const includeComments = this.get('includeComments');
            const formTables = VariableSetService.getAll();
            const commentsTable = Ext4.getCmp(document.querySelector("div[id^=commentstable]").id);
            const tabBar = Ext4.getCmp(document.querySelector("div[id^=tabbar]").id);
            const rootElement = Ext4.getCmp(document.querySelector("[id^=suncodeform]").id);

            PW.ui.showLoading();

            this.setScreenshotLayout(rootElement, expandComments, expandTables, includeComments, includeTabBar, formTables, commentsTable, tabBar);
            let outputFileFormat = this.get('format')?.toLowerCase() ?? 'pdf';
            if (outputFileFormat !== 'pdf' && outputFileFormat !== 'png' && outputFileFormat !== 'jpg') outputFileFormat = 'pdf';
            let pdfPageOrientation = this.get('orientation')?.toLowerCase() ?? 'portrait';
            if (pdfPageOrientation !== 'landscape' && pdfPageOrientation !== 'portrait') pdfPageOrientation = 'portrait';
            let outputFilename = this.get('filename') ?? 'screenshot_' + Date.now();
            outputFilename = (outputFilename.replace(/[^a-zA-Z0-9]/g, ''));
            let fitTo = this.get('fitTo')?.toLowerCase() ?? 'pageheight';
            if (fitTo !== 'pageheight' && fitTo !== 'pagewidth') fitTo = 'pageheight';
            const jpegQuality = this.get("jpegQuality") ?? 0.8;
            let pdfCompression = this.get("pdfCompression")?.toUpperCase() ?? "NONE";
            if (!["NONE", "FAST", "MEDIUM", "SLOW"].includes(pdfCompression)) pdfCompression = "NONE";

            let options = {
                useProxy: true,
                excludedNodes: [],
                rootNode: rootElement.el.dom,
                width: rootElement.getWidth(),
                height: rootElement.getHeight(),
                excludedCssRules: ['text-overflow', 'box-shadow']
            };

            let imageData = await this.render(rootElement.el.dom, options);
            let canvas = await this.createCanvas(imageData, options);
            this.saveFile(canvas, outputFileFormat, outputFilename, pdfPageOrientation, fitTo, jpegQuality, pdfCompression);
            this.resetScreenshotLayout(rootElement, formTables, commentsTable, tabBar);
            PW.ui.hideLoading();
        }
    },
    setScreenshotLayout: function (rootElement, expandComments, expandTables, includeComments, includeTabBar, formTables, commentsTable, tabBar) {
        let maxWidthDifference = 0;

        if (!includeComments) {
            commentsTable.setVisible(false);
        } else if (expandComments) {
            const difference = this.getScrollableSizeDifference(commentsTable);
            commentsTable.setHeight(commentsTable.getHeight() + difference.deltaY);
            maxWidthDifference = Math.max(maxWidthDifference, difference.deltaX);
        }

        if (!includeTabBar) {
            tabBar.setVisible(false);
        }

        if (expandTables) {
            for (let i = 0; i < formTables.length; i++) {
                const table = formTables[i];
                table["initialHeight"] = table.initialConfig.autoHeightConfig ? table.getHeight() : table.initialConfig.height;
                const difference = this.getScrollableSizeDifference(table);

                // do zmiany po wydaniu PWFL-10611
                const skipColumnResizeTask = table.skipColumnResizeTask;
                table["skipColumnResizeTask"] = {
                    delay: () => {
                    }
                };
                table.setHeight(table.getHeight() + difference.deltaY + 20); // 20 - wysokość scrollbara
                table["skipColumnResizeTask"] = skipColumnResizeTask;

                maxWidthDifference = Math.max(maxWidthDifference, difference.deltaX);
            }
        }

        rootElement.setWidth(rootElement.getWidth() + maxWidthDifference);
    },
    resetScreenshotLayout: function (rootElement, formTables, commentsTable, tabBar) {
        rootElement.setWidth(rootElement.initialConfig.width);
        for (let i = 0; i < formTables.length; i++) {
            const table = formTables[i];
            table.setHeight(table["initialHeight"]);
            delete table["initialHeight"];
        }
        commentsTable.setHeight(commentsTable.initialConfig.height);
        commentsTable.setVisible(true);
        tabBar.setVisible(true);
    },
    getScrollableSizeDifference: function (extElement) {
        const scrollableDomElement = extElement.body.dom.firstChild;

        return {
            deltaX: scrollableDomElement.scrollWidth - scrollableDomElement.clientWidth,
            deltaY: scrollableDomElement.scrollHeight - scrollableDomElement.clientHeight
        }
    },
    createCanvas: async function (imageData, options) {
        let canvas = document.createElement('canvas');
        let context = canvas.getContext('2d');

        canvas.width = options['width'];
        canvas.height = options['height'];
        context.fillStyle = "rgb(255, 255, 255)";
        context.fillRect(0, 0, canvas.width, canvas.height);

        try {
            await this.util().preloadFonts();
        } catch (error) {
            console.error("Failed to preload font!");
        } finally {
            context.drawImage(imageData, 0, 0);
        }

        return canvas;
    },
    saveFile: function (canvas, fileFormat, fileName, pageOrientation, fitTo, jpegQuality, pdfCompression) {
        let saveLink = document.createElement('a');

        switch (fileFormat) {
            case 'png': {
                saveLink.href = canvas.toDataURL();
                saveLink.download = fileName;
                saveLink.click();
                break;
            }
            case 'jpg': {
                saveLink.href = canvas.toDataURL('image/jpeg', jpegQuality);
                saveLink.download = fileName;
                saveLink.click();
                break;
            }
            case 'pdf': {
                let doc = new window.jspdf.jsPDF({orientation: pageOrientation, unit: "px", format: "a4", hotfixes: ['px_scaling']});
                let pdfPageWidth = doc.internal.pageSize.getWidth();
                let pdfPageHeight = doc.internal.pageSize.getHeight();

                if (fitTo === 'pageheight') {
                    let ratio = Math.min(pdfPageWidth / canvas.width, pdfPageHeight / canvas.height);
                    doc.addImage(canvas.toDataURL(), 'JPEG', 0, 0, canvas.width * ratio, canvas.height * ratio, "", pdfCompression);
                } else if (fitTo === 'pagewidth') {
                    let ratio = pdfPageWidth / canvas.width;
                    let newHeight = canvas.height * ratio;
                    let numberOfPages = Math.ceil(newHeight / pdfPageHeight);
                    doc.addImage(canvas.toDataURL(), 'JPEG', 0, 0, pdfPageWidth, newHeight, "", pdfCompression);

                    for (let i = 1; i < numberOfPages; i++) {
                        doc.addPage("a4", pageOrientation);
                        doc.addImage(canvas.toDataURL(), 'JPEG', 0, (i * pdfPageHeight * -1), pdfPageWidth, newHeight, "", pdfCompression);
                    }
                }

                doc.setProperties({title: fileName});
                doc.output('save', fileName + '.pdf');
                break;
            }
        }
    },
    render: async function (startingPoint, options) {
        let util = this.util();
        let clone = await cloneNode(startingPoint, options);
        let svgStyle = await createSvgStyleObject();
        let svgForeignObject = createSvgForeignObject(clone);
        let svgNode = createSvgNode(svgStyle, svgForeignObject, options);
        let svgDataUri = createSvgDataUri(svgNode);
        return createImageFromDataUri(svgDataUri);

        async function cloneNode(node, options) {
            if (options['excludedNodes'].includes(node)) return;

            let clone = node.cloneNode(false);
            let children = node.childNodes;

            for (let i = 0; i < children.length; i++) {
                let copy = await cloneNode(children[i], options);
                if (copy) clone.appendChild(copy);
            }

            let cloneWithStyles = await copyStyles(node, clone, options);
            let cloneWithInputValue = copyInputValue(node, cloneWithStyles, options);
            return embedImage(node, cloneWithInputValue, options);

            async function copyStyles(node, clone, options) {
                if (!(node instanceof Element)) return clone;

                let source = getComputedStyle(node);
                let target = clone.style;

                for (let i = 0; i < source.length; i++) {
                    let cssProperty = source[i];
                    let cssPropertyValue = source.getPropertyValue(source[i]);
                    let cssPropertyPriority = source.getPropertyPriority(source[i]);

                    if (options['excludedCssRules'].includes(cssProperty)) continue;

                    if (cssProperty === 'background-image' && (cssPropertyValue !== '' && cssPropertyValue !== 'none')) {
                        try {
                            let url = util.extractUrlFromCssSrcString(cssPropertyValue);
                            cssPropertyValue = 'url("' + await util.downloadAsBase64(util.resolveUrl(url)) + '")';
                            cssPropertyPriority = 'important';
                        } catch (error) {
                            console.log(error);
                        }
                    }
                    target.setProperty(cssProperty, cssPropertyValue, cssPropertyPriority);
                }

                if (node === options['rootNode']) {
                    target.setProperty('margin-top', '0px', 'important');
                    target.setProperty('margin-right', '0px', 'important');
                    target.setProperty('margin-bottom', '0px', 'important');
                    target.setProperty('margin-left', '0px', 'important');
                    target.setProperty('position', 'absolute', 'important');
                    target.setProperty('top', '0px', 'important');
                    target.setProperty('left', '0px', 'important');
                    target.setProperty('border', 'none', 'important');
                    target.setProperty('border-radius', '0', 'important');
                }

                return clone;
            }

            function copyInputValue(node, clone) {
                if (node instanceof HTMLTextAreaElement) clone.innerHTML = node.value;
                if (node instanceof HTMLInputElement) clone.setAttribute('value', node.value);

                return clone;
            }

            async function embedImage(node, clone, options) {
                if (!(node instanceof HTMLImageElement)) return clone;

                try {
                    clone.src = await util.downloadAsBase64(util.resolveUrl(node.src));
                } catch (error) {
                    if (!options.useProxy) {
                        clone.src = util.createPlaceholderImage(node.clientWidth, node.clientHeight);
                    } else {
                        try {
                            clone.src = await util.downloadAsBase64WithProxy(util.resolveUrl(node.src));
                        } catch (error) {
                            clone.src = util.createPlaceholderImage(node.clientWidth, node.clientHeight);
                        }
                    }
                }

                return clone;
            }
        }

        async function createSvgStyleObject() {
            let style = document.createElement('style');
            let defs = document.createElement('defs');

            for (let i = 0; i < document.styleSheets.length; i++) {
                for (let j = 0; j < document.styleSheets[i].cssRules.length; j++) {
                    let rule = document.styleSheets[i].cssRules[j];
                    if (rule instanceof CSSFontFaceRule) {
                        try {
                            let url = util.extractUrlFromCssSrcString(rule.style.getPropertyValue('src'));
                            let baseUrl = (rule.parentStyleSheet || {}).href;
                            let base64FontObject = await util.downloadAsBase64(util.resolveUrl(url, baseUrl));

                            style.appendChild(document.createTextNode('@font-face {font-display: swap;' +
                                'font-family: ' + rule.style.getPropertyValue("font-family") + ';' +
                                'font-style: ' + rule.style.getPropertyValue("font-style") + ';' +
                                'font-weight: ' + rule.style.getPropertyValue("font-weight") + ';' +
                                'unicode-range: ' + rule.style.getPropertyValue("unicode-range") + ';' +
                                'src: url("' + base64FontObject + '");}'));
                        } catch (error) {
                            console.log(error);
                        }
                    }
                }
            }

            defs.appendChild(style);
            return escapeXhtml(new XMLSerializer().serializeToString(defs));
        }

        function createSvgForeignObject(cloneNode) {
            cloneNode.setAttribute('xmlns', 'http://www.w3.org/1999/xhtml');

            let serialized = escapeXhtml(new XMLSerializer().serializeToString(cloneNode));
            return '<foreignObject x="0" y="0" width="100%" height="100%">' + serialized + '</foreignObject>';
        }

        function escapeXhtml(string) {
            return string
                .replace(/#/g, '%23')
                .replace(/\n/g, '%0A')
                .replace(/[\u0000-\u001F\u007F-\u009F\u061C\u200E\u200F\u202A-\u202E\u2066-\u2069]/g, "");
        }

        function createSvgNode(svgStyleNode, svgForeignObject, options) {
            return '<svg xmlns="http://www.w3.org/2000/svg" width="' + options.width + '" height="' + options.height + '">' +
                svgStyleNode + svgForeignObject + '</svg>';
        }

        function svgToBase64(svg) {
            const uint8 = new TextEncoder().encode(svg);
            let binary = "";
            uint8.forEach(value => binary += String.fromCharCode(value));
            return btoa(binary);
        }

        function createSvgDataUri(svg) {
            return 'data:image/svg+xml;base64,' + svgToBase64(svg);
        }

        async function createImageFromDataUri(dataUri) {
            let image = new Image();
            image.src = dataUri;
            await image.decode();
            return image;
        }
    },
    util: function () {
        return {
            resolveUrl: resolveUrl,
            downloadAsBase64: downloadAsBase64,
            createPlaceholderImage: createPlaceholderImage,
            downloadAsBase64WithProxy: downloadAsBase64WithProxy,
            extractUrlFromCssSrcString: extractUrlFromCssSrcString,
            preloadFonts: preloadFonts
        }

        function preloadFonts() {
            const promises = [];

            for (let i = 0; i < document.styleSheets.length; i++) {
                for (let j = 0; j < document.styleSheets[i].cssRules.length; j++) {
                    let rule = document.styleSheets[i].cssRules[j];
                    if (rule instanceof CSSFontFaceRule) {
                        let url = extractUrlFromCssSrcString(rule.style.getPropertyValue('src'));
                        let baseUrl = (rule.parentStyleSheet || {}).href;
                        let resolvedUrl = resolveUrl(url, baseUrl);
                        let font = new FontFace(rule.style.getPropertyValue("font-family"), "url(" + resolvedUrl + ")", {
                            style: rule.style.getPropertyValue("font-style"),
                            weight: rule.style.getPropertyValue("font-weight")
                        });
                        document.fonts.add(font);
                        promises.push(font.load());
                    }
                }
            }

            return Promise.all(promises);
        }

        function resolveUrl(url, baseUrl) {
            if (!baseUrl) return url;

            let doc = document.implementation.createHTMLDocument();
            let base = document.createElement('base');
            let anchor = document.createElement('a');

            doc.head.appendChild(base);
            doc.body.appendChild(anchor);
            base.href = baseUrl;
            anchor.href = url;

            return anchor.href;
        }

        async function downloadAsBase64(url) {
            if (_isDataUrl(url)) return url;

            let response = await fetch(url);
            let blob = await response.blob();

            return new Promise((resolve, reject) => {
                const reader = new FileReader();
                reader.readAsDataURL(blob);
                reader.onload = () => resolve(reader.result);
                reader.onerror = (error) => reject(error);
            });
        }

        function _isDataUrl(url) {
            return url.search(/^(data:)/) !== -1;
        }

        function createPlaceholderImage(width, height) {
            let svg = '<svg xmlns="http://www.w3.org/2000/svg" style="display:block;" width="' + width + '" height="' + height + '">' +
                '<rect x="0" y="0" width="' + width + '" height="' + height + '" style="fill: none; stroke: darkgrey" />' +
                '<line x1="0" y1="0" x2="' + width + '" y2="' + height + '" style="stroke: darkgrey" />' +
                '<line x1="0" y1="' + height + '" x2="' + width + '" y2="0" style="stroke: darkgrey" /></svg>';
            return "data:image/svg+xml;base64," + btoa(svg);
        }

        async function downloadAsBase64WithProxy(url) {
            return downloadAsBase64("plugin/com.suncode.plugin-tools/proxy/?path=" + url);
        }

        function extractUrlFromCssSrcString(cssSrcString) {
            let URL_REGEX = /\burl\((?<url>[^()]+)\)/gm;
            let matches = URL_REGEX.exec(cssSrcString);
            return _removeQuotesFromUrlString(matches[1]);
        }

        function _removeQuotesFromUrlString(url) {
            return url.replace(/^["'](.+(?=["']$))["']$/, '$1');
        }
    }
});