RESTDS = {
    t: PW.I18N.createT('com.suncode.plugin-rest-datasources')
};

PW.DataSources.register("restQueryDatasource", {
    buildParams: function (form, configuration) {
        switch (configuration.operation) {
            case "READ": {
                this.buildAuthorizationForm(form);
                this.buildHeaderForm(form);
                this.buildUrlForm(form);
                this.buildContentTypeForm(form);
                this.buildResponseContentForm(form);
                this.buildSchemaForm(form);
                this.buildSplitCharForm(form);
                this.buildConnectionForm(form);
                this.buildUploadFilesForm(form);
                this.buildInputParametersForm(form);
                this.buildOutputParametersFormForRead(form);
                this.buildRawDataPreviewForm(form);
                break;
            }
            case "FILE": {
                this.buildAuthorizationForm(form);
                this.buildHeaderForm(form);
                this.buildUrlForm(form);
                this.buildContentTypeForm(form);
                this.buildSchemaForm(form);
                this.buildSplitCharForm(form);
                this.buildConnectionForm(form);
                this.buildDocumentConfigForm(form);
                this.buildInputParametersForm(form);
                this.buildOutputParametersFormForFile(form);
                break;
            }
        }
    },

    rawDataPreview: async function (form) {
        const me = this;

        function sleep(ms) {
            return new Promise(function (resolve) {
                setTimeout(resolve, ms)
            });
        }

        if (!form.configurationPanel.isValid()) return;

        form.form.mask();
        let queryResultColumnsId = form.getValue('queryResultColumnsId');
        let queryResultColumnsName = form.getValue('queryResultColumnsName');
        let queryResultColumnsType = form.getValue('queryResultColumnsType');
        let queryResultColumnsPath = form.getValue('queryResultColumnsPath');
        let queryResultColumnsChildNodeName = form.getValue('queryResultColumnsChildNodeName');
        let outputParameters = form.form.down("#outputParameters");
        outputParameters.setValues({
            queryResultColumnsId: [{value: 'rawdata'}],
            queryResultColumnsName: [{value: 'rawdata'}],
            queryResultColumnsType: [{value: 'body_param'}],
            queryResultColumnsPath: [{value: 'string'}],
            queryResultColumnsChildNodeName: [{value: 'string'}]
        });

        await Suncode.datasourcemanager.getApplication().getSuncodeDatasourcemanagerManagerController().onExecuteButtonClick();

        if (queryResultColumnsId.length === 0 || (queryResultColumnsId.length === 1 && queryResultColumnsId[0] === '')) {
            outputParameters.addRow(true);
            outputParameters.removeRow(1);
        } else {
            let params = {
                queryResultColumnsId: queryResultColumnsId.map(function (id) {
                    return {value: id}
                }),
                queryResultColumnsName: queryResultColumnsName.map(function (name) {
                    return {value: name}
                }),
                queryResultColumnsPath: queryResultColumnsPath.map(function (path) {
                    return {value: path}
                }),
                queryResultColumnsType: queryResultColumnsType.map(function (type) {
                    return {value: type}
                }),
                queryResultColumnsChildNodeName: queryResultColumnsChildNodeName.map(function (childNode) {
                    return {value: childNode}
                })
            };

            outputParameters.setValues(params);
        }

        let executionWindow;
        while (!executionWindow) {
            executionWindow = Ext4.ComponentQuery.query('executionparameterswindow')[0];
            await sleep(10);
        }
        form.form.unmask();

        let executeButton = executionWindow.down("button[text='" + PW.t('datasources-execute') + "']")
        executeButton.handler = function () {
            let executionWindowForm = executionWindow.down('form');
            if (executionWindowForm.isValid()) {
                let operation = executionWindow.temporaryDataSource.dataSource.declaration.operation;
                executionWindowForm.submit({
                    url: PW.getAbsolutePath('api/datasources/temporary/' + executionWindow.temporaryDataSource.uuid
                        + '/' + operation.toLowerCase()),
                    headers: {
                        //musi być, bo w przypadku braku pól na formie wysyłany jest zły contentType.
                        'Content-Type': 'application/x-www-form-urlencoded'
                    },
                    waitMsg: PW.t('progressTitle'),
                    failure: function (form, action) {
                        let response = Ext4.decode(action.response.responseText);
                        executionWindow.close();
                        PW.ui.Message.warning(response.message);
                    },
                    success: async function (form, action) {
                        executionWindow.mask();
                        let response = await jQuery.ajax(Suncode.getAbsolutePath('api/datasources/temporary/'
                            + executionWindow.temporaryDataSource.uuid
                            + '/data?start=0&limit=1&direction=ASC&property=rawdata'));
                        if (response.data[0]) {
                            executionWindow.close();
                            me.showPreviewResult(response.data[0]);
                        } else {
                            executionWindow.close();
                            PW.ui.Message.error(RESTDS.t('restDatasource.testError'));
                        }
                    }
                });
            }
        }
    },

    showPreviewResult: function (result) {
        Ext4.create('Ext4.window.Window', {
            title: RESTDS.t('restDatasource.window.rawPreview.title'),
            width: 700,
            modal: true,
            frame: false,
            layout: 'fit',
            autoShow: true,
            items: [{
                xtype: 'form',
                layout: 'form',
                frame: false,
                border: false,
                bodyPadding: 10,
                defaults: {
                    labelWidth: 0
                },
                items: [{
                    xtype: 'textarea',
                    id: 'requestPreview',
                    fieldLabel: RESTDS.t('restDatasource.window.rawPreview.textarea.request.name'),
                    labelAlign: 'top',
                    minHeight: 250,
                    value: result['request']
                },
                    {
                        xtype: 'textarea',
                        id: 'responseBodyPreview',
                        fieldLabel: RESTDS.t('restDatasource.window.rawPreview.textarea.responseBody.name'),
                        labelAlign: 'top',
                        minHeight: 250,
                        value: result['response']
                    }, {
                        xtype: 'textarea',
                        id: 'responseHeadersPreview',
                        fieldLabel: RESTDS.t('restDatasource.window.rawPreview.textarea.responseHeaders.name'),
                        labelAlign: 'top',
                        minHeight: 250,
                        value: result['headers']
                    }]
            }],
            buttons: [{
                text: RESTDS.t('restDatasource.window.rawPreview.button.ok.text'),
                handler: function (button) {
                    button.up('window').close();
                }
            }],
        });
    },

    showSchemaFields: function (form) {
        const contentTypeField = this.getContentTypeField(form);
        return contentTypeField.getValue() === "json";
    },


    buildSchemaForm: function (form) {
        form.addTextArea({
            id: "schema",
            hidden: !this.showSchemaFields(form),
            listeners: {
                beforerender: function (field) {
                    field.minHeight = 250;
                }
            }
        });
    },

    buildSplitCharForm: function (form) {
        form.addField({
            id: "splitCharacter",
            hidden: !this.showSchemaFields(form)
        });
    },

    buildConnectionForm: function (form) {
        form.addField("connectionTimeout");
        form.addField("allowUntrustedCertificates");
        form.addField("isLoggingActive");
        form.addField("statusCodeRegex");
    },

    buildInputParametersForm: function (form) {
        const inputParametersTable = form.addTable({id: "inputParameters"});
        inputParametersTable.addField("queryParametersId");
        inputParametersTable.addField("queryParametersName");
        inputParametersTable.addCombobox({
            id: "queryParametersType",
            valueField: "id",
            displayField: "display",
            fields: [
                {name: "id", type: "string"},
                {name: "display", type: "string"},
            ],
            values: [
                {id: "body_param", display: RESTDS.t("restDatasource.bodyParam")},
                {id: "url_param", display: RESTDS.t("restDatasource.urlParam")},
                {id: "header_param", display: RESTDS.t("restDatasource.headerParam")},
                {id: "auth_param", display: RESTDS.t("restDatasource.authParam")},
                {id: "file_param", display: RESTDS.t("restDatasource.fileParam")},
                {id: "path_key_param", display: RESTDS.t("restDatasource.pathKeyParam")}
            ],
        });
        inputParametersTable.addField("queryParametersArrayElement");
    },

    buildOutputParametersFormForRead: function (form) {
        var outputParametersTable = form.addTable({id: "outputParameters"});
        outputParametersTable.addField("queryResultColumnsId");
        outputParametersTable.addField("queryResultColumnsName");
        outputParametersTable.addField("queryResultColumnsPath");
        outputParametersTable.addField("queryResultColumnsChildNodeName");
        outputParametersTable.addCombobox({
            id: "queryResultColumnsType",
            valueField: "id",
            displayField: "display",
            fields: [
                {name: "id", type: "string"},
                {name: "display", type: "string"},
            ],
            values: [
                {id: "body_param", display: RESTDS.t("restDatasource.bodyParam")},
                {id: "header_param", display: RESTDS.t("restDatasource.headerParam")},
                {id: "cookie_param", display: RESTDS.t("restDatasource.cookieParam")}
            ]
        });
    },

    buildOutputParametersFormForFile: function (form) {
        var outputParametersTable = form.addTable({id: "outputParameters"});
        outputParametersTable.addField("queryResultColumnsId");
        outputParametersTable.addField("queryResultColumnsName");
        outputParametersTable.addField("queryResultColumnsPath");
        outputParametersTable.addCombobox({
            id: "queryResultColumnsType",
            valueField: "id",
            displayField: "display",
            fields: [
                {name: "id", type: "string"},
                {name: "display", type: "string"},
            ],
            values: [
                {id: "header_param", display: RESTDS.t("restDatasource.headerParam")},
                {id: "cookie_param", display: RESTDS.t("restDatasource.cookieParam")}
            ],
        });
    },

    buildRawDataPreviewForm: function (form) {
        form.form.add({
            xtype: 'button',
            id: 'rawdata_preview',
            text: RESTDS.t('restDatasource.button.rawDataPreview.text'),
            margin: '0 0 5 0',
            handler: () => {
                this.rawDataPreview(form);
            }
        });
    },

    buildAuthorizationForm: function (form) {
        var authorizationRow = form.addRow({id: "authorization-row"});
        authorizationRow.addCombobox({
            id: "authorization",
            valueField: "id",
            displayField: "name",
            queryCaching: false,
            remote: {
                url: PW.getAbsolutePath("/plugin/com.suncode.plugin-rest-datasources/authorizations"),
                fields: ["id", "name"],
                remoteSort: true,
            },
            sort: [
                {
                    property: "name",
                    direction: "ASC",
                },
            ],
        });
        authorizationRow.addButton({
            text: RESTDS.t("restDatasource.createAuthorizations"),
            handler: function () {
                this.showManageAuthorizationsWindow(authorizationRow, false);
            }.bind(this),
            id: "createAuthorizationsButton",
        });
        authorizationRow.addButton({
            text: RESTDS.t("restDatasource.modifyAuthorizations"),
            handler: function () {
                this.showManageAuthorizationsWindow(authorizationRow, true);
            }.bind(this),
            id: "modifyAuthorizationsButton",
        });
    },

    buildHeaderForm: function (form) {
        var headersTable = form.addTable({
            id: "headersTable",
            name: RESTDS.t("restDatasource.additionalHeaders.name"),
            description: RESTDS.t("restDatasource.additionalHeaders.desc"),
        });
        headersTable.addField("customHeadersKeys");
        headersTable.addField("customHeadersValues");
    },

    buildUrlForm: function (form) {
        var row = form.addRow({id: "urlConfig"});
        row.addCombobox({
            id: "httpMethod",
            valueField: "id",
            displayField: "display",
            fields: [
                {name: "id", type: "string"},
                {name: "display", type: "string"},
            ],
            values: [
                {id: "GET", display: "GET"},
                {id: "HEAD", display: "HEAD"},
                {id: "POST", display: "POST"},
                {id: "PUT", display: "PUT"},
                {id: "DELETE", display: "DELETE"},
                {id: "PATCH", display: "PATCH"},
            ],
            listeners: {
                added: function (combobox, _container, _position) {
                    if (!form.getValue("httpMethod")) {
                        combobox.setValue("GET");
                    }
                },
            },
        });
        row.addField("url");
        form.form.items.get("urlConfig").items.get("url").labelWidth = 50;
    },

    buildContentTypeForm: function (form) {
        form.addCombobox({
            id: "contentType",
            valueField: "id",
            displayField: "display",
            fields: [
                {name: "id", type: "string"},
                {name: "display", type: "string"},
            ],
            values: [
                {id: "json", display: "application/json"},
                {id: "formUrlencoded", display: "application/x-www-form-urlencoded"},
                {id: "formData", display: "multipart/form-data"},
            ],
            value: "json",
            listeners: {
                change: () => {
                    if (this.showSchemaFields(form)) {
                        form.show("schema");
                        form.show("splitCharacter");
                    } else {
                        form.hide("schema");
                        form.hide("splitCharacter");
                    }

                    this.showUploadFilesForm(form);
                }
            }
        });

        let contentTypeField = this.getContentTypeField(form);
        if (contentTypeField.getValue() === null) {
            contentTypeField.setValue("json");
        }
    },

    buildUploadFilesForm: function (form) {
        const uploadFilesTable = form.addTable({
            id: "uploadFilesTable"
        });

        uploadFilesTable.addField("uploadFileKey");
        uploadFilesTable.addField("uploadFileId");

        this.showUploadFilesForm(form);
    },

    showUploadFilesForm: function (form) {
        if (this.getContentTypeField(form).getValue() === "formData") {
            form.show("uploadFilesTable");
        } else {
            form.hide("uploadFilesTable");
        }
    },

    buildResponseContentForm: function (form) {
        form.addCombobox({
            id: "responseContentType",
            valueField: "id",
            displayField: "display",
            fields: [
                {name: "id", type: "string"},
                {name: "display", type: "string"},
            ],
            values: [
                {id: "json", display: "application/json"},
                {id: "xml", display: "application/xml"},
            ],
            value: "json",
        });

        let responseContentTypeField = form.form.items.get("responseContentType");
        if (responseContentTypeField.getValue() === null) {
            responseContentTypeField.setValue("json");
        }
    },

    getContentTypeField: function (form) {
        return form.form.items.get("contentType");
    },

    getAttachToProcessField: function (form) {
        return form.form.items.get("attachToProcess");
    },

    buildDocumentConfigForm: function (form) {
        form.addField("documentClass");
        form.addField("filename");
        form.addField("fileDescription");
        form.addCheckbox("executeDocumentAction");

        form.form.add({
            id: "attachToProcess",
            name: "attachToProcess",
            fieldLabel: RESTDS.t("restDatasource.attachToProcess.label"),
            valueField: "id",
            displayField: "display",
            store: Ext4.create("Ext4.data.Store", {
                fields: ["id", "display"],
                data: [
                    {id: "none", display: RESTDS.t("restDatasource.attachToProcessParam.none")},
                    {id: "add_to_process", display: RESTDS.t("restDatasource.attachToProcessParam.add_to_process")},
                    {id: "add_to_activity", display: RESTDS.t("restDatasource.attachToProcessParam.add_to_activity")},
                ],
            }),
            allowBlank: false,
            submitValue: false,
            xtype: "combo",
            listeners: {
                change: () => {
                    this.showAttachToProcessFields(form);
                },
                afterrender: () => {
                    this.selectAttachToProcess(form);
                }
            }
        });

        form.addField("processId");
        form.addField("activityId");

        form.addCombobox({
            id: "saveAsNewVersion",
            valueField: "id",
            displayField: "display",
            fields: [
                {name: "id", type: "string"},
                {name: "display", type: "string"},
            ],
            values: [
                {id: "none", display: RESTDS.t("restDatasource.saveAsNewVersion.none")},
                {id: "fileid", display: RESTDS.t("restDatasource.saveAsNewVersion.fileid")},
                {id: "indices", display: RESTDS.t("restDatasource.saveAsNewVersion.indices")}
            ],
            listeners: {
                change: () => {
                    form.setValue("fileId", "");

                    if (form.getValue("saveAsNewVersion") === "fileid") {
                        form.show("fileId");
                    } else {
                        form.hide("fileId");
                    }
                },
                afterrender: () => {
                    if (form.getValue("saveAsNewVersion") === "fileid") {
                        form.show("fileId");
                    } else {
                        form.hide("fileId");
                    }
                }
            }
        });

        form.addField("fileId");

        const classIndexTable = form.addTable({id: "classIndexTable"});
        classIndexTable.addField("documentIndex");
        classIndexTable.addField("documentIndexValue");
    },

    selectAttachToProcess: function (form) {
        const attachToProcessField = this.getAttachToProcessField(form);
        const processId = form.getValue("processId") ?? "";
        const activityId = form.getValue("activityId") ?? "";

        if (processId !== "" && activityId !== "") {
            attachToProcessField.setValue("add_to_activity");
        } else if (processId === "" && activityId === "") {
            attachToProcessField.setValue("none");
        } else if (processId !== "" && activityId === "") {
            attachToProcessField.setValue("add_to_process");
        }
    },

    showAttachToProcessFields: function (form) {
        const attachToProcessField = this.getAttachToProcessField(form);

        switch (attachToProcessField.getValue()) {
            case "none": {
                form.hide("processId");
                form.hide("activityId");

                form.setValue("processId", "");
                form.setValue("activityId", "");
                break;
            }
            case "add_to_process": {
                form.show("processId");
                form.hide("activityId");

                form.setValue("activityId", "");
                break;
            }
            case "add_to_activity": {
                form.show("processId");
                form.show("activityId");
                break;
            }
            default:
                throw new Error("Illegal argument");
        }
    },

    authCustomFields: [
        // Api Key
        {
            id: "api_key_key",
            name: "key",
            fieldLabel: RESTDS.t("restDatasource.apiKey.key"),
            allowBlank: false,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "api_key_value",
            name: "value",
            fieldLabel: RESTDS.t("restDatasource.apiKey.value"),
            allowBlank: false,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "api_key_add_to",
            name: "add_to",
            fieldLabel: RESTDS.t("restDatasource.apiKey.addTo"),
            valueField: "id",
            displayField: "display",
            forceSelection: true,
            store: Ext4.create("Ext4.data.Store", {
                fields: ["id", "display"],
                data: [
                    {id: "header", display: RESTDS.t("restDatasource.apiKey.addTo.header")},
                    {id: "query_params", display: RESTDS.t("restDatasource.apiKey.addTo.queryParams")},
                ],
            }),
            allowBlank: false,
            submitValue: false,
            xtype: "combo",
        },
        // Basic Auth
        {
            id: "basic_auth_username",
            name: "username",
            fieldLabel: RESTDS.t("restDatasource.basic.username"),
            allowBlank: false,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "basic_auth_password",
            name: "password",
            fieldLabel: RESTDS.t("restDatasource.basic.password"),
            inputType: "password",
            allowBlank: false,
            submitValue: false,
            xtype: "textfield",
        },
        // Bearer Token
        {
            id: "bearer_token_token_source",
            name: "tokenSource",
            fieldLabel: RESTDS.t("restDatasource.bearer.tokenSource"),
            valueField: "id",
            displayField: "display",
            forceSelection: true,
            store: Ext4.create("Ext4.data.Store", {
                fields: ["id", "display"],
                data: [
                    {id: "DATASOURCE", display: RESTDS.t("restDatasource.bearer.tokenSource.datasource")},
                    {id: "VALUE", display: RESTDS.t("restDatasource.bearer.tokenSource.value")},
                ],
            }),
            allowBlank: false,
            submitValue: false,
            xtype: "combo",
            listeners: {
                afterrender: function (field) {
                    field.up("window").tokenSourceHandler(field);
                },
                select: function (field) {
                    field.up("window").tokenSourceHandler(field);
                },
            },
        },
        {
            id: "bearer_token_value",
            name: "token",
            fieldLabel: RESTDS.t("restDatasource.bearer.token"),
            allowBlank: true,
            submitValue: false,
            hidden: true,
            xtype: "textfield",
            listeners: {
                afterrender: function (field) {
                    field.up("window").checkTokenSource(field);
                }
            }
        },
        {
            id: "bearer_token_datasource_id",
            name: "datasourceId",
            fieldLabel: RESTDS.t("restDatasource.bearer.datasourceId"),
            allowBlank: true,
            submitValue: false,
            hidden: true,
            xtype: "textfield",
            listeners: {
                afterrender: function (field) {
                    field.up("window").checkTokenSource(field);
                }
            }
        },
        {
            id: "bearer_token_datasource_parameters",
            name: "datasourceParameters",
            fieldLabel: RESTDS.t("restDatasource.bearer.datasourceParameters"),
            allowBlank: true,
            submitValue: false,
            hidden: true,
            xtype: "textfield",
            listeners: {
                afterrender: function (field) {
                    field.up("window").checkTokenSource(field);
                }
            }
        },
        // Cookie
        {
            id: "cookie_http_method",
            name: "http_method",
            fieldLabel: RESTDS.t("restDatasource.cookie.httpMethod"),
            valueField: "id",
            displayField: "display",
            forceSelection: true,
            store: Ext4.create("Ext4.data.Store", {
                fields: ["id", "display"],
                data: [
                    {id: "GET", display: "GET"},
                    {id: "POST", display: "POST"},
                ],
            }),
            allowBlank: false,
            submitValue: false,
            value: "GET",
            xtype: "combo",
            listeners: {
                afterrender: function (field) {
                    field.up("window").httpMethodHandler(field);
                },
                select: function (field) {
                    field.up("window").httpMethodHandler(field);
                },
            },
        },
        {
            id: "cookie_url",
            name: "url",
            fieldLabel: "URL",
            allowBlank: false,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "cookie_content_type",
            name: "content_type",
            fieldLabel: "Content-Type",
            valueField: "id",
            displayField: "display",
            forceSelection: true,
            store: Ext4.create("Ext4.data.Store", {
                fields: ["id", "display"],
                data: [
                    {id: "json", display: "application/json"},
                    {id: "form-urlencoded", display: "application/x-www-form-urlencoded"},
                    {id: "form-data", display: "multipart/form-data"},
                ],
            }),
            hidden: true,
            allowBlank: true,
            submitValue: false,
            xtype: "combo",
        },
        {
            id: "cookie_parameters",
            name: "parameters",
            fieldLabel: RESTDS.t("restDatasource.cookie.parameters"),
            labelWidth: 100,
            hidden: true,
            allowBlank: true,
            submitValue: false,
            xtype: "table",
        },
        {
            id: "cookie_parameters_container",
            name: "parameters",
            fieldLabel: "Parameters container",
            labelWidth: 100,
            hidden: true,
            allowBlank: true,
            submitValue: false,
            listeners: {
                afterrender: function (field) {
                    var paramsField = field.up("form").getForm().getFields().get("cookie_parameters");
                    try {
                        var parameters = JSON.parse(field.getValue());
                        for (var key in parameters) {
                            if (parameters.hasOwnProperty(key)) {
                                var row = paramsField.addRow();
                                setTimeout(function () {
                                    row.items.getAt(1).el.dom.onclick = function () {
                                        var paramsCt = field
                                            .up("form")
                                            .getForm()
                                            .getFields()
                                            .get("cookie_parameters_container");
                                        var keys = paramsField.items.keys
                                            .filter(function (item) {
                                                return item.startsWith("removablefield");
                                            })
                                            .map(function (item) {
                                                return paramsField.items
                                                    .get(item)
                                                    .items.get("fields-form")
                                                    .items.get("key")
                                                    .getValue();
                                            });
                                        var values = paramsField.items.keys
                                            .filter(function (item) {
                                                return item.startsWith("removablefield");
                                            })
                                            .map(function (item) {
                                                return paramsField.items
                                                    .get(item)
                                                    .items.get("fields-form")
                                                    .items.get("value")
                                                    .getValue();
                                            });
                                        var result = {};
                                        for (var i = 0; i < keys.length; i++) {
                                            if (keys[i] !== "") {
                                                result[keys[i]] = values[i];
                                            }
                                        }
                                        paramsCt.setValue(JSON.stringify(result));
                                    };
                                }, 500);
                                row.items.get("fields-form").items.get("key").setValue(key);
                                row.items.get("fields-form").items.get("value").setValue(parameters[key]);
                            }
                        }
                        paramsField.removeRow(1);
                    } catch (err) {
                        console.error(err);
                    }
                },
            },
            xtype: "textfield",
        },
        {
            id: "cookie_header",
            name: "header",
            fieldLabel: "Header",
            hidden: true,
            allowBlank: true,
            submitValue: false,
            xtype: "textfield",
        },
        // OAuth 2.0
        {
            id: "oauth2_grant_type",
            name: "grant_type",
            fieldLabel: "Grant type",
            valueField: "id",
            displayField: "display",
            forceSelection: true,
            store: Ext4.create("Ext4.data.Store", {
                fields: ["id", "display"],
                data: [
                    {id: "client_credentials", display: "Client credentials"},
                    {id: "password_credentials", display: "Password credentials"},
                ],
            }),
            allowBlank: false,
            submitValue: false,
            value: "client_credentials",
            xtype: "combo",
            listeners: {
                afterrender: function (field) {
                    field.up("window").grantTypeHandler(field);
                },
                select: function (field) {
                    field.up("window").grantTypeHandler(field);
                },
            },
        },
        {
            id: "oauth2_access_token_url",
            name: "access_token_url",
            fieldLabel: "Acess Token URL",
            allowBlank: false,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "oauth2_username",
            name: "username",
            fieldLabel: "Username",
            allowBlank: true,
            hidden: true,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "oauth2_password",
            name: "password",
            fieldLabel: "Password",
            allowBlank: true,
            hidden: true,
            submitValue: false,
            inputType: "password",
            xtype: "textfield",
        },
        {
            id: "oauth2_client_id",
            name: "client_id",
            fieldLabel: "Client ID",
            allowBlank: false,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "oauth2_client_secret",
            name: "client_secret",
            fieldLabel: "Client Secret",
            allowBlank: false,
            submitValue: false,
            inputType: "password",
            xtype: "textfield",
        },
        {
            id: "oauth2_scope",
            name: "access_scope",
            fieldLabel: "Scope",
            allowBlank: true,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "oauth2_resource",
            name: "resource",
            fieldLabel: "Resource",
            allowBlank: true,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "oauth2_token",
            name: "access_token",
            fieldLabel: "Token",
            hidden: true,
            allowBlank: true,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "oauth2_client_authentication",
            name: "client_authentication",
            fieldLabel: RESTDS.t("restDatasource.oauth2.clientAuthentication"),
            valueField: "id",
            displayField: "display",
            forceSelection: true,
            store: Ext4.create("Ext4.data.Store", {
                fields: ["id", "display"],
                data: [
                    {id: "send_as_basic", display: RESTDS.t("restDatasource.oauth2.sendAsBasic")},
                    {id: "send_in_body", display: RESTDS.t("restDatasource.oauth2.sendClientCredentials")},
                ],
            }),
            allowBlank: false,
            submitValue: false,
            value: "send_as_basic",
            xtype: "combo",
        },
        {
            id: "oauth2_add_to",
            name: "add_to",
            fieldLabel: RESTDS.t("restDatasource.oauth2.addAuthorizationDataTo"),
            valueField: "id",
            displayField: "display",
            forceSelection: true,
            store: Ext4.create("Ext4.data.Store", {
                fields: ["id", "display"],
                data: [
                    {id: "request_header", display: RESTDS.t("restDatasource.oauth2.requestHeader")},
                    {id: "request_url", display: RESTDS.t("restDatasource.oauth2.requestUrl")},
                ],
            }),
            allowBlank: false,
            submitValue: false,
            value: "request_header",
            xtype: "combo",
        },
        // NTLM
        {
            id: "ntlm_username",
            name: "username",
            fieldLabel: RESTDS.t("restDatasource.ntlm.username"),
            allowBlank: false,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "ntlm_password",
            name: "password",
            fieldLabel: RESTDS.t("restDatasource.ntlm.password"),
            inputType: "password",
            allowBlank: false,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "ntlm_domain",
            name: "domain",
            fieldLabel: RESTDS.t("restDatasource.ntlm.domain"),
            allowBlank: true,
            submitValue: false,
            xtype: "textfield",
        },
        {
            id: "ntlm_workstation",
            name: "workstation",
            fieldLabel: RESTDS.t("restDatasource.ntlm.workstation"),
            allowBlank: true,
            submitValue: false,
            xtype: "textfield",
        },
    ],

    showManageAuthorizationsWindow: function (row, modify) {
        function camelToSnakeCase(str) {
            return str.replace(/[A-Z]/g, function (letter) {
                return '_' + letter.toLowerCase()
            });
        }

        Ext4.create("Ext4.window.Window", {
            title: modify ? RESTDS.t("restDatasource.modifyAuthorizations") : RESTDS.t("restDatasource.createAuthorizations"),
            width: 500,
            modal: true,
            frame: false,
            layout: "fit",
            autoShow: true,

            items: [
                {
                    xtype: "form",
                    layout: "form",
                    frame: false,
                    border: false,
                    bodyPadding: 10,
                    defaults: {
                        labelWidth: 150,
                    },
                    items: [
                        {
                            xtype: "combo",
                            name: "template",
                            fieldLabel: RESTDS.t("restDatasource.template"),
                            queryCaching: false,
                            store: Ext4.create("Ext4.data.Store", {
                                fields: ["id", "name", "description", "authorizationType", "authorizationConfig"],
                                remoteSort: true,
                                pageSize: 20,
                                proxy: {
                                    type: "ajax",
                                    url: PW.getAbsolutePath(
                                        "/plugin/com.suncode.plugin-rest-datasources/authorizations"
                                    ),
                                    reader: {
                                        type: "json",
                                        root: "data",
                                        totalProperty: "total",
                                    },
                                },
                                sorters: [
                                    {
                                        property: "name",
                                        direction: "ASC",
                                    },
                                ],
                            }),
                            pageSize: 20,
                            minChars: 0,
                            queryMode: "remote",
                            displayField: "name",
                            valueField: "id",
                            submitValue: false,
                            padding: "0 0 10 0",
                            listeners: {
                                select: function (combo, record) {
                                    var form = combo.up("form").getForm();
                                    if (!modify) {
                                        record[0].data.id = "";
                                        record[0].data.name = "";
                                    }
                                    var authConfig = {};
                                    var parsedAuthConfig = JSON.parse(record[0].raw.authorizationConfig);

                                    Object.keys(parsedAuthConfig).forEach(function (id) {
                                        var rawValue = parsedAuthConfig[id];
                                        if (record[0].data.authorizationType === 'BEARER_TOKEN' && id === 'token') id = 'value';
                                        if (record[0].data.authorizationType === 'COOKIE' && id === 'parameters') id = 'parameters_container';
                                        authConfig[record[0].data.authorizationType.toLowerCase() + '_' + camelToSnakeCase(id)] = rawValue;
                                    });
                                    combo.up("form").down("#authorizationType").select("");
                                    record[0].data.authorizationConfig = JSON.stringify(authConfig);
                                    form.setValues(record[0].data);
                                },
                            },
                        },
                        {
                            xtype: "box",
                            autoEl: "div",
                            height: 5,
                        },
                        {
                            xtype: "textfield",
                            name: "id",
                            fieldLabel: RESTDS.t("restDatasource.id"),
                            allowBlank: false,
                            hidden: modify,
                            validator: function (value) {
                                if (/[^A-Za-z\d~_.-]/.test(value)) {
                                    return RESTDS.t("restDatasource.invalidId");
                                }
                                return true;
                            },
                        },
                        {
                            xtype: "textfield",
                            name: "name",
                            fieldLabel: RESTDS.t("restDatasource.name"),
                            allowBlank: false,
                        },
                        {
                            xtype: "textfield",
                            name: "description",
                            fieldLabel: RESTDS.t("restDatasource.description"),
                        },
                        {
                            xtype: "combo",
                            id: "authorizationType",
                            hidden: modify,
                            name: "authorizationType",
                            fieldLabel: RESTDS.t("restDatasource.authorizationType"),
                            allowBlank: false,
                            forceSelection: true,
                            store: Ext4.create("Ext4.data.Store", {
                                fields: ["id", "display"],
                                data: [
                                    {id: "API_KEY", display: "API Key"},
                                    {id: "BASIC_AUTH", display: "Basic Auth"},
                                    {id: "BEARER_TOKEN", display: "Bearer Token"},
                                    {id: "COOKIE", display: "Cookie"},
                                    {id: "OAUTH2", display: "OAuth 2.0"},
                                    {id: "NTLM", display: "NTLM"},
                                ],
                            }),
                            queryMode: "local",
                            displayField: "display",
                            valueField: "id",
                            listeners: {
                                change: function (combobox, newValue, oldValue) {
                                    if (newValue === null) return;
                                    var form = combobox.up("form");
                                    form.remove("authCustomFields");
                                    var fieldCt = Ext4.create("Ext4.form.FieldContainer", {
                                        id: "authCustomFields",
                                        layout: "fit",
                                        flex: 1,
                                    });
                                    form.add(fieldCt);
                                    var authorizationConfig = {};
                                    this.authCustomFields
                                        .filter(function (fieldDeclaration) {
                                            return fieldDeclaration.id.startsWith(newValue.toLowerCase());
                                        })
                                        .forEach(
                                            function (fieldDeclaration) {
                                                authorizationConfig[fieldDeclaration.id] = fieldDeclaration.value
                                                    ? fieldDeclaration.value
                                                    : "";
                                                if (fieldDeclaration.listeners) {
                                                    fieldDeclaration.listeners.change = this.customFieldOnChangeHandler;
                                                    fieldDeclaration.listeners.render = this.customFieldOnRenderHandler;
                                                } else {
                                                    fieldDeclaration.listeners = {
                                                        change: this.customFieldOnChangeHandler,
                                                        render: this.customFieldOnRenderHandler,
                                                    };
                                                }
                                                switch (fieldDeclaration.xtype) {
                                                    case "textfield": {
                                                        fieldCt.add(Ext4.create("Ext4.form.TextField", fieldDeclaration));
                                                        break;
                                                    }
                                                    case "combo": {
                                                        fieldCt.add(Ext4.create("Ext4.form.ComboBox", fieldDeclaration));
                                                        break;
                                                    }
                                                    case "table": {
                                                        fieldDeclaration.listeners = {};
                                                        var parametersTable = Ext4.create(
                                                            "Suncode.datasourcemanager.component.GroupedMultiValueField",
                                                            fieldDeclaration
                                                        );
                                                        var parametersForm = fieldCt.add(parametersTable).getParametersForm();
                                                        parametersForm.addParameter({
                                                            id: "key",
                                                            name: RESTDS.t("restDatasource.apiKey.key"),
                                                            submitValue: false,
                                                            listeners: {
                                                                change: function (field) {
                                                                    this.customTableFieldOnChangeHandler(field, parametersTable);
                                                                }.bind(this),
                                                            },
                                                        });
                                                        parametersForm.addParameter({
                                                            id: "value",
                                                            name: RESTDS.t("restDatasource.apiKey.value"),
                                                            submitValue: false,
                                                            listeners: {
                                                                change: function (field) {
                                                                    this.customTableFieldOnChangeHandler(field, parametersTable);
                                                                }.bind(this),
                                                            },
                                                        });
                                                        parametersForm.setDisabled(true);
                                                        break;
                                                    }
                                                    default: {
                                                        fieldCt.add(Ext4.create("Ext4.form.field.Base", fieldDeclaration));
                                                    }
                                                }
                                            }.bind(this)
                                        );
                                    form
                                        .getForm()
                                        .getFields()
                                        .get("authorizationConfig")
                                        .setValue(JSON.stringify(authorizationConfig));
                                }.bind(this),
                            },
                        },
                        {
                            xtype: "textfield",
                            id: "authorizationConfig",
                            name: "authorizationConfig",
                            fieldLabel: RESTDS.t("restDatasource.authorizationConfig"),
                            allowBlank: false,
                            hidden: true,
                        },
                        {
                            xtype: "box",
                            autoEl: "div",
                            height: 15,
                        },
                    ],
                },
            ],

            buttons: [
                {
                    text: RESTDS.t("restDatasource.createAuthorizations"),
                    handler: function (button) {
                        var win = button.up("window");
                        win.createOrModifyAuthorization(win, row);
                    },
                    hidden: modify,
                },
                {
                    text: RESTDS.t("restDatasource.modifyAuthorizations"),
                    handler: function (button) {
                        var win = button.up("window"),
                            connectionId = win.down("[name=id]").getValue();

                        win.createOrModifyAuthorization(win, row, connectionId);
                    },
                    hidden: !modify,
                },
                {
                    text: RESTDS.t("restDatasource.deleteAuthorizations"),
                    handler: function (button) {
                        Ext4.MessageBox.confirm(
                            RESTDS.t("restDatasource.deleteAuthorizationTitle"),
                            RESTDS.t("restDatasource.deleteAuthorizationConfirm"),
                            function (btn) {
                                if (btn === "yes") {
                                    var win = button.up("window"),
                                        connectionId = win.down("[name=id]").getValue();
                                    win.deleteAuthorization(win, row, connectionId);
                                }
                            }
                        );
                    },
                    hidden: !modify,
                },
                {
                    text: PW.t("cancel"),
                    handler: function (button) {
                        button.up("window").close();
                    },
                },
            ],
            createOrModifyAuthorization: function (win, row, connectionId) {
                var url = "/plugin/com.suncode.plugin-rest-datasources/authorizations";
                if (connectionId) {
                    url += "/" + connectionId;
                }
                var form = this.down("form");
                var values = form.getValues();
                delete values["key"];
                delete values["value"];
                if (form.isValid()) {
                    form.mask();
                    Ext4.Ajax.request({
                        url: PW.getAbsolutePath(url),
                        method: "POST",
                        jsonData: Ext4.encode(values),
                        failure: function () {
                            PW.ui.Message.error(RESTDS.t("restDatasource.unknownException"));
                            form.unmask();
                        },
                        success: function (resp) {
                            var response = Ext4.decode(resp.responseText);
                            if (response.success) {
                                PW.ui.Message.success(response.message);
                                win.close();
                            } else {
                                PW.ui.Message.error(response.message);
                                form.unmask();
                            }
                        },
                    });
                }
            },
            deleteAuthorization: function (win, row, authorizationId) {
                var url = "/plugin/com.suncode.plugin-rest-datasources/authorizations/" + authorizationId;
                var form = this.down("form");
                if (form.isValid()) {
                    form.mask();
                    Ext4.Ajax.request({
                        url: PW.getAbsolutePath(url),
                        method: "DELETE",
                        failure: function () {
                            PW.ui.Message.error(RESTDS.t("restDatasource.unknownException"));
                            form.unmask();
                        },
                        success: function (resp) {
                            var response = Ext4.decode(resp.responseText);
                            if (response.success) {
                                PW.ui.Message.success(response.message);
                                win.close();
                            } else {
                                PW.ui.Message.error(response.message);
                                form.unmask();
                            }
                        },
                    });
                }
            },
            checkTokenSource: function (field) {
                const fields = field.up("form").getForm().getFields();
                const tokenSourceField = fields.get("bearer_token_token_source");

                if (tokenSourceField && !tokenSourceField.getValue()) {
                    field.hide()
                }
            },
            tokenSourceHandler: function (field) {
                const fields = field.up("form").getForm().getFields();
                const tokenValueField = fields.get("bearer_token_value");
                const datasourceIdField = fields.get("bearer_token_datasource_id");
                const datasourceParametersField = fields.get("bearer_token_datasource_parameters");

                if (field.getValue() && tokenValueField && datasourceIdField && datasourceParametersField) {
                    if (field.getValue() === "DATASOURCE") {
                        tokenValueField.allowBlank = true;
                        tokenValueField.hide();
                        datasourceIdField.allowBlank = false;
                        datasourceIdField.show();
                        datasourceParametersField.show();
                    } else if (field.getValue() === "VALUE") {
                        tokenValueField.allowBlank = false;
                        tokenValueField.show();
                        datasourceIdField.allowBlank = true;
                        datasourceIdField.hide();
                        datasourceParametersField.hide();
                    }
                }
            },
            grantTypeHandler: function (field) {
                var fields = field.up("form").getForm().getFields();
                var username = fields.get("oauth2_username");
                var password = fields.get("oauth2_password");
                if (username && password) {
                    if (field.getValue() === "client_credentials") {
                        username.allowBlank = password.allowBlank = true;
                        username.hide();
                        password.hide();
                        username.setValue("");
                        password.setValue("");
                    } else {
                        username.allowBlank = password.allowBlank = false;
                        username.show();
                        password.show();
                    }
                }
            },
            httpMethodHandler: function (field) {
                var fields = field.up("form").getForm().getFields();
                var contentType = fields.get("cookie_content_type");
                var parameters = fields.get("cookie_parameters");
                if (contentType && parameters) {
                    if (field.getValue() === "GET") {
                        contentType.allowBlank = parameters.allowBlank = true;
                        contentType.hide();
                        parameters.hide();
                        contentType.setValue("json");
                        parameters.setValue("");
                    } else {
                        contentType.allowBlank = parameters.allowBlank = false;
                        contentType.show();
                        parameters.show();
                    }
                }
            },
        });
    },

    customFieldOnChangeHandler: function (field, newValue, _oldValue) {
        var form = field.up("form").getForm();
        var authorizationConfigObject = JSON.parse(form.getValues().authorizationConfig);
        authorizationConfigObject[field.id] = newValue;
        form.getFields().get("authorizationConfig").setValue(JSON.stringify(authorizationConfigObject));
    },

    customFieldOnRenderHandler: function (field) {
        try {
            var authConfig = JSON.parse(
                field.up("form").getForm().getFields().get("authorizationConfig").getValue()
            );
            if (authConfig[field.id]) field.setValue(authConfig[field.id]);
        } catch (err) {
        }
    },

    customTableFieldOnChangeHandler: function (field, parametersTable) {
        var paramsCt = field.up("form").getForm().getFields().get("cookie_parameters_container");
        var keys = parametersTable.items.keys
            .filter(function (item) {
                return item.startsWith("removablefield");
            })
            .map(function (item) {
                return parametersTable.items.get(item).items.get("fields-form").items.get("key").getValue();
            });
        var values = parametersTable.items.keys
            .filter(function (item) {
                return item.startsWith("removablefield");
            })
            .map(function (item) {
                return parametersTable.items
                    .get(item)
                    .items.get("fields-form")
                    .items.get("value")
                    .getValue();
            });
        var result = {};
        for (var i = 0; i < keys.length; i++) {
            if (keys[i] !== "") {
                result[keys[i]] = values[i];
            }
        }
        paramsCt.setValue(JSON.stringify(result));
    },
});