function writeXpdl( packageNode, processDefId ) {
    return writeXpdlPackage( packageNode, processDefId );
}

function writeXpdlPackage( packageNode, processDefId ) {
    var mainPanel = Ext.getCmp( 'main_panel' );
    var xpdl = new Object();

    var scrpt = new Object( {
        Type: packageNode.attributes.scriptType
    } );

    if ( !Ext.isEmpty( packageNode.attributes.scriptGrammar ) ) {
        scrpt = Ext.apply( {
            Grammar: packageNode.attributes.scriptGrammar
        }, scrpt );
    }
    if ( !Ext.isEmpty( packageNode.attributes.scriptVersion ) ) {
        scrpt = Ext.apply( {
            Version: packageNode.attributes.scriptVersion
        }, scrpt );
    }

    var pack = new Array();
    pack.push( {
        'xmlns:xpdl': 'http://www.wfmc.org/2002/XPDL1.0'
    } );
    pack.push( {
        xmlns: 'http://www.wfmc.org/2002/XPDL1.0'
    } );
    pack.push( {
        'xmlns:xsi': 'http://www.w3.org/2001/XMLSchema-instance'
    } );
    if ( Ext.isEmpty( processDefId ) ) {
    	pack.push( {
            Id: packageNode.attributes.packageId
        } );
        pack.push( {
            Name: packageNode.attributes.packageName
        } );
    } else {
    	pack.push( {
            Id: generateId( packageNode.attributes.packageId + '_' + processDefId, Ext.ux.suncode.Constants.PACKAGE_ID_MAX_LENGTH )
        } );
    	
    	var packageName = packageNode.attributes.packageName;
    	if ( packageNode.hasChildNodes() ) {
    		packageNode.eachChild( function( processNode ) {
    	    	if ( processDefId === processNode.attributes.processDefId ) {
    	    		packageName += ' - ' + processNode.attributes.processName;
    	    	}
    	    } );
        }
    	
        pack.push( {
            Name: packageName
        } );
    }
    pack.push( {
        'xsi:schemaLocation': 'http://www.wfmc.org/2002/XPDL1.0 http://wfmc.org/standards/docs/TC-1025_schema_10_xpdl.xsd'
    } );

    var packageHeader = new Array();
    packageHeader.push( {
        XPDLVersion: {
            'XPDLNODEVALUE': '1.0'
        }
    } );
    packageHeader.push( {
        Vendor: {
            'XPDLNODEVALUE': 'Suncode'
        }
    } );
    packageHeader.push( {
        Created: {
            'XPDLNODEVALUE': packageNode.attributes.packageCreationTime
        }
    } );

    if ( !Ext.isEmpty( packageNode.attributes.packageDescr ) ) {
        packageHeader.push( {
            Description: {
                'XPDLNODEVALUE': packageNode.attributes.packageDescr
            }
        } );
    }

    pack.push( {
        PackageHeader: packageHeader
    } );

    if ( !Ext.isEmpty( packageNode.attributes.packageAuthor ) ) {
        pack.push( {
            RedefinableHeader: [ {
                Author: {
                    'XPDLNODEVALUE': packageNode.attributes.packageAuthor
                }
            } ]
        } );
    }

    pack.push( {
        Script: scrpt
    } );

    if ( !Ext.isEmpty( packageNode.attributes.participants ) ) {
        pack.push( {
            Participants: xpdlPackageParticipants( packageNode )
        } );
    }

    if ( !Ext.isEmpty( packageNode.attributes.applications ) ) {
        pack.push( {
            Applications: xpdlPackageApplications( packageNode )
        } );
    }

    if ( packageNode.hasChildNodes() ) {
        pack.push( {
            WorkflowProcesses: xpdlWorkflowProcesses( packageNode, processDefId )
        } );
    }

    pack.push( {
        ExtendedAttributes: [ {
            ExtendedAttribute: {
                Name: 'JaWE_CONFIGURATION',
                Value: 'default'
            }
        }, {
            ExtendedAttribute: {
                Name: 'EDITING_TOOL',
                Value: 'Plus Workflow Editor'
            }
        }, {
            ExtendedAttribute: {
                Name: 'EDITING_TOOL_VERSION',
                Value: '2.2'
            }
        }, {
          ExtendedAttribute: {
            Name: 'EVENT_ACTION_ID_COUNTER',
            Value: packageNode.attributes.eventActionIdCounter
          }
        }, {
          ExtendedAttribute: {
            Name: 'PWE_VERSION',
            Value: mainPanel.isCurrentXpdlVersionOlder() ? mainPanel.getCurrentXpdlVersion() : mainPanel.getVersion()
          }
        } ]
    } );

    xpdl = Ext.apply( {
        Package: pack
    }, xpdl );

    return xpdl;
}

function xpdlPackageParticipants( packageNode ) {
    var participants = new Array();

    var ps = packageNode.attributes.participants;
    for ( var i = 0; i < ps.length; i++ ) {
        participants.push( {
            Participant: xpdlPackageParticipant( ps[i] )
        } );
    }

    return participants;
}

function xpdlPackageParticipant( p ) {
    var participant = new Array();

    participant.push( {
        Id: p.roleId
    } );
    participant.push( {
        Name: p.roleName
    } );
    participant.push( {
        ParticipantType: {
            Type: p.roleType
        }
    } );

    if ( !Ext.isEmpty( p.roleDescr ) ) {
        participant.push( {
            Description: {
                'XPDLNODEVALUE': p.roleDescr
            }
        } );
    }

    return participant;
}

function xpdlPackageApplications( packageNode ) {
    var applications = new Array();

    var as = packageNode.attributes.applications;
    for ( var i = 0; i < as.length; i++ ) {
        applications.push( {
            Application: xpdlApplication( as[i] )
        } );
    }

    return applications;
}

function xpdlWorkflowProcesses( packageNode, processDefId ) {
    var processes = new Array();

    packageNode.eachChild( function( processNode ) {
    	if ( Ext.isEmpty( processDefId ) || processDefId === processNode.attributes.processDefId ) {
    		processes.push( {
                WorkflowProcess: xpdlWorkflowProcess( processNode )
            } );
    	}
    } );

    return processes;
}

function xpdlWorkflowProcess( processNode ) {
    var process = new Array();

    process.push( {
        Id: processNode.attributes.processDefId
    } );
    process.push( {
        Name: processNode.attributes.processName
    } );

    var processHeader = new Array();
    processHeader.push( {
        Created: {
            'XPDLNODEVALUE': processNode.attributes.processCreationTime
        }
    } );

    if ( !Ext.isEmpty( processNode.attributes.processDescr ) ) {
        processHeader.push( {
            Description: {
                'XPDLNODEVALUE': processNode.attributes.processDescr
            }
        } );
    }

    process.push( {
        ProcessHeader: processHeader
    } );

    process.push( {
        FormalParameters: xpdlFormalParameters( processNode )
    } );

    var dataFields = xpdlDataFields( processNode );
    if ( dataFields.length > 0 ) {
        process.push( {
            DataFields: dataFields
        } );
    }

    var participants = processNode.attributes.participants;
    var participantsOrder = '';
    if ( !Ext.isEmpty( participants ) ) {
        var processParticipants = xpdlParticipants( processNode );

        if ( !Ext.isEmpty( processParticipants ) ) {
            process.push( {
                Participants: processParticipants
            } );
        }

        for ( var i = 0; i < participants.length; i++ ) {
            if ( i == participants.length - 1 ) {
                participantsOrder += participants[i].roleId;
            } else {
                participantsOrder += participants[i].roleId + ';';
            }
        };
    }

    var applications = xpdlApplications( processNode );
    if ( applications.length > 0 ) {
        process.push( {
            Applications: applications
        } );
    }

    var activities = xpdlActivities( processNode );
    if ( activities.length > 0 ) {
        process.push( {
            Activities: activities
        } );
    }

    var transitions = xpdlTransitions( processNode );
    if ( transitions.length > 0 ) {
        process.push( {
            Transitions: transitions
        } );
    }

    var extAttrs = new Array();
    if ( Ext.isEmpty( participantsOrder ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'JaWE_GRAPH_WORKFLOW_PARTICIPANT_ORDER'
            }
        } );
    } else {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'JaWE_GRAPH_WORKFLOW_PARTICIPANT_ORDER',
                Value: participantsOrder
            }
        } );
    }

    var starts = processNode.attributes.starts;
    for ( var i = 0; i < starts.length; i++ ) {
        var s = starts[i];

        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'JaWE_GRAPH_START_OF_WORKFLOW',
                Value: 'JaWE_GRAPH_PARTICIPANT_ID=' + s.roleId + ',CONNECTING_ACTIVITY_ID=' + s.connectingActivity + ',X_OFFSET=' + s.xOffset
                                + ',Y_OFFSET=' + s.yOffset + ',JaWE_GRAPH_TRANSITION_STYLE=NO_ROUTING_ORTHOGONAL,TYPE=END_DEFAULT'
            }
        } );
    }

    var stops = processNode.attributes.stops;
    for ( var i = 0; i < stops.length; i++ ) {
        var s = stops[i];

        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'JaWE_GRAPH_END_OF_WORKFLOW',
                Value: 'JaWE_GRAPH_PARTICIPANT_ID=' + s.roleId + ',CONNECTING_ACTIVITY_ID=' + s.connectingActivity + ',X_OFFSET=' + s.xOffset
                                + ',Y_OFFSET=' + s.yOffset + ',JaWE_GRAPH_TRANSITION_STYLE=NO_ROUTING_ORTHOGONAL,TYPE=END_DEFAULT'
            }
        } );
    }

    if ( !Ext.isEmpty( processNode.attributes.processNameMask ) ) {
        switch ( processNode.attributes.processNameMask ) {
            case 'VER':
                extAttrs.push( {
                    ExtendedAttribute: {
                        Name: 'PROCESS_NAME_MASK',
                        Value: processNode.attributes.processName + '_ VER/#####'
                    }
                } );
                break;
            case 'NAME':
                extAttrs.push( {
                    ExtendedAttribute: {
                        Name: 'PROCESS_NAME_MASK',
                        Value: processNode.attributes.processName
                    }
                } );
                break;
            default:
                extAttrs.push( {
                    ExtendedAttribute: {
                        Name: 'PROCESS_NAME_MASK',
                        Value: processNode.attributes.processNameMask
                    }
                } );
                break;
        }
    }

    if ( !Ext.isEmpty( processNode.attributes.processColor ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'PROCESS_COLOR',
                Value: processNode.attributes.processColor
            }
        } );
    }

    if ( !Ext.isEmpty( participants ) ) {
        extAttrs = xpdlProcessGlobalParticipants( processNode, extAttrs );
    }
    
    if ( !Ext.isEmpty( processNode.attributes.templates.validators ) ) {
    	extAttrs.push( {
            ExtendedAttribute: {
                Name: 'VALIDATOR_TEMPLATES',
                Value: Ext.util.JSON.encode( processNode.attributes.templates.validators )
            }
        } );
    }
    
    if ( !Ext.isEmpty( processNode.attributes.templates.variablesSetters ) ) {
    	extAttrs.push( {
            ExtendedAttribute: {
                Name: 'VARIABLES_SETTER_TEMPLATES',
                Value: Ext.util.JSON.encode( processNode.attributes.templates.variablesSetters )
            }
        } );
    }
    
    if ( !Ext.isEmpty( processNode.attributes.labels ) ) {
    	var labels = processNode.attributes.labels;
        for ( var i = 0; i < labels.length; i++ ) {
            var label = labels[i];
            var labelDef = 'id=' + label.id;
            labelDef += '\nname=' + label.name;
            labelDef += '\nformActions=' + Ext.util.JSON.encode( label.formActions );

            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'LABEL',
                    Value: labelDef
                }
            } );
        }
    }

    extAttrs.push( {
      ExtendedAttribute: {
        Name: 'TABLES',
        Value: Ext.util.JSON.encode( processNode.attributes.tables )
      }
    } );

    extAttrs.push( {
      ExtendedAttribute: {
        Name: 'ATTACHMENT_DIRECTORY',
        Value: processNode.attributes.attachmentDirectory
      }
    } );
    
    process.push( {
        ExtendedAttributes: extAttrs
    } );

    return process;
}

function xpdlFormalParameters( processNode ) {
    var formalParams = new Array();

    var fps = processNode.attributes.formalParameters;
    for ( var i = 0; i < fps.length; i++ ) {
        formalParams.push( {
            FormalParameter: xpdlFormalParameter( fps[i] )
        } );
    }

    return formalParams;
}

function xpdlFormalParameter( fp ) {
    var formalParam = new Array();

    formalParam.push( {
        Id: fp.id
    } );
    formalParam.push( {
        Mode: fp.mode
    } );
    formalParam.push( {
        DataType: {
            BasicType: {
                Type: fp.type
            }
        }
    } );

    if ( !Ext.isEmpty( fp.descr ) ) {
        formalParam.push( {
            Description: {
                'XPDLNODEVALUE': fp.descr
            }
        } );
    }

    return formalParam;
}

function xpdlDataFields( processNode ) {
    var dataFields = new Array();

    var dfs = processNode.attributes.variables;
    for ( var i = 0; i < dfs.length; i++ ) {
        dataFields.push( {
            DataField: xpdlDataField( dfs[i] )
        } );
    }

    return dataFields;
}

function xpdlDataField( df ) {
    var dataField = new Array();

    dataField.push( {
        Id: df.id
    } );
    dataField.push( {
        IsArray: 'FALSE'
    } );
    dataField.push( {
        Name: df.name
    } );

    var extAttrs = new Array();

    if ( df.type == 'DATE' || df.type == 'RADIOBUTTON' || df.type == 'CHECKBOX' || df.type == 'LISTBOX'
                    || df.type == 'LISTBOX_NO_FILTER' || df.type == 'LISTBOX_EDIT' || df.type == 'LISTBOX_EDIT_NO_FILTER'
                    || df.type == 'LISTBOX_NO_LAZY' || df.type == 'LISTBOX_NO_FILTER_NO_LAZY' || df.type == 'USERLIST'
                    || df.type == 'USERLISTLISTBOX' || df.type == 'ROLEUSERS' || df.type == 'ROLEUSERS_NO_FILTER' || df.type == 'TEXTAREA' ) {
        dataField.push( {
            DataType: {
                BasicType: {
                    Type: 'STRING'
                }
            }
        } );

        if ( df.type == 'ROLEUSERS' || df.type == 'ROLEUSERS_NO_FILTER' ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'SUBTYPE',
                    Value: df.type.replace( 'ROLEUSERS', 'LISTBOX' )
                }
            } );

            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'VALUES',
                    Value: 'ROLEUSERS'
                }
            } );
        } else if ( df.type == 'USERLISTLISTBOX' ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'SUBTYPE',
                    Value: 'LISTBOX'
                }
            } );

            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'VALUES',
                    Value: 'USERLIST'
                }
            } );
        } else {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'SUBTYPE',
                    Value: df.type
                }
            } );

            if ( !( df.type == 'TEXTAREA' || df.type == 'DATE' || df.type == 'USERLIST' ) ) {
                if ( !Ext.isEmpty( df.valuesList ) ) {
                    extAttrs.push( {
                        ExtendedAttribute: {
                            Name: 'VALUES',
                            Value: xpdlValues( df.valuesList )
                        }
                    } );
                }
            } else if ( df.type == 'TEXTAREA' ) {
              extAttrs.push( {
                ExtendedAttribute: {
                  Name: 'NOT_REMEMBER_SIZE',
                  Value: df.rememberSize ? 'FALSE' : 'TRUE'
                }
              } );
            }
        }
    } else if ( df.type == 'DATA_CHOOSER' ) {
        dataField.push( {
            DataType: {
                BasicType: {
                    Type: 'STRING'
                }
            }
        } );
        if ( df.fieldType == 'textarea' ) {
        	extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'SUBTYPE',
                    Value: 'TEXTAREA'
                }
            } );
        }
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DATA_CHOOSER',
                Value: xpdlDataChooser( df )
            }
        } );
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DATA_CHOOSER_DEFINITION',
                Value: Ext.util.JSON.encode( df.definition )
            }
        } );
    } else if ( df.type == 'INTEGER' || df.type == 'FLOAT' || df.type == 'AMOUNT' ) {
    	var xpdlType = ( df.type == 'AMOUNT' ) ? 'FLOAT' : df.type
    	var basicType = '';
    	
    	if ( df.placement === 'table' ) {
    		basicType = 'STRING';
    		
    		extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'SUBTYPE',
                    Value: xpdlType
                }
            } );
    	} else {
    		basicType = xpdlType;
    	}
    	
    	dataField.push( {
            DataType: {
                BasicType: {
                    Type: basicType
                }
            }
        } );
    } else {
        dataField.push( {
            DataType: {
                BasicType: {
                    Type: df.type
                }
            }
        } );
    }

    if ( !Ext.isEmpty( df.placement ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'ARRAY',
                Value: df.placement == 'table' ? 'TRUE' : 'FALSE'
            }
        } );
    }
    if ( !Ext.isEmpty( df.dtFormat ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DT_FORMAT',
                Value: df.dtFormat
            }
        } );
    }
    if ( !Ext.isEmpty( df.format ) ) {
      var formatExtAttrName = '';

      if ( df.placement === 'table' ) {
        formatExtAttrName = 'DT_FORMAT';
      } else {
        formatExtAttrName = 'FORMAT';
      }

      extAttrs.push( {
        ExtendedAttribute: {
          Name: formatExtAttrName,
          Value: df.format
        }
      } );
    }
    if ( !Ext.isEmpty( df.decimalPrecision ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DECIMAL_PRECISION',
                Value: df.decimalPrecision
            }
        } );
    }
    if ( !Ext.isEmpty( df.initialValue ) ) {
        var initialValue = df.initialValue.toString();
        while( initialValue.startsWith( '\n' ) ) {
          initialValue = initialValue.substring( 1 );
        }

        dataField.push( {
            InitialValue: {
                'XPDLNODEVALUE': initialValue
            }
        } );
    }
    if ( !Ext.isEmpty( df.descr ) ) {
        dataField.push( {
            Description: {
                'XPDLNODEVALUE': df.descr
            }
        } );
    }
    if ( !Ext.isEmpty( df.thousandSeparator, true ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'THOUSAND_SEPARATOR',
                Value: df.thousandSeparator
            }
        } );
    }
    if ( !Ext.isEmpty( df.regex ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'REG_EXPRESSION',
                Value: df.regex
            }
        } );
    }
    if ( !Ext.isEmpty( df.exType ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'EXTYPE',
                Value: df.exType
            }
        } );
    }
    if ( df.requirement ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'REQUIREMENT',
                Value: 'YES'
            }
        } );
    }
    if ( df.readonly ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'READONLY',
                Value: 'TRUE'
            }
        } );
    }
    if ( df.dtHidden ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DT_HIDDEN',
                Value: 'TRUE'
            }
        } );
    }
    if ( df.dtSize > 0 ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'SIZE',
                Value: df.dtSize
            }
        } );
    }
    if ( df.dtMinSize > 0 ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'MIN_SIZE',
                Value: df.dtMinSize
            }
        } );
    }
    if ( !df.dtMenuDisabled ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DT_MENU_DISABLED',
                Value: df.dtMenuDisabled.toString().toUpperCase()
            }
        } );
    }
    if ( !Ext.isEmpty( df.dtAlign ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DT_ALIGN',
                Value: df.dtAlign
            }
        } );
    }
    if ( !Ext.isEmpty( df.dtDecimalSeparator ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DT_DECIMAL_SEPARATOR',
                Value: df.dtDecimalSeparator
            }
        } );
    }
    if ( !df.dtAllowDecimals ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DT_ALLOW_DECIMALS',
                Value: df.dtAllowDecimals.toString().toUpperCase()
            }
        } );
    }
    if ( df.dtSortable ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DT_SORTABLE',
                Value: df.dtSortable.toString().toUpperCase()
            }
        } );
    }
    if ( !df.dtGroupable ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DT_GROUPABLE',
                Value: df.dtGroupable.toString().toUpperCase()
            }
        } );
    }
    if ( !Ext.isEmpty( df.dcAlign ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'DC_ALIGN',
                Value: df.dcAlign
            }
        } );
    }
    if ( !Ext.isEmpty( df.formActions ) ) {
      extAttrs.push( {
        ExtendedAttribute: {
          Name: 'FORM_ACTIONS',
          Value: Ext.util.JSON.encode( df.formActions )
        }
      } );
    }
    if ( !Ext.isEmpty( df.eventActions ) ) {
      extAttrs.push( {
        ExtendedAttribute: {
          Name: 'EVENT_ACTIONS',
          Value: Ext.util.JSON.encode( df.eventActions )
        }
      } );
    }

    extAttrs.push( {
      ExtendedAttribute: {
        Name: 'CREATION_DATE',
        Value: df.creationDate
      }
    } );
    extAttrs.push( {
      ExtendedAttribute: {
        Name: 'MODIFICATION_DATE',
        Value: df.modificationDate
      }
    } );

    if ( extAttrs.length > 0 ) {
        dataField.push( {
            ExtendedAttributes: extAttrs
        } );
    }

    return dataField;
}

function xpdlParticipants( processNode ) {
    var participants = new Array();

    var ps = processNode.attributes.participants;
    for ( var i = 0; i < ps.length; i++ ) {
        if ( !ps[i].isGlobal ) {
            participants.push( {
                Participant: xpdlParticipant( ps[i] )
            } );
        }
    }

    return participants;
}

function xpdlParticipant( p ) {
    var participant = new Array();

    participant.push( {
        Id: p.roleId
    } );
    participant.push( {
        Name: p.roleName
    } );
    participant.push( {
        ParticipantType: {
            Type: p.roleType
        }
    } );

    if ( !Ext.isEmpty( p.roleDescr ) ) {
        participant.push( {
            Description: {
                'XPDLNODEVALUE': p.roleDescr
            }
        } );
    }

    var extAttrs = new Array();
    extAttrs.push( {
        ExtendedAttribute: {
            Name: 'WIDTH',
            Value: p.roleWidth
        }
    } );
    extAttrs.push( {
        ExtendedAttribute: {
            Name: 'HEIGHT',
            Value: p.roleHeight
        }
    } );
    participant.push( {
        ExtendedAttributes: extAttrs
    } );

    return participant;
}

function xpdlProcessGlobalParticipants( processNode, extAttrs ) {
    var ps = processNode.attributes.participants;
    for ( var i = 0; i < ps.length; i++ ) {
        var p = ps[i];

        if ( p.isGlobal ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'GLOBAL_PARTICIPANT',
                    Value: 'GLOBAL_PARTICIPANT_ID=' + p.roleId + ',WIDTH=' + p.roleWidth + ',HEIGHT=' + p.roleHeight
                }
            } );
        }
    }

    return extAttrs;
}

function xpdlApplications( processNode ) {
    var applications = new Array();

    var apps = processNode.attributes.applications;
    for ( var i = 0; i < apps.length; i++ ) {
        applications.push( {
            Application: xpdlApplication( apps[i] )
        } );
    }

    return applications;
}

function xpdlApplication( app ) {
    var application = new Array();

    application.push( {
        Id: app.id
    } );
    application.push( {
        Name: app.name
    } );

    if ( !Ext.isEmpty( app.descr ) ) {
        application.push( {
            Description: {
                'XPDLNODEVALUE': app.descr
            }
        } );
    }

    var fps = new Array();

    var params = app.parameters;
    for ( var i = 0; i < params.length; i++ ) {
        var param = params[i];
        var fp = new Array();

        fp.push( {
            Id: param.id
        } );
        fp.push( {
            Mode: param.mode
        } );
        fp.push( {
            DataType: {
                BasicType: {
                    Type: param.type
                }
            }
        } );

        if ( !Ext.isEmpty( param.descr ) ) {
            fp.push( {
                Description: {
                    'XPDLNODEVALUE': param.descr
                }
            } );
        }

        fps.push( {
            FormalParameter: fp
        } );
    }

    if ( !Ext.isEmpty( fps ) ) {
    	application.push( {
            FormalParameters: fps
        } );
    }

    var extAttrs = new Array();
    extAttrs.push( {
        ExtendedAttribute: [ {
            Name: 'ToolAgentClass'
        }, {
            Value: app.appType
        } ]
    } );

    if ( app.appType == 'org.enhydra.shark.toolagent.JavaClassToolAgent' || app.appType == 'org.enhydra.shark.toolagent.JavaClassTransToolAgent'
                    || app.appType == 'org.enhydra.shark.toolagent.SOAPToolAgent' ) {
        extAttrs.push( {
            ExtendedAttribute: [ {
                Name: 'AppName'
            }, {
                Value: app.realization.agent
            } ]
        } );

        var parameters = new Array();
        for ( var i = 0; i < params.length; i++ ) {
            parameters.push( 'AppParameter arg' + i );
        }

        Ext.getCmp( 'main_panel' ).addJavaClass( {
            sourceCodeType: 'JAVA_APPLICATION',
            qualifiedName: app.realization.agent,
            options: {
                parameters: parameters
            }
        } );
    } else if ( app.appType == 'org.enhydra.shark.toolagent.JavaScriptToolAgent' ) {
        extAttrs.push( {
            ExtendedAttribute: [ {
                Name: 'Script'
            }, {
                Value: app.realization.agent
            } ]
        } );
    } else if ( app.appType == 'org.enhydra.shark.toolagent.RuntimeApplicationToolAgent' ) {
        extAttrs.push( {
            ExtendedAttribute: [ {
                Name: 'AppMode'
            }, {
                Value: app.realization.mode
            } ]
        } );
        extAttrs.push( {
            ExtendedAttribute: [ {
                Name: 'AppName'
            }, {
                Value: app.realization.agent
            } ]
        } );
    } else {
    	extAttrs.push( {
            ExtendedAttribute: [ {
                Name: 'AppName'
            }, {
                Value: app.realization.agent
            } ]
        } );
    }

    if ( extAttrs.length > 0 ) {
        application.push( {
            ExtendedAttributes: extAttrs
        } );
    }

    return application;
}

function xpdlValues( valuesList ) {
    var valueProperty = '';

    for ( var i = 0; i < valuesList.length; i++ ) {
        if ( i == valuesList.length - 1 ) {
            valueProperty += valuesList[i].value;
        } else {
            valueProperty += valuesList[i].value + '\n';
        }
    }

    return valueProperty;
}

function xpdlDataChooser( df ) {
    Ext.getCmp( 'main_panel' ).addJavaClass( {
        sourceCodeType: 'JAVA_DATA_CHOOSER',
        qualifiedName: df.classpath,
        options: new Object()
    } );

    var dcDefString = 'class=' + df.classpath + '\nformCriteria=';

    var crits = df.formCriteria;
    for ( var i = 0; i < crits.length; i++ ) {
        if ( i == crits.length - 1 ) {
            dcDefString += crits[i].varId;
        } else {
            dcDefString += crits[i].varId + ';';
        }
    }

    var mappings = df.mappings;
    for ( var i = 0; i < mappings.length; i++ ) {
        var mapping = mappings[i];
        dcDefString += '\nmapping' + ( i + 1 ) + '=' + mapping.id + ';' + mapping.varId + ';' + mapping.display;

        if ( mapping.verify ) {
            dcDefString += ';check';
        }
    }

    dcDefString += '\ntype=' + df.sort;

    if ( df.comboForceSelection ) {
        dcDefString += '\ncomboForceSelection=true';
    }
    if ( df.comboFilteringDisabled ) {
        dcDefString += '\ncomboFilteringDisabled=true';
    }
    if ( df.comboNoLazyInit ) {
        dcDefString += '\ncomboLazyInit=false';
    }
    if ( !Ext.isEmpty( df.comboQueryDelay ) ) {
        dcDefString += '\ncomboQueryDelay=' + df.comboQueryDelay;
    }

    var keys = df.customKeys;
    for ( var i = 0; i < keys.length; i++ ) {
        var key = keys[i];
        dcDefString += '\n' + key.id + '=' + key.value;
    }

    return dcDefString;
}

function xpdlActivities( processNode ) {
    var activities = new Array();

    processNode.eachChild( function( activityNode ) {
        activities.push( {
            Activity: xpdlActivity( activityNode.hasXpdlSubstitution() ? activityNode.getXpdlSubstitution() : activityNode )
        } );
    } );

    return activities;
}

function xpdlActivity( activityNode ) {
    var activity = new Array();
    var configuration = new Array();
    var isRoute = false;
    var isTool = false;
    var isSubflow = false;

    activity.push( {
        Id: activityNode.attributes.activityDefId
    } );
    activity.push( {
        Name: activityNode.attributes.activityName
    } );

    if ( !Ext.isEmpty( activityNode.attributes.activityDescr ) ) {
        activity.push( {
            Description: {
                'XPDLNODEVALUE': activityNode.attributes.activityDescr
            }
        } );
    }

    if ( activityNode.attributes.activityType == Ext.ux.suncode.Constants.ROUTE ) {
        isRoute = true;
        activity.push( {
            Route: new Array()
        } );
    }
    if ( activityNode.attributes.activityType == Ext.ux.suncode.Constants.SUBFLOW ) {
        isSubflow = true;
    }
    if ( activityNode.attributes.activityType == Ext.ux.suncode.Constants.TOOL ) {
        isTool = true;
    }

    var implementation = new Array();
    if ( Ext.isEmpty( activityNode.attributes.applications ) && !( isRoute || isSubflow ) ) {
        implementation.push( {
            No: new Array()
        } );
    } else if ( isTool ) {
        var apps = activityNode.attributes.applications;
        for ( var i = 0; i < apps.length; i++ ) {
            var tool = new Array();
            var app = apps[i];

            tool.push( {
                Id: app.id
            } );

            if ( app.integrationComponent ) {
            	var extAttrs = new Array();
            	extAttrs.push( {
                    ExtendedAttribute: {
                        Name: 'CUSTOM_DESCRIPTION',
                        Value: app.customDescription
                    }
                } );
            	if ( !Ext.isEmpty( app.condition ) ) {
            		extAttrs.push( {
                        ExtendedAttribute: {
                            Name: 'CONDITION',
                            Value: Ext.util.JSON.encode( app.condition )
                        }
                    } );
                }
            	if ( !Ext.isEmpty( app.errorHandling ) ) {
            		extAttrs.push( {
                        ExtendedAttribute: {
                            Name: 'ERROR_HANDLER',
                            Value: Ext.util.JSON.encode( app.errorHandling )
                        }
                    } );
                }
            	if ( app.inactive ) {
            		extAttrs.push( {
                        ExtendedAttribute: {
                            Name: 'INACTIVE',
                            Value: 'TRUE'
                        }
                    } );
                }
              if ( !Ext.isEmpty( app.creationDate ) ) {
                extAttrs.push( {
                  ExtendedAttribute: {
                    Name: 'CREATION_DATE',
                    Value: app.creationDate
                  }
                } );
              }
              if ( !Ext.isEmpty( app.modificationDate ) ) {
                extAttrs.push( {
                  ExtendedAttribute: {
                    Name: 'MODIFICATION_DATE',
                    Value: app.modificationDate
                  }
                } );
              }
                extAttrs.push( {
                    ExtendedAttribute: {
                        Name: 'PARAMETERS',
                        Value: Ext.util.JSON.encode( app.parameters )
                    }
                } );
                
            	tool.push( {
                    ExtendedAttributes: extAttrs
                } );
            } else if ( !Ext.isEmpty( app.parameters ) ) {
            	var params = new Array();

                var ps = app.parameters;
                for ( var j = 0; j < ps.length; j++ ) {
                    var p = ps[j];
                    var param = new Array();

                    if ( !Ext.isEmpty( p.varId ) ) {
                        param.push( {
                            'XPDLNODEVALUE': p.varId
                        } );

                        params.push( {
                            ActualParameter: param
                        } );
                    }
                }

                tool.push( {
                    ActualParameters: params
                } );
            }

            implementation.push( {
                Tool: tool
            } );
        }
    } else if ( isSubflow ) {
        var subflow = new Array();

        subflow.push( {
            Id: activityNode.attributes.subflow.id
        } );

        var actualParameters = new Array();
        var subflowParams = activityNode.attributes.subflow.params;
        for ( var i = 0; i < subflowParams.length; i++ ) {
            var actualParamValue = subflowParams[i].actualParam;

            if ( !Ext.isEmpty( actualParamValue ) ) {
                var oneActualParam = new Array();
                oneActualParam.push( {
                    'XPDLNODEVALUE': actualParamValue
                } );
                actualParameters.push( {
                    ActualParameter: oneActualParam
                } );
            }
        }

        if ( !Ext.isEmpty( actualParameters ) ) {
            subflow.push( {
                ActualParameters: actualParameters
            } );
        }

        implementation.push( {
            SubFlow: subflow
        } );
    }

    if ( !isRoute ) {
        activity.push( {
            Implementation: implementation
        } );
    }

    if ( !isRoute ) {
        activity.push( {
            Performer: {
                'XPDLNODEVALUE': activityNode.attributes.roleId
            }
        } );
    }

    var deadlines = activityNode.attributes.deadlines;
    for ( var i = 0; i < activityNode.attributes.deadlines.length; i++ ) {
        var d = deadlines[i];
        var condition = '';

        if ( Ext.isEmpty( d.condition ) ) {
            var unit = d.unit;
            var varId = d.varId;
            var amount = d.amount;
            var skipWeekends = d.skipWeekends;
            var skipHolidays = d.skipHolidays;

            condition = "var d=Packages.com.suncode.pwfl.shark.Deadlines." + unit + "s("
              + varId + "," + amount + "," + skipWeekends + "," + skipHolidays + ");d;";
        } else {
            condition = d.condition;
        }

        var deadline = new Array();
        deadline.push( {
            Execution: d.execution
        } );

        var deadlineCond = new Array();
        deadlineCond.push( {
            'XPDLNODEVALUE': condition
        } );

        var exceptionName = new Array();
        exceptionName.push( {
            'XPDLNODEVALUE': d.exceptionName
        } );

        deadline.push( {
            DeadlineCondition: deadlineCond,
            ExceptionName: exceptionName
        } );

        activity.push( {
            Deadline: deadline
        } );
    }

    var allTrs = activityNode.attributes.transitions;
    var allTrsCount = allTrs.basic.length + allTrs.cond.length + allTrs.otherwise.length + allTrs.exception.length;

    if ( allTrsCount > 1 || allTrs.incoming > 1 ) {
        var transitionRestriction = new Array();

        if ( allTrs.incoming > 1 ) {
            transitionRestriction.push( {
                Join: {
                    Type: activityNode.attributes.transitionRestriction.join
                }
            } );
        }

        if ( allTrsCount > 1 ) {
            var transitionRefs = new Array();

            for ( var i = 0; i < allTrs.cond.length; i++ ) {
                transitionRefs.push( {
                    TransitionRef: {
                        Id: allTrs.cond[i].id
                    }
                } );
            }

            for ( var i = 0; i < allTrs.basic.length; i++ ) {
                transitionRefs.push( {
                    TransitionRef: {
                        Id: allTrs.basic[i].id
                    }
                } );
            }

            for ( var i = 0; i < allTrs.exception.length; i++ ) {
                transitionRefs.push( {
                    TransitionRef: {
                        Id: allTrs.exception[i].id
                    }
                } );
            }

            for ( var i = 0; i < allTrs.otherwise.length; i++ ) {
                transitionRefs.push( {
                    TransitionRef: {
                        Id: allTrs.otherwise[i].id
                    }
                } );
            }

            transitionRestriction.push( {
                Split: {
                    Type: activityNode.attributes.transitionRestriction.split,
                    TransitionRefs: transitionRefs
                }
            } );
        }

        activity.push( {
            TransitionRestrictions: {
                TransitionRestriction: transitionRestriction
            }
        } );
    }

    var extAttrs = new Array();
    extAttrs.push( {
        ExtendedAttribute: {
            Name: 'JaWE_GRAPH_PARTICIPANT_ID',
            Value: activityNode.attributes.roleId
        }
    } );
    extAttrs.push( {
        ExtendedAttribute: {
            Name: 'JaWE_GRAPH_OFFSET',
            Value: activityNode.attributes.xOffset + ',' + activityNode.attributes.yOffset
        }
    } );

    extAttrs = xpdlVariablesToProcess( activityNode, extAttrs );
    extAttrs = xpdlHttpLinks( activityNode, extAttrs );
    extAttrs = xpdlAcceptButtons( activityNode, extAttrs );
    extAttrs = xpdlAcceptances( activityNode, extAttrs );
    extAttrs = xpdlFormActions( activityNode, extAttrs );
    extAttrs = xpdlEvents( activityNode, extAttrs );
    extAttrs = xpdlValidators( activityNode, extAttrs );
    extAttrs = xpdlVariablesSetters( activityNode, extAttrs );

    if ( !Ext.isEmpty( activityNode.attributes.assignment.type ) && activityNode.attributes.assignment.type != 'role' ) {
        var setAssignmentValue = activityNode.attributes.assignment.type + '=' + activityNode.attributes.assignment.value;

        if ( !Ext.isEmpty( activityNode.attributes.assignment.filters ) ) {
            var assignmentFilters = activityNode.attributes.assignment.filters;

            for ( var i = 0; i < assignmentFilters.length; i++ ) {
                var assignmentFilter = assignmentFilters[i];

                setAssignmentValue += '\n' + assignmentFilter.type + '=' + assignmentFilter.value;
            }
        }

        if ( !Ext.isEmpty( activityNode.attributes.assignment.exception ) ) {
            setAssignmentValue += '\nException=' + activityNode.attributes.assignment.exception;
        }

        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'SET_ASSIGNMENT',
                Value: setAssignmentValue
            }
        } );
    } else if ( !Ext.isEmpty( activityNode.attributes.assignment.type ) && activityNode.attributes.assignment.type == 'role' ) {
        if ( !Ext.isEmpty( activityNode.attributes.assignment.filters ) ) {
            var setAssignmentValues = new Array();
            var assignmentFilters = activityNode.attributes.assignment.filters;

            for ( var i = 0; i < assignmentFilters.length; i++ ) {
                var assignmentFilter = assignmentFilters[i];

                setAssignmentValues.push( assignmentFilter.type + '=' + assignmentFilter.value );
            }

            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'SET_ASSIGNMENT',
                    Value: setAssignmentValues.join( '\n' )
                }
            } );
        }
    }

    if ( activityNode.attributes.nextActivity && !Ext.isEmpty( activityNode.attributes.acceptedNextActivities ) ) {
        var acceptedNextActivities = activityNode.attributes.acceptedNextActivities;
        var acceptedNextActivitiesValue = '';

        for ( var i = 0; i < acceptedNextActivities.length; i++ ) {
            if ( i == ( acceptedNextActivities.length - 1 ) ) {
                acceptedNextActivitiesValue += acceptedNextActivities[i].activityDefId;
            } else {
                acceptedNextActivitiesValue += acceptedNextActivities[i].activityDefId + '\n';
            }
        }

        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'NEXT_ACTIVITY',
                Value: acceptedNextActivitiesValue
            }
        } );
    } else if ( activityNode.attributes.nextActivity ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'NEXT_ACTIVITY'
            }
        } );
    }

    if ( !activityNode.attributes.acceptConcurrently ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'ACCEPT_CONCURRENTLY',
                Value: 'FALSE'
            }
        } );
    }

    if ( activityNode.attributes.acceptSequentially ) {
      extAttrs.push( {
        ExtendedAttribute: {
          Name: 'ACCEPT_SEQUENTIALLY',
          Value: 'TRUE'
        }
      } );
    }

    if ( activityNode.attributes.newDocumentAccept ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'NEW_DOCUMENT_ACCEPT',
                Value: 'TRUE'
            }
        } );
    }

    if ( activityNode.attributes.allowMultipleActiveActivities ) {
        configuration.push( {
            id: 'allowMultipleActiveActivities',
            value: 'true'
        } );
    }

    if ( !Ext.isEmpty( activityNode.attributes.additionals.defaults ) ) {
        var defaults = activityNode.attributes.additionals.defaults;

        for ( var i = 0; i < defaults.length; i++ ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'SET_DEFAULT',
                    Value: defaults[i].varId + '=' + defaults[i].varValue
                }
            } );
        }
    }

    if ( !Ext.isEmpty( activityNode.attributes.additionals.checkAppendices ) ) {
        var checkAppendicesText = '';
        var checkAppendices = activityNode.attributes.additionals.checkAppendices;
        for ( var i = 0; i < checkAppendices.length; i++ ) {
            if ( i == checkAppendices.length - 1 ) {
                checkAppendicesText += checkAppendices[i].docClassName;
            } else {
                checkAppendicesText += checkAppendices[i].docClassName + '\n';
            }
        }

        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'CHECK_APPENDICES',
                Value: checkAppendicesText
            }
        } );
    }

    if ( !Ext.isEmpty( activityNode.attributes.additionals.additionalJsChecks ) ) {
        var additionalJsChecks = activityNode.attributes.additionals.additionalJsChecks;

        for ( var i = 0; i < additionalJsChecks.length; i++ ) {
            var additionalJsCheck = additionalJsChecks[i].additionalJsCheck;

            if ( !additionalJsCheck.endsWith( ')' ) ) {
                additionalJsCheck += '()';
            }

            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'ADDITIONAL_JS_CHECK',
                    Value: additionalJsCheck
                }
            } );
        }
    }

    if ( !Ext.isEmpty( activityNode.attributes.additionals.extraDataChooserConfigs ) ) {
        var extraDataChooserConfigs = activityNode.attributes.additionals.extraDataChooserConfigs;

        for ( var i = 0; i < extraDataChooserConfigs.length; i++ ) {
            var extraDataChooserConfig = extraDataChooserConfigs[i];
            var extraConfigDef = extraDataChooserConfig.dataChooserId;

            for ( var j = 0; j < extraDataChooserConfig.attributes.length; j++ ) {
                var attribute = extraDataChooserConfig.attributes[j];
                extraConfigDef += '\n' + attribute.key + '=' + attribute.value;
            }

            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'DATA_CHOOSER',
                    Value: extraConfigDef
                }
            } );
        }
    }

    if ( !Ext.isEmpty( configuration ) ) {
        var configOptions = new Array();

        for ( var i = 0; i < configuration.length; i++ ) {
            var oneConfig = configuration[i];
            configOptions.push( oneConfig.id + '=' + oneConfig.value );
        }

        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'CONFIGURATION',
                Value: configOptions.join( '\n' )
            }
        } );
    }
    
    if ( !Ext.isEmpty( activityNode.attributes.templates.validators ) ) {
    	extAttrs.push( {
            ExtendedAttribute: {
                Name: 'VALIDATOR_TEMPLATES',
                Value: Ext.util.JSON.encode( activityNode.attributes.templates.validators )
            }
        } );
    }
    
    if ( !Ext.isEmpty( activityNode.attributes.templates.variablesSetters ) ) {
    	extAttrs.push( {
            ExtendedAttribute: {
                Name: 'VARIABLES_SETTER_TEMPLATES',
                Value: Ext.util.JSON.encode( activityNode.attributes.templates.variablesSetters )
            }
        } );
    }

    if ( !Ext.isEmpty( extAttrs ) ) {
        activity.push( {
            ExtendedAttributes: extAttrs
        } );
    }

    return activity;
}

function xpdlVariablesToProcess( activityNode, extAttrs ) {
    var vs = activityNode.attributes.form.variables;
    for ( var i = 0; i < vs.length; i++ ) {
        extAttrs = xpdlVariableToProcess( vs[i], extAttrs );
    }

    return extAttrs;
}

function xpdlVariableToProcess( variable, extAttrs ) {
    var extAttrName = '';
    var extAttrValue = '';
    var autoUpdateExtAttrs = new Array();

    switch ( variable.genre ) {
        case 'VARIABLE':
            if ( variable.hidden ) {
                extAttrName = 'VariableToProcess_HIDDEN';
            } else if ( !variable.editable ) {
                extAttrName = 'VariableToProcess_VIEW';
            } else {
                extAttrName = 'VariableToProcess_UPDATE';
            }

            if ( !Ext.isEmpty( variable.autoUpdates ) ) {
                for ( var i = 0; i < variable.autoUpdates.length; i++ ) {
                    var autoUpdate = variable.autoUpdates[i];
                    var autoUpdateDef = 'actionsource=' + variable.varId;

                    autoUpdateDef += '\ndatasource=';
                    var dataSources = autoUpdate.dataSources;
                    for ( var j = 0; j < dataSources.length; j++ ) {
                        if ( j == ( dataSources.length - 1 ) ) {
                            autoUpdateDef += dataSources[j].varId;
                        } else {
                            autoUpdateDef += dataSources[j].varId + ',';
                        }
                    }

                    autoUpdateDef += '\ndestination=';
                    var destinations = autoUpdate.destinations;
                    for ( var j = 0; j < destinations.length; j++ ) {
                        if ( j == ( destinations.length - 1 ) ) {
                            autoUpdateDef += destinations[j].varId;
                        } else {
                            autoUpdateDef += destinations[j].varId + ',';
                        }
                    }

                    autoUpdateDef += '\ntaskName=';

                    switch ( autoUpdate.taskNameType ) {
                        case 'Struts':
                            autoUpdateDef += autoUpdate.taskName + '.do';
                            break;
                        case 'Servlet':
                            autoUpdateDef += autoUpdate.taskName + '.customServlet';
                            break;
                        case 'RESTful':
                            autoUpdateDef += 'api/' + autoUpdate.taskName;
                            break;
                        case 'Applet':
                            autoUpdateDef += autoUpdate.taskName;
                            break;
                        default:
                            break;
                    }

                    autoUpdateDef += '\neventType=' + autoUpdate.eventType;

                    if ( autoUpdate.taskNameType == 'Applet' ) {
                        autoUpdateDef += '\nbuttonName=' + autoUpdate.buttonName;
                    }

                    autoUpdateExtAttrs.push( {
                        ExtendedAttribute: {
                            Name: 'AUTO_UPDATE',
                            Value: autoUpdateDef
                        }
                    } );
                }
            }

            extAttrValue = variable.varId;

            if ( !Ext.isEmpty( variable.roleusersFilters ) ) {
                extAttrValue += '\nconjunction=' + variable.roleusersConjunction;

                var roleusersFilters = variable.roleusersFilters;

                for ( var i = 0; i < roleusersFilters.length; i++ ) {
                    var roleusersFilter = roleusersFilters[i];

                    extAttrValue += '\n' + roleusersFilter.type + '=' + roleusersFilter.value;
                }
            }

            break;
        case 'VARIABLE_SET':
            if ( !variable.editable ) {
                extAttrName = 'VARIABLE_SET_VIEW';
            } else {
                extAttrName = 'VARIABLE_SET';
            }

            var dtConfValue = '';
            var columns = variable.columns;
            for ( var i = 0; i < columns.length; i++ ) {
                var colDef = columns[i];

                if ( i == columns.length - 1 ) {
                    extAttrValue += colDef.varId;
                } else {
                    extAttrValue += colDef.varId + '\n';
                }

                if ( colDef.differentHidden || colDef.hidden ) {
                    if ( Ext.isEmpty( dtConfValue ) ) {
                        dtConfValue += 'DT_CONF_' + colDef.varId + '_DT_HIDDEN=' + colDef.hidden;
                    } else {
                        dtConfValue += '\nDT_CONF_' + colDef.varId + '_DT_HIDDEN=' + colDef.hidden;
                    }

                    if ( colDef.differentHidden ) {
                      dtConfValue += '\nDT_CONF_' + colDef.varId + '_DIFFERENT_HIDDEN=true';
                    }
                }

                if ( colDef.differentReadonly || colDef.readonly ) {
                    if ( Ext.isEmpty( dtConfValue ) ) {
                        dtConfValue += 'DT_CONF_' + colDef.varId + '_READONLY=' + colDef.readonly;
                    } else {
                        dtConfValue += '\nDT_CONF_' + colDef.varId + '_READONLY=' + colDef.readonly;
                    }

                    if ( colDef.differentReadonly ) {
                      dtConfValue += '\nDT_CONF_' + colDef.varId + '_DIFFERENT_READONLY=true';
                    }
                }
            }

            if ( !Ext.isEmpty( dtConfValue ) ) {
                extAttrValue += '\n' + dtConfValue;
            }

            extAttrValue += '\nDT_TABLE_ID=' + variable.varId;

            if ( !Ext.isEmpty( variable.name ) ) {
                extAttrValue += '\nDT_TABLE_NAME=' + variable.name;
            }
            if ( !Ext.isEmpty( variable.clicksToEdit ) ) {
              extAttrValue += '\nDT_CLICKS_TO_EDIT=' + variable.clicksToEdit;
            }
            if ( !Ext.isEmpty( variable.showDeleteConfirmation ) ) {
              extAttrValue += '\nDT_TABLE_SHOW_DELETE_CONFIRMATION=' + variable.showDeleteConfirmation.toString().toUpperCase();
            }
            if ( variable.hideRowNumber ) {
              extAttrValue += '\nDT_TABLE_HIDE_ROW_NUMBER=' + variable.hideRowNumber.toString().toUpperCase();
            }
            if ( variable.local != 'DEFAULT' ) {
                extAttrValue += '\nDT_LOCAL=' + variable.local;
            }
            if ( !variable.collapsible ) {
                extAttrValue += '\nDT_COLLAPSIBLE=' + variable.collapsible.toString().toUpperCase();
            }
            if ( !variable.stateful ) {
                extAttrValue += '\nDT_STATEFUL=' + variable.stateful.toString().toUpperCase();
            }
            if ( variable.autoHeight ) {
                extAttrValue += '\nDT_TABLE_AUTO_HEIGHT=' + variable.autoHeight.toString().toUpperCase();
                extAttrValue += '\nDT_TABLE_MIN_HEIGHT=' + variable.autoHeightMin;
                extAttrValue += '\nDT_TABLE_MAX_HEIGHT=' + variable.autoHeightMax;
            }
            if ( !Ext.isEmpty( variable.dtButtonsFillPosition ) ) {
              extAttrValue += '\nDT_BUTTONS_FILL_POSITION=' + variable.dtButtonsFillPosition;
            }

            var mainPanel = Ext.getCmp( 'main_panel' );
            var actions = variable.actions;
            for ( var i = 0; i < actions.length; i++ ) {
                var action = actions[i];
                extAttrValue += '\n' + action.source + '=' + action.func;

                mainPanel.storeDTActionJS( action );
            }

            if ( variable.editable ) {
                var DTButtons = variable.DTButtons;
                for ( var i = 0; i < DTButtons.length; i++ ) {
                    var DTButton = DTButtons[i];
                    extAttrs = xpdlDTButton( variable, DTButton, extAttrs );

                    mainPanel.storeDTButtonJS( DTButton );
                }
            }

            if ( !Ext.isEmpty( variable.groupers ) ) {
                var groupers = variable.groupers;
                for ( var i = 0; i < groupers.length; i++ ) {
                    var grouper = groupers[i];
                    extAttrValue += '\n' + 'DT_GROUPER_' + grouper.direction + '=' + grouper.columnId;
                    extAttrValue += ';groupNameColumnId=' + grouper.groupNameColumnId;
                    extAttrValue += ';aggregations=' + Ext.util.JSON.encode( grouper.aggregations );
                }
            }

            if ( !Ext.isEmpty( variable.sorters ) ) {
              var sorters = variable.sorters;
              for ( var i = 0; i < sorters.length; i++ ) {
                var sorter = sorters[i];
                extAttrValue += '\n' + 'DT_SORTER_' + sorter.direction + '=' + sorter.columnId;
              }
            }
            break;
        default:
            break;
    }

    if ( !Ext.isEmpty( extAttrValue ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: extAttrName,
                Value: extAttrValue
            }
        } );
    }

    if ( !Ext.isEmpty( autoUpdateExtAttrs ) ) {
        extAttrs = extAttrs.concat( autoUpdateExtAttrs );
    }

    return extAttrs;
}

function xpdlDTButton( variable, DTButton, extAttrs ) {
    var extAttrValue = 'variableSetId=' + DTButton.vsId;
    extAttrValue += '\nvariable_set=';

    var columns = variable.columns;
    for ( var i = 0; i < columns.length; i++ ) {
        if ( i == columns.length - 1 ) {
            extAttrValue += columns[i].varId;
        } else {
            extAttrValue += columns[i].varId + ';';
        }
    }

    if ( DTButton.disabled ) {
        extAttrValue += '\ndisabled=' + DTButton.disabled;
    }

    extAttrValue += '\ndestination=' + DTButton.destination;
    extAttrValue += '\nhandler=' + DTButton.handler;
    extAttrValue += '\nid=' + DTButton.buttonId;

    if ( !Ext.isEmpty( DTButton.name ) && !Ext.isEmpty( DTButton.icon ) ) {
        extAttrValue += '\ncls=x-btn-text-icon';
        extAttrValue += '\ntext=' + DTButton.name;
        extAttrValue += '\nicon=' + DTButton.icon;
    } else if ( !Ext.isEmpty( DTButton.name ) ) {
        extAttrValue += '\ncls=x-btn-text';
        extAttrValue += '\ntext=' + DTButton.name;
    } else {
        extAttrValue += '\ncls=x-btn-icon';
        extAttrValue += '\nicon=' + DTButton.icon;
    }

    if ( !Ext.isEmpty( DTButton.color ) ) {
      extAttrValue += '\ncolor=' + DTButton.color;
    }

    if ( !Ext.isEmpty( DTButton.tooltip ) ) {
        extAttrValue += '\ntooltip=' + DTButton.tooltip;
    }

    if ( !Ext.isEmpty( DTButton.globalId ) ) {
      extAttrValue += '\nglobalId=' + DTButton.globalId;
    }

    if ( !Ext.isEmpty( DTButton.buttonDefId ) ) {
        extAttrValue += '\nbuttonDefId=' + DTButton.buttonDefId;

        if ( DTButton.parameterized && !Ext.isEmpty( DTButton.params ) ) {
            var ps = DTButton.params;
            for ( var i = 0; i < ps.length; i++ ) {
                var p = ps[i];
                extAttrValue += '\nparam' + ( i + 1 ) + '=' + p.realName + '=' + p.value;
            }
        }
    }

    extAttrs.push( {
        ExtendedAttribute: {
            Name: 'DT_BUTTON',
            Value: extAttrValue
        }
    } );

    return extAttrs;
}

function xpdlAcceptButtons( activityNode, extAttrs ) {
    var btns = activityNode.attributes.form.buttons;
    for ( var i = 0; i < btns.length; i++ ) {
        extAttrs = xpdlAcceptButton( btns[i], extAttrs );
    }

    return extAttrs;
}

function xpdlAcceptButton( button, extAttrs ) {
    var extAttrValue = '';

    switch ( button.genre ) {
        case 'ACTION_ACCEPT_BUTTON':
            extAttrValue = 'actionName=' + button.actionName;
            extAttrValue += '\nbuttonName=' + button.buttonName;

            if ( !Ext.isEmpty( button.tooltip ) ) {
              extAttrValue += '\ntooltip=' + button.tooltip;
            }

            var actions = button.actions;
            for ( var i = 0; i < actions.length; i++ ) {
                var action = actions[i];
                var attributionsText = '';
                var attributions = action.attributions;

                for ( var j = 0; j < attributions.length; j++ ) {
                    var attribution = attributions[j];

                    if ( j == attributions.length - 1 ) {
                        attributionsText += attribution.varId + '=' + attribution.value;
                    } else {
                        attributionsText += attribution.varId + '=' + attribution.value + ',';
                    }
                }

                extAttrValue += '\naction' + ( i + 1 ) + '=' + attributionsText;

                var conditions = action.conditions;
                if ( !Ext.isEmpty( conditions ) ) {
                    var conditionsText = '';

                    for ( var j = 0; j < conditions.length; j++ ) {
                        var condition = conditions[j];
                        var value = '';

                        if ( Ext.isEmpty( condition.value ) ) {
                            value = '\'\'';
                        } else {
                            value = condition.value;
                        }

                        if ( j == conditions.length - 1 ) {
                            conditionsText += condition.varId + condition.operator + value;
                        } else {
                            conditionsText += condition.varId + condition.operator + value + ' AND ';
                        }
                    }

                    extAttrValue += '\nCondition' + '=' + conditionsText;
                }
            }

            if ( !Ext.isEmpty( button.jsAction ) ) {
                var jsAction = button.jsAction;
                if ( !jsAction.endsWith( ')' ) ) {
                    jsAction += '()';
                }

                extAttrValue += '\njsAction=' + jsAction;

                Ext.getCmp( 'main_panel' ).storeAcceptButtonJS( button );
            }

            extAttrValue += '\ncheckForm=' + button.checkForm.toString().toUpperCase();
            extAttrValue += '\nsaveBeforeAccept=' + button.saveBeforeAccept.toString().toUpperCase();

            if ( !Ext.isEmpty( button.destination ) ) {
                extAttrValue += '\ndestination=' + button.destination;
            }

            if ( !Ext.isEmpty( button.forwardTo ) ) {
                extAttrValue += '\nforwardTo=' + button.forwardTo;
            }
            
            if ( !Ext.isEmpty( button.acceptConcurrently ) ) {
            	extAttrValue += '\nacceptConcurrently=' + button.acceptConcurrently;
            }
            
            if ( button.nextActivity == 'true' && !Ext.isEmpty( button.acceptedNextActivities ) ) {
            	var acceptedNextActivities = button.acceptedNextActivities;
                var acceptedNextActivitiesValue = '';

                for ( var i = 0; i < acceptedNextActivities.length; i++ ) {
                    if ( i == ( acceptedNextActivities.length - 1 ) ) {
                        acceptedNextActivitiesValue += acceptedNextActivities[i].activityDefId;
                    } else {
                        acceptedNextActivitiesValue += acceptedNextActivities[i].activityDefId + ';';
                    }
                }
                
                extAttrValue += '\nnextActivity=' + acceptedNextActivitiesValue;
            } else if ( !Ext.isEmpty( button.nextActivity ) ) {
            	extAttrValue += '\nnextActivity=' + button.nextActivity;
            }

            if ( !Ext.isEmpty( button.dcIncludes ) ) {
                extAttrValue += '\ndcIncludes=';

                var dcIncludes = button.dcIncludes;
                for ( var i = 0; i < dcIncludes.length; i++ ) {
                    if ( i == dcIncludes.length - 1 ) {
                        extAttrValue += dcIncludes[i].dataChooserId;
                    } else {
                        extAttrValue += dcIncludes[i].dataChooserId + ',';
                    }
                }
            }

            if ( !Ext.isEmpty( button.dcExcludes ) ) {
                extAttrValue += '\ndcExcludes=';

                var dcExcludes = button.dcExcludes;
                for ( var i = 0; i < dcExcludes.length; i++ ) {
                    if ( i == dcExcludes.length - 1 ) {
                        extAttrValue += dcExcludes[i].dataChooserId;
                    } else {
                        extAttrValue += dcExcludes[i].dataChooserId + ',';
                    }
                }
            }

            if ( !Ext.isEmpty( button.oldValidators ) ) {
                extAttrValue += '\nvalidators=';

                var oldValidators = button.oldValidators;
                for ( var i = 0; i < oldValidators.length; i++ ) {
                    if ( i == oldValidators.length - 1 ) {
                        extAttrValue += xpdlClass( oldValidators[i] );
                    } else {
                        extAttrValue += xpdlClass( oldValidators[i] ) + ';';
                    }
                }
            }

            if ( !Ext.isEmpty( button.oldVariablesSetters ) ) {
                extAttrValue += '\nvariablesSetters=';

                var oldVariablesSetters = button.oldVariablesSetters;
                for ( var i = 0; i < oldVariablesSetters.length; i++ ) {
                    if ( i == oldVariablesSetters.length - 1 ) {
                        extAttrValue += xpdlClass( oldVariablesSetters[i] );
                    } else {
                        extAttrValue += xpdlClass( oldVariablesSetters[i] ) + ';';
                    }
                }
            }

            extAttrValue += '\nfilled=' + button.filled;
            extAttrValue += '\nhideFrame=' + button.hideFrame;

            if ( !Ext.isEmpty( button.width ) ) {
              extAttrValue += '\nbuttonWidth=' + button.width;
            }

            if ( !Ext.isEmpty( button.height ) ) {
              extAttrValue += '\nbuttonHeight=' + button.height;
            }
            break;
        case 'GENERATE_PDF_BUTTON':
            extAttrValue = 'buttonName=' + button.buttonName;
            extAttrValue += '\nDocClass=' + button.docClassName;
            extAttrValue += '\nTemplate=' + button.templateName;
            extAttrValue += '\ncheckForm=' + button.checkForm.toString().toUpperCase();
            break;
        case 'ADD_FILE_BUTTON':
            extAttrValue = 'buttonName=' + button.buttonName;
            extAttrValue += '\nClassName=' + button.docClassName;
            extAttrValue += '\nsourceName=' + button.buttonName;
            break;
        case 'BARCODE_PRINT':
            extAttrValue += 'LabelPath=' + button.labelPath;
            extAttrValue += '\n' + button.labelTextVar;
            break;
        default:
            break;
    }

    if ( !Ext.isEmpty( extAttrValue ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: button.genre,
                Value: extAttrValue
            }
        } );
    }

    return extAttrs;
}

function xpdlClass( clazz ) {
    var value = clazz.classpath + '(';

    var parameters = clazz.parameters;
    if ( !Ext.isEmpty( parameters ) ) {
        for ( var i = 0; i < parameters.length; i++ ) {
            if ( i == parameters.length - 1 ) {
                value += parameters[i].id;
            } else {
                value += parameters[i].id + ',';
            }
        }
    }

    value += ')';

    return value;
}

function xpdlHttpLinks( activityNode, extAttrs ) {
    var mainPanel = Ext.getCmp( 'main_panel' );
    var hls = activityNode.attributes.form.httpLinks;
    for ( var i = 0; i < hls.length; i++ ) {
        var hl = hls[i];

        if ( hl.type == 'JavaScriptLink' ) {
            extAttrs = xpdlHttpLink( hl, extAttrs );
            mainPanel.storeHttpLinkJS( hl );
        } else if ( hl.type == 'UrlLink' || hl.type == 'Actions' ) {
            extAttrs = xpdlHttpLink( hl, extAttrs );
        } else {
            extAttrs = xpdlActionButton( hl, extAttrs );
        }
    }

    return extAttrs;
}

function xpdlHttpLink( httpLink, extAttrs ) {
    var extAttrValue = 'type=' + httpLink.type;
    extAttrValue += '\ndestination=' + httpLink.destination;
    extAttrValue += '\nName=' + httpLink.buttonName;

    if ( !Ext.isEmpty( httpLink.tooltip ) ) {
      extAttrValue += '\nTooltip=' + httpLink.tooltip;
    }

    if ( httpLink.type == 'UrlLink' || httpLink.type == 'JavaScriptLink' ) {
        extAttrValue += '\nUrl=' + httpLink.url;
    }

    if ( !Ext.isEmpty( httpLink.params ) ) {
	    extAttrValue += '\nParams=';
	
	    var params = httpLink.params;
	    for ( var i = 0; i < params.length; i++ ) {
	        if ( i == params.length - 1 ) {
	            extAttrValue += params[i].varId;
	        } else {
	            extAttrValue += params[i].varId + ',';
	        }
	    }
    }

    extAttrValue += '\nsaveForm=' + httpLink.saveForm.toString().toUpperCase();

    extAttrs.push( {
        ExtendedAttribute: {
            Name: 'HTTP_LINK',
            Value: extAttrValue
        }
    } );

    return extAttrs;
}

function xpdlActionButton( httpLink, extAttrs ) {
    var extAttrValue = 'buttonName=' + httpLink.buttonName;
    extAttrValue += '\ndestination=' + httpLink.destination;

    if ( !Ext.isEmpty( httpLink.tooltip ) ) {
      extAttrValue += '\ntooltip=' + httpLink.tooltip;
    }

    if ( httpLink.type == 'ActionLink' ) {
        extAttrValue += '\nactionLink=' + httpLink.url;
    } else if ( httpLink.type == 'WindowConf' ) {
        extAttrValue += '\nwindowConf=' + httpLink.url;
    }

    extAttrValue += '\ntype=AJAXLINK';
    extAttrValue += '\nsaveForm=' + httpLink.saveForm.toString().toUpperCase();

    extAttrs.push( {
        ExtendedAttribute: {
            Name: 'ACTIONBUTTON',
            Value: extAttrValue
        }
    } );

    return extAttrs;
}

function xpdlAcceptances( activityNode, extAttrs ) {
    var acceptances = activityNode.attributes.acceptances;
    for ( var i = 0; i < acceptances.length; i++ ) {
        extAttrs = xpdlAcceptance( acceptances[i], extAttrs );
    }

    return extAttrs;
}

function xpdlAcceptance( acceptance, extAttrs ) {
    var extAttrValue = 'acceptVar=' + acceptance.acceptVar;
    extAttrValue += '\ncheckVar=' + acceptance.checkVar;
    extAttrValue += '\nsetVar=' + acceptance.resultVar + ':' + acceptance.acceptVarValue + '?' + acceptance.checkVarValue;

    if ( !Ext.isEmpty( acceptance.buttonName ) ) {
        extAttrValue += '\nactionName=' + acceptance.buttonName;
    }

    extAttrs.push( {
        ExtendedAttribute: {
            Name: 'CHECK_ACCEPTATION',
            Value: extAttrValue
        }
    } );

    return extAttrs;
}

function xpdlValidators( activityNode, extAttrs ) {
	var validators = new Array();
	var btns = activityNode.attributes.form.buttons;
    for ( var i = 0; i < btns.length; i++ ) {
    	var btn = btns[i];
    	if ( btn.genre == 'ACTION_ACCEPT_BUTTON' ) {
    		var vs = btn.validators;
    		for ( var j = 0; j < vs.length; j++ ) {
    			validators.push( xpdlValidator( btn, vs[j] ) );
    		}
    	}
    }

    if ( !Ext.isEmpty( validators ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'VALIDATORS',
                Value: Ext.util.JSON.encode( validators )
            }
        } );
    }
    
    return extAttrs;
}

function xpdlValidator( button, validator ) {
    var def = new Object();
    def = Ext.apply( {
        id: validator.id,
        action: button.actionName,
        customDescription: validator.customDescription,
        condition: validator.condition,
        confirmation: validator.confirmation,
        error: validator.error,
        inactive: validator.inactive,
        creationDate: validator.creationDate,
        modificationDate: validator.modificationDate,
        parameters: validator.parameters
    }, def );

    return def;
}

function xpdlVariablesSetters( activityNode, extAttrs ) {
	var variablesSetters = new Array();
	var btns = activityNode.attributes.form.buttons;
    for ( var i = 0; i < btns.length; i++ ) {
    	var btn = btns[i];
    	if ( btn.genre == 'ACTION_ACCEPT_BUTTON' ) {
    		var vss = btn.variablesSetters;
    		for ( var j = 0; j < vss.length; j++ ) {
    			variablesSetters.push( xpdlVariablesSetter( btn, vss[j] ) );
    		}
    	}
    }

    if ( !Ext.isEmpty( variablesSetters ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'VARIABLES_SETTERS',
                Value: Ext.util.JSON.encode( variablesSetters )
            }
        } );
    }
    
    return extAttrs;
}

function xpdlVariablesSetter( button, variablesSetter ) {
    var def = new Object();
    def = Ext.apply( {
        id: variablesSetter.id,
        action: button.actionName,
        customDescription: variablesSetter.customDescription,
        condition: variablesSetter.condition,
        inactive: variablesSetter.inactive,
        creationDate: variablesSetter.creationDate,
        modificationDate: variablesSetter.modificationDate,
        parameters: variablesSetter.parameters
    }, def );

    return def;
}

function xpdlFormActions( activityNode, extAttrs ) {
    var formActions = new Array();
    formActions = xpdlFormFormActions( activityNode.attributes.form.actions, formActions );
    formActions = xpdlFormElementFormActions( activityNode.attributes.form.variables, formActions, 'varId' );
    formActions = xpdlFormElementFormActions( activityNode.attributes.form.httpLinks, formActions, 'actionName' );
    formActions = xpdlFormElementFormActions( activityNode.attributes.form.buttons, formActions, 'actionName' );
    formActions = xpdlDtButtonFormActions( activityNode.attributes.form.variables, formActions );
    formActions = xpdlLabelFormActions( activityNode.attributes.form.labelActions, formActions );

    if ( !Ext.isEmpty( formActions ) ) {
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'ACTIONS',
                Value: Ext.util.JSON.encode( formActions )
            }
        } );
    }

    return extAttrs;
}

function xpdlFormFormActions( formFormActions, formActions ) {
    if ( !Ext.isEmpty( formFormActions ) ) {
        for ( var i = 0; i < formFormActions.length; i++ ) {
            formActions.push( xpdlFormFormAction( formFormActions[i] ) );
        }
    }

    return formActions;
}

function xpdlFormFormAction( formFormAction ) {
    var formAction = new Object();
    formAction = Ext.apply( {
        id: formFormAction.id,
        target: {
            type: Ext.ux.suncode.IntegrationComponentService.FORM_DESTINATION
        },
        customDescription: formFormAction.customDescription,
        condition: formFormAction.condition,
        inactive: formFormAction.inactive,
        creationDate: formFormAction.creationDate,
        modificationDate: formFormAction.modificationDate,
        parameters: formFormAction.parameters
    }, formAction );

    return formAction;
}

function xpdlFormElementFormActions( elements, formActions, idKey ) {
    if ( !Ext.isEmpty( elements ) ) {
        for ( var i = 0; i < elements.length; i++ ) {
            var element = elements[i];
            var elementFormActions = element.formActions;

            if ( !Ext.isEmpty( elementFormActions ) ) {
                for ( var j = 0; j < elementFormActions.length; j++ ) {
                    formActions.push( xpdlFormElementFormAction( element, elementFormActions[j], idKey ) );
                }
            }
        }
    }

    return formActions;
}

function xpdlFormElementFormAction( element, elementFormAction, idKey ) {
    var targetType = '';
    switch ( element.genre ) {
        case 'VARIABLE':
            targetType = Ext.ux.suncode.IntegrationComponentService.VARIABLE_DESTINATION;
            break;
        case 'VARIABLE_SET':
            targetType = Ext.ux.suncode.IntegrationComponentService.VARIABLE_SET_DESTINATION;
            break;
        default:
            targetType = Ext.ux.suncode.IntegrationComponentService.BUTTON_DESTINATION;
            break;
    }

    var formAction = new Object();
    formAction = Ext.apply( {
        id: elementFormAction.id,
        target: {
            type: targetType,
            id: element[idKey]
        },
        customDescription: elementFormAction.customDescription,
        condition: elementFormAction.condition,
        inactive: elementFormAction.inactive,
        creationDate: elementFormAction.creationDate,
        modificationDate: elementFormAction.modificationDate,
        parameters: elementFormAction.parameters
    }, formAction );

    if ( !Ext.isEmpty( elementFormAction.globalId ) ) {
      formAction = Ext.apply( {
        globalId: elementFormAction.globalId
      }, formAction );
    }

    return formAction;
}

function xpdlDtButtonFormActions( variables, formActions ) {
  if ( !Ext.isEmpty( variables ) ) {
    for ( var i = 0; i < variables.length; i++ ) {
      var variable = variables[i];

      if ( variable.genre == 'VARIABLE_SET' && !Ext.isEmpty( variable.DTButtons ) ) {
        var dtButtons = variable.DTButtons;

        for ( var j = 0; j < dtButtons.length; j++ ) {
          var dtButton = dtButtons[j];
          var dtButtonFormActions = dtButton.formActions;

          if ( !Ext.isEmpty( dtButtonFormActions ) ) {
            for ( var k = 0; k < dtButtonFormActions.length; k++ ) {
              var dtButtonFormAction = dtButtonFormActions[k];

              var formAction = new Object();
              formAction = Ext.apply( {
                id: dtButtonFormAction.id,
                target: {
                  type: Ext.ux.suncode.IntegrationComponentService.DT_BUTTON_DESTINATION,
                  id: dtButton.buttonId
                },
                customDescription: dtButtonFormAction.customDescription,
                condition: dtButtonFormAction.condition,
                inactive: dtButtonFormAction.inactive,
                creationDate: dtButtonFormAction.creationDate,
                modificationDate: dtButtonFormAction.modificationDate,
                parameters: dtButtonFormAction.parameters
              }, formAction );

              if ( !Ext.isEmpty( dtButtonFormAction.globalId ) ) {
                formAction = Ext.apply( {
                  globalId: dtButtonFormAction.globalId
                }, formAction );
              }

              formActions.push( formAction );
            }
          }
        }
      }
    }
  }

  return formActions;
}

function xpdlLabelFormActions( labelActions, formActions ) {
    if ( !Ext.isEmpty( labelActions ) ) {
        for ( var i = 0; i < labelActions.length; i++ ) {
            var labelAction = labelActions[i];
            var labelFormActions = labelAction.formActions;
    		
    		if ( !Ext.isEmpty( labelFormActions ) ) {
                for ( var j = 0; j < labelFormActions.length; j++ ) {
                  var labelFormAction = labelFormActions[j];
                  var formAction = {
                    id: labelFormAction.id,
                    target: {
                      type: Ext.ux.suncode.IntegrationComponentService.LABEL_DESTINATION,
                      id: labelAction.labelId
                    },
                    customDescription: labelFormAction.customDescription,
                    condition: labelFormAction.condition,
                    inactive: labelFormAction.inactive,
                    creationDate: labelFormAction.creationDate,
                    modificationDate: labelFormAction.modificationDate,
                    parameters: labelFormAction.parameters
                  };

                  if ( !Ext.isEmpty( labelFormAction.globalId ) ) {
                    formAction = Ext.apply( {
                      globalId: labelFormAction.globalId
                    }, formAction );
                  }

                  formActions.push( formAction );
                }
            }
        }
    }

    return formActions;
}

function xpdlEvents( activityNode, extAttrs ) {
  var variables = new Array();
  var variableSets = new Array();

  Ext.each( activityNode.attributes.form.variables, function( variable, index, allVariables ) {
    if ( variable.genre == 'VARIABLE_SET' ) {
      variableSets.push( variable );
    } else {
      variables.push( variable );
    }
  } );

  var events = new Object();
  events = Ext.apply( events, {
    FORM: xpdlElementEvents( activityNode.attributes.form.events )
  } );
  events = Ext.apply( events, {
    VARIABLE: xpdlElementsEvents( variables, 'varId' )
  } );
  events = Ext.apply( events, {
    GRID: xpdlElementsEvents( variableSets, 'varId' )
  } );
  events = Ext.apply( events, {
    DT_BUTTON: xpdlDtButtonEvents( variableSets )
  } );
  events = Ext.apply( events, {
    ACTION_BUTTON: xpdlElementsEvents( activityNode.attributes.form.httpLinks, 'actionName' )
  } );
  events = Ext.apply( events, {
    ACCEPT_BUTTON: xpdlElementsEvents( activityNode.attributes.form.buttons, 'actionName' )
  } );

  if ( !Ext.isEmpty( events ) ) {
    extAttrs.push( {
      ExtendedAttribute: {
        Name: 'EVENTS',
        Value: Ext.util.JSON.encode( events )
      }
    } );
  }

  return extAttrs;
}

function xpdlElementsEvents( elements, idKey ) {
  var events = new Object();

  if ( !Ext.isEmpty( elements ) ) {
    for ( var i = 0; i < elements.length; i++ ) {
      var element = elements[i];
      var eventActions = element.eventActions;

      if ( !Ext.isEmpty( eventActions ) ) {
        var key = element[idKey];
        events[key] = xpdlElementEvents( eventActions );
      }
    }
  }

  return events;
}

function xpdlElementEvents( eventActions, eventOwnerId ) {
  var events = new Object();

  for ( var i = 0; i < eventActions.length; i++ ) {
    var eventAction = eventActions[i];

    if ( ( Ext.isEmpty( eventOwnerId ) && Ext.isEmpty( eventAction.eventParentId ) )
        || ( !Ext.isEmpty( eventOwnerId ) && eventAction.eventParentId === eventOwnerId ) ) {
      var eventName = eventAction.eventName;
      var eventsByName = events[eventName];

      if ( Ext.isEmpty( eventsByName ) ) {
        eventsByName = new Array();
      }

      var eventByName = {
        id: eventAction.eventOwnerId,
        action: {
          id: eventAction.id,
          customDescription: eventAction.customDescription,
          condition: eventAction.condition,
          inactive: eventAction.inactive,
          creationDate: eventAction.creationDate,
          modificationDate: eventAction.modificationDate,
          parameters: eventAction.parameters
        }
      };
      if ( !Ext.isEmpty( eventAction.globalId ) ) {
        eventByName = Ext.apply( eventByName, {
          globalId: eventAction.globalId
        } );
      }
      var onEvents = xpdlElementEvents( eventActions, eventAction.eventOwnerId );

      if ( Object.keys( onEvents ).length > 0 ) {
        eventByName = Ext.apply( eventByName, {
          on: onEvents
        } );
      }

      eventsByName.push( eventByName );
      events[eventName] = eventsByName;
    }
  }

  return events;
}

function xpdlDtButtonEvents( variableSets ) {
  var events = new Object();

  if ( !Ext.isEmpty( variableSets ) ) {
    for ( var i = 0; i < variableSets.length; i++ ) {
      var variableSet = variableSets[i];

      events = Ext.apply( events, xpdlElementsEvents( variableSet.DTButtons, 'buttonId' ) );
    }
  }

  return events;
}

function xpdlTransitions( processNode ) {
    var transitions = new Array();

    processNode.eachChild( function( activityNode ) {
        transitions = transitions.concat( xpdlTransition( activityNode.hasXpdlSubstitution() ?
        		activityNode.getXpdlSubstitution() : activityNode ) );
    } );

    return transitions;
}

function xpdlTransition( activityNode ) {
    var transitions = new Array();

    var cts = activityNode.attributes.transitions.cond;
    for ( var i = 0; i < cts.length; i++ ) {
        var t = cts[i];
        var condTransition = new Array();
        condTransition.push( {
            From: activityNode.attributes.activityDefId
        } );
        condTransition.push( {
            Id: t.id
        } );
        condTransition.push( {
            To: t.to
        } );
        condTransition.push( {
            Condition: [ {
                Type: 'CONDITION',
                'XPDLNODEVALUE': t.condition
            } ]
        } );

        var extAttrs = new Array();
        var lineBreakPoints = decomposeCondPathToBreakPoints( t.boxPaths[0], t.linePaths[0] );
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'JaWE_GRAPH_BREAK_POINTS',
                Value: lineBreakPoints
            }
        } );
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'JaWE_GRAPH_TRANSITION_STYLE',
                Value: 'NO_ROUTING_ORTHOGONAL'
            }
        } );
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'BOX_DIRECTION',
                Value: t.boxDirection
            }
        } );
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'FIRST_LINE_DIRECTION',
                Value: t.firstLineDirection
            }
        } );
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'SECOND_LINE_DIRECTION',
                Value: t.secondLineDirection
            }
        } );

        var bps = t.boxPaths;
        for ( var j = 0; j < bps.length; j++ ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'BOX_PATH_DEF',
                    Value: bps[j]
                }
            } );
        }

        var lps = t.linePaths;
        for ( var j = 0; j < lps.length; j++ ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'LINE_PATH_DEF',
                    Value: lps[j]
                }
            } );
        }

        condTransition.push( {
            ExtendedAttributes: extAttrs
        } );

        transitions.push( {
            Transition: condTransition
        } );
    }

    var bts = activityNode.attributes.transitions.basic;
    for ( var i = 0; i < bts.length; i++ ) {
        var t = bts[i];
        var basicTransition = new Array();
        basicTransition.push( {
            From: activityNode.attributes.activityDefId
        } );
        basicTransition.push( {
            Id: t.id
        } );
        basicTransition.push( {
            To: t.to
        } );

        var extAttrs = new Array();
        var lineBreakPoints = decomposeBasicPathToBreakPoints( t.linePaths[0] );
        if ( Ext.isEmpty( lineBreakPoints ) ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'JaWE_GRAPH_TRANSITION_STYLE',
                    Value: 'NO_ROUTING_SPLINE'
                }
            } );
        } else {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'JaWE_GRAPH_BREAK_POINTS',
                    Value: lineBreakPoints
                }
            } );
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'JaWE_GRAPH_TRANSITION_STYLE',
                    Value: 'NO_ROUTING_ORTHOGONAL'
                }
            } );
        }
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'FIRST_LINE_DIRECTION',
                Value: t.firstLineDirection
            }
        } );

        var lps = t.linePaths;
        for ( var j = 0; j < lps.length; j++ ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'LINE_PATH_DEF',
                    Value: lps[j]
                }
            } );
        }

        basicTransition.push( {
            ExtendedAttributes: extAttrs
        } );

        transitions.push( {
            Transition: basicTransition
        } );
    }

    var ets = activityNode.attributes.transitions.exception;
    for ( var i = 0; i < ets.length; i++ ) {
        var t = ets[i];
        var deadlineTransition = new Array();
        deadlineTransition.push( {
            From: activityNode.attributes.activityDefId
        } );
        deadlineTransition.push( {
            Id: t.id
        } );
        deadlineTransition.push( {
            To: t.to
        } );
        deadlineTransition.push( {
            Condition: [ {
                Type: 'EXCEPTION',
                'XPDLNODEVALUE': t.exceptionName
            } ]
        } );

        var extAttrs = new Array();
        var lineBreakPoints = decomposeBasicPathToBreakPoints( t.linePaths[0] );
        if ( Ext.isEmpty( lineBreakPoints ) ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'JaWE_GRAPH_TRANSITION_STYLE',
                    Value: 'NO_ROUTING_SPLINE'
                }
            } );
        } else {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'JaWE_GRAPH_BREAK_POINTS',
                    Value: lineBreakPoints
                }
            } );
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'JaWE_GRAPH_TRANSITION_STYLE',
                    Value: 'NO_ROUTING_ORTHOGONAL'
                }
            } );
        }
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'FIRST_LINE_DIRECTION',
                Value: t.firstLineDirection
            }
        } );

        var lps = t.linePaths;
        for ( var j = 0; j < lps.length; j++ ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'LINE_PATH_DEF',
                    Value: lps[j]
                }
            } );
        }

        deadlineTransition.push( {
            ExtendedAttributes: extAttrs
        } );

        transitions.push( {
            Transition: deadlineTransition
        } );
    }

    var ots = activityNode.attributes.transitions.otherwise;
    for ( var i = 0; i < ots.length; i++ ) {
        var t = ots[i];
        var otherwiseTransition = new Array();
        otherwiseTransition.push( {
            From: activityNode.attributes.activityDefId
        } );
        otherwiseTransition.push( {
            Id: t.id
        } );
        otherwiseTransition.push( {
            To: t.to
        } );
        otherwiseTransition.push( {
            Condition: [ {
                Type: 'OTHERWISE'
            } ]
        } );

        var extAttrs = new Array();
        var lineBreakPoints = decomposeBasicPathToBreakPoints( t.linePaths[0] );
        if ( Ext.isEmpty( lineBreakPoints ) ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'JaWE_GRAPH_TRANSITION_STYLE',
                    Value: 'NO_ROUTING_SPLINE'
                }
            } );
        } else {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'JaWE_GRAPH_BREAK_POINTS',
                    Value: lineBreakPoints
                }
            } );
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'JaWE_GRAPH_TRANSITION_STYLE',
                    Value: 'NO_ROUTING_ORTHOGONAL'
                }
            } );
        }
        extAttrs.push( {
            ExtendedAttribute: {
                Name: 'FIRST_LINE_DIRECTION',
                Value: t.firstLineDirection
            }
        } );

        var lps = t.linePaths;
        for ( var j = 0; j < lps.length; j++ ) {
            extAttrs.push( {
                ExtendedAttribute: {
                    Name: 'LINE_PATH_DEF',
                    Value: lps[j]
                }
            } );
        }

        otherwiseTransition.push( {
            ExtendedAttributes: extAttrs
        } );

        transitions.push( {
            Transition: otherwiseTransition
        } );
    }

    return transitions;
}