function newXpdl() {
    if ( Ext.getCmp( 'main_panel' ).isSaved() ) {
        executeNewXpdl();
        return;
    }

    Ext.Msg.show( {
        title: getTranslation( 'Uwaga' ),
        msg: getTranslation( 'Plik jest niezapisany. Zapisać przed wykonaniem operacji?' ),
        buttons: {
            yes: getTranslation( 'Tak' ),
            no: getTranslation( 'Nie' ),
            cancel: getTranslation( 'Anuluj' )
        },
        fn: function( buttonId ) {
            switch ( buttonId ) {
                case 'yes':
                    saveXpdl( executeNewXpdl );
                    break;
                case 'no':
                	  resetAutoSaved();
                    executeNewXpdl();
                    break;
                default:
                    break;
            }
        },
        icon: Ext.Msg.QUESTION
    } );
}

function executeNewXpdl() {
    var packagePanel = Ext.getCmp( 'package_panel' );
    var packageNode = packagePanel.getRootNode();
    packageNode.resetPackage();

    var paperPanel = Ext.getCmp( 'paper_panel' );
    paperPanel.resetPanel();

    var paperToolbar = paperPanel.getTopToolbar();
    paperToolbar.disableAllButtons();
    Ext.getCmp( 'main_panel' ).setSaved( true );
    clearMessageToolbarWithDelay( 0 );
    Ext.ux.suncode.Clipboard.reset();
    Ext.ux.suncode.I18NService.resetAllPackageTranslations();
    Suncode.context( 'pwe' ).openedAutoSavedPath = null;
}

function openXpdl() {
    var win = new Ext.ux.suncode.OpenXpdlWindow();
    win.show();
}

function openRecentFile( item ) {
    var recentFilePath = item.recentFilePath;

    if ( Ext.getCmp( 'main_panel' ).isSaved() ) {
        executeNewXpdl();
        executeOpenRecentFile( recentFilePath );
    } else {
        Ext.Msg.show( {
            title: getTranslation( 'Uwaga' ),
            msg: getTranslation( 'Plik jest niezapisany. Zapisać przed wykonaniem operacji?' ),
            buttons: {
                yes: getTranslation( 'Tak' ),
                no: getTranslation( 'Nie' ),
                cancel: getTranslation( 'Anuluj' )
            },
            fn: function( buttonId ) {
                switch ( buttonId ) {
                    case 'yes':
                        saveXpdl( function() {
                          executeNewXpdl();
                          executeOpenRecentFile( recentFilePath );
                        } );
                        break;
                    case 'no':
                    	resetAutoSaved();
                        executeNewXpdl();
                        executeOpenRecentFile( recentFilePath );
                        break;
                    default:
                        break;
                }
            },
            icon: Ext.Msg.QUESTION
        } );
    }
}

function executeOpenRecentFile( recentFilePath ) {
    var formPanel = new Ext.FormPanel( {
        renderTo: document.body,
        hidden: true,
        items: [ {
            xtype: 'textfield',
            name: 'path',
            value: recentFilePath
        } ]
    } );
    var form = formPanel.getForm();
    form.submit( {
        url: 'api/xpdl/openRecentFile',
        waitMsg: getTranslation( 'Trwa przetwarzanie pliku...' ),
        waitTitle: getTranslation( 'Proszę czekać' ),
        method: 'POST',
        timeout: 600,
        success: function( form, action ) {
            if ( action.result.success ) {
                var mainPanel = Ext.getCmp( 'main_panel' );
            	  executeOpen( action.result );
                loadRecentFiles();

                if ( Ext.ux.suncode.XpdlErrors.hasErrors() ) {
                    var xpdlErrorsWindow = new Ext.ux.suncode.XpdlErrorsWindow();
                    xpdlErrorsWindow.show();
                }

                if ( mainPanel.isCurrentXpdlVersionOlder() ) {
                  showWarn( getTranslation( 'Mapa pakietu została utworzona w edytorze w wersji' )
                      + ' ' + mainPanel.getCurrentXpdlVersion() + '. '
                      + getTranslation( 'Aktualna wersja edytora to' )
                      + ' ' + mainPanel.getVersion() + '.' );
                }
            } else {
                showWarn( getTranslation( 'Wystąpił błąd.' ) );
            }
        },
        failure: function( form, action ) {
        	showServerFailure( action.response );
        }
    } );

    var task = new Ext.util.DelayedTask( function() {
        formPanel.destroy();
    }, this );
    task.delay( 3000 );
}

function executeOpen( result ) {
	var mainPanel = Ext.getCmp( 'main_panel' );
	mainPanel.suspendAutoSave();
	
	try {
    if ( !Ext.isEmpty( result.xpdlTranslations ) ) {
      Ext.ux.suncode.I18NService.loadAllPackageTranslations(
          Ext.util.JSON.decode( decodeXpdlTranslations( result.xpdlTranslations ) ) );
    }
		readXpdl( Ext.util.JSON.decode( result.xpdl ) );
    if ( !Ext.isEmpty( result.formTemplates ) ) {
      readFormTemplates( Ext.util.JSON.decode( result.formTemplates ) );
    }
    initMissingFormTemplates();
    if ( !Ext.isEmpty( result.processSpecifications ) ) {
      Ext.ux.suncode.DocumentationService.setProcessSpecifications( Ext.util.JSON.decode( result.processSpecifications ) );
    }
    if ( !Ext.isEmpty( result.formPreviewProcessDataSets ) ) {
      Ext.ux.suncode.FormPreviewService.setDataSets( Ext.util.JSON.decode( result.formPreviewProcessDataSets ) );
    }
    if ( !Ext.isEmpty( result.processGlobalSettings ) ) {
      Ext.ux.suncode.GlobalSettingsService.setGlobalSettings( Ext.util.JSON.decode( result.processGlobalSettings ) );
    }
    Ext.ux.suncode.I18NService.fixTranslations();
    if ( result.additionalsMissing ) {
      showWarn( getTranslation( 'Mapa pakietu nie zawiera szablonów formularzy i specyfikacji procesów.' ) );
    }
    mainPanel.setSaved( true );
	} finally {
    mainPanel.resumeAutoSave();
	}
}

function openExample( item ) {
    var examplePath = item.examplePath;

    if ( Ext.getCmp( 'main_panel' ).isSaved() ) {
        executeNewXpdl();
        executeOpenExample( examplePath );
    } else {
        Ext.Msg.show( {
            title: getTranslation( 'Uwaga' ),
            msg: getTranslation( 'Plik jest niezapisany. Zapisać przed wykonaniem operacji?' ),
            buttons: {
                yes: getTranslation( 'Tak' ),
                no: getTranslation( 'Nie' ),
                cancel: getTranslation( 'Anuluj' )
            },
            fn: function( buttonId ) {
                switch ( buttonId ) {
                    case 'yes':
                        saveXpdl( function() {
                          executeNewXpdl();
                          executeOpenExample( examplePath );
                        } );
                        break;
                    case 'no':
                    	resetAutoSaved();
                        executeNewXpdl();
                        executeOpenExample( examplePath );
                        break;
                    default:
                        break;
                }
            },
            icon: Ext.Msg.QUESTION
        } );
    }
}

function executeOpenExample( examplePath ) {
    var formPanel = new Ext.FormPanel( {
        renderTo: document.body,
        hidden: true,
        items: [ {
            xtype: 'textfield',
            name: 'path',
            value: examplePath
        } ]
    } );
    var form = formPanel.getForm();
    form.submit( {
        url: 'api/xpdl/openExample',
        waitMsg: getTranslation( 'Trwa przetwarzanie pliku...' ),
        waitTitle: getTranslation( 'Proszę czekać' ),
        method: 'POST',
        success: function( form, action ) {
            if ( action.result.success ) {
            	executeOpen( action.result );
            } else {
                showWarn( getTranslation( 'Wystąpił błąd.' ) );
            }
        },
        failure: function( form, action ) {
        	showServerFailure( action.response );
        }
    } );

    var task = new Ext.util.DelayedTask( function() {
        formPanel.destroy();
    }, this );
    task.delay( 3000 );
}

function saveXpdl( afterSaveFunction, afterSaveScope ) {
	executeSave( 'api/xpdl/save', function( response, opts ) {
		if ( parseBoolean( response.responseText ) ) {
            notifyMessageToolbar( getTranslation( 'Zapis zakończony sukcesem' ), true );
            clearMessageToolbarWithDelay( 3 );
            loadRecentFiles();
            Suncode.context( 'pwe' ).openedAutoSavedPath = null;
            Ext.getCmp( 'main_panel' ).setSaved( true );

            if ( Ext.isFunction( afterSaveFunction ) ) {
              var scope = !Ext.isEmpty( afterSaveScope ) ? afterSaveScope : window;
              afterSaveFunction.apply( scope, [] );
            }
        } else {
            showWarn( getTranslation( 'Wystąpił błąd zapisu.' ) );
        }
	}, function( response, opts ) {
		showWarn( getTranslation( 'Wystąpił błąd.' ) );
	} );
}

function autoSave() {
	executeSave( 'api/xpdl/autoSave', function( response, opts ) {
		if ( !parseBoolean( response.responseText ) ) {
			showWarn( getTranslation( 'Problem z komunikacją z serwerem. Zmiany mogą zostać utracone.' ) );
		}
	}, function( response, opts ) {
		showWarn( getTranslation( 'Problem z komunikacją z serwerem. Zmiany mogą zostać utracone.' ) );
	} );
}

function executeSave( url, successFunc, failureFunc ) {
    lockHotkeys();
	  var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();
    var xpdl = Ext.util.JSON.encode( writeXpdl( packageNode ) );
    var formTemplates = Ext.util.JSON.encode( Ext.ux.suncode.FormTemplateWriter.write( packageNode ) );
    var processSpecifications = Ext.util.JSON.encode( packageNode.getProcessSpecifications() );
    var formPreviewProcessDataSets = Ext.util.JSON.encode( packageNode.getFormPreviewProcessDataSets() );
    var processGlobalSettings = Ext.util.JSON.encode( packageNode.getProcessGlobalSettings() );
    var xpdlTranslations = Ext.util.JSON.encode( Ext.ux.suncode.I18NService.getAllPackageTranslations() );
    var attachmentDirectories = Ext.util.JSON.encode( packageNode.getAttachmentDirectories() );

    Ext.Ajax.request( {
        url: url,
        method: 'POST',
        params: {
            xpdl: LZString.compressToBase64( xpdl ),
            formTemplates: LZString.compressToBase64( formTemplates ),
            processSpecifications: LZString.compressToBase64( processSpecifications ),
            formPreviewProcessDataSets: LZString.compressToBase64( formPreviewProcessDataSets ),
            processGlobalSettings: LZString.compressToBase64( processGlobalSettings ),
            packageId: packageNode.attributes.packageId,
            fileName: packageNode.attributes.packageName,
            xpdlTranslations: LZString.compressToBase64( xpdlTranslations ),
            moduleStartTime: Suncode.context( 'pwe' ).moduleStartTime,
            openedAutoSavedPath: Suncode.context( 'pwe' ).openedAutoSavedPath,
            attachmentDirectories: attachmentDirectories
        },
        success: function( response, opts ) {
        	if ( Ext.isFunction( successFunc ) ) {
            unlockHotkeys();
        		successFunc.apply( this, [ response, opts ] );
        	}
        },
        failure: function( response, opts ) {
        	if ( Ext.isFunction( failureFunc ) ) {
            unlockHotkeys();
        		failureFunc.apply( this, [ response, opts ] );
        	}
        }
    } );
}

function resetAutoSaved() {
	Ext.Ajax.request( {
        url: 'api/xpdl/resetAutoSaved',
        method: 'GET',
        params: {
        	moduleStartTime: Suncode.context( 'pwe' ).moduleStartTime
        },
        failure: function( response, opts ) {
        	showWarn( getTranslation( 'Wystąpił błąd.' ) );
        }
    } );
}

function showXpdlPreview() {
    var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();
    var xpdl = Ext.util.JSON.encode( writeXpdl( packageNode ) );

    executeGetXpdlPreview( {
        xpdl: xpdl,
        method: 'POST',
        successFunction: function( preview ) {
        	var win = new Ext.ux.suncode.XpdlPreviewWindow( {
                xpdlPreview: preview
            } );
            win.show();
        }
    } );
}

function executeGetXpdlPreview( config ) {
    var maskId = 'show_xpdl_preview_mask';
    showLoadingMask( maskId, getTranslation( 'Trwa generowanie podglądu...' ) );
    var params = new Object();
    
    if ( !Ext.isEmpty( config.xpdl ) ) {
    	params = Ext.apply( {
            xpdl: LZString.compressToBase64( config.xpdl )
    	}, params );
    } else {
    	params = Ext.apply( {
    		packageId: config.packageId,
            packageVersion: config.packageVersion
    	}, params );
    }

    Ext.Ajax.request( {
        url: 'api/xpdl/getPreview',
        method: config.method,
        params: params,
        success: function( response, opts ) {
            removeLoadingMask( maskId );
            var successFunction = config.successFunction;
            
            if ( Ext.isFunction( successFunction ) ) {
        		var successScope = !Ext.isEmpty( config.successScope ) ? config.successScope : window;
        		successFunction.apply( successScope, [ response.responseText ] );
        	}
        },
        failure: function( response, opts ) {
            removeLoadingMask( maskId );
            showWarn( getTranslation( 'Wystąpił błąd.' ) );
        }
    } );
}

function generateProcessMapImage() {
    var processPanel = Ext.getCmp( 'paper_panel' ).getActiveTab();

    if ( !Ext.isEmpty( processPanel ) ) {
        Ext.ux.suncode.ProcessMapImageService.generateImage( processPanel );
    } else {
        showWarn( getTranslation( 'Wybierz proces.' ) );
    }
}

function generateXpdl( config ) {
	  var mainPanel = Ext.getCmp( 'main_panel' );
    var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();
    var xpdl = writeXpdl( packageNode, config.processDefId );
    var packageId = packageNode.attributes.packageId;
    var packageName = packageNode.attributes.packageName;
    var fileName = getXpdlPackageNameTranslation( packageId, packageName );
    
    if ( !Ext.isEmpty( config.processName ) ) {
    	fileName += ' - ' + getXpdlProcessNameTranslation( config.processDefId, config.processName );
    }
    
    var form = jQuery( '<form/>', { action: 'api/xpdl/generate', method:'post' } );
    form.append( jQuery( '<input>', { type: 'text', name: 'fileName', value: fileName } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'xpdl', value: LZString.compressToBase64( Ext.util.JSON.encode( xpdl ) ) } ) );
    
    var iframe = jQuery( '#download-frame' );
    iframe.empty();
    iframe.append( form );
    
    mainPanel.setCheckSavedOnUnload( false );
    
    form.submit();
    
    var task = new Ext.util.DelayedTask( function() {
        mainPanel.setCheckSavedOnUnload( true );
    }, this );
    task.delay( 1000 );
}

function generateJavaClasses() {
    var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.clearJavaClasses();
    writeXpdl( packageNode );

    var packageId = packageNode.attributes.packageId;
    var packageName = packageNode.attributes.packageName;
    var fileName = getXpdlPackageNameTranslation( packageId, packageName );

    var form = jQuery( '<form/>', { action: 'api/javacode/generate', method:'post' } );
    form.append( jQuery( '<input>', { type: 'text', name: 'fileName', value: fileName } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'javaClasses', value: Ext.util.JSON.encode( mainPanel.javaClasses ) } ) );
    
    var iframe = jQuery( '#download-frame' );
    iframe.empty();
    iframe.append( form );
    
    mainPanel.setCheckSavedOnUnload( false );
    
    form.submit();
    
    var task = new Ext.util.DelayedTask( function() {
        mainPanel.setCheckSavedOnUnload( true );
    }, this );
    task.delay( 1000 );
}

function generateCustomJs() {
    var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.clearCustomJS();
    writeXpdl( packageNode );
    
    var form = jQuery( '<form/>', { action: 'api/jscode/generate', method:'post' } );
    form.append( jQuery( '<input>', { type: 'text', name: 'packageId', value: packageNode.attributes.packageId } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'customJs', value: Ext.util.JSON.encode( mainPanel.customJS ) } ) );
    
    var iframe = jQuery( '#download-frame' );
    iframe.empty();
    iframe.append( form );
    
    mainPanel.setCheckSavedOnUnload( false );
    
    form.submit();
    
    var task = new Ext.util.DelayedTask( function() {
        mainPanel.setCheckSavedOnUnload( true );
    }, this );
    task.delay( 1000 );
}

function generateProcessDocumentation() {
    var processPanel = Ext.getCmp( 'paper_panel' ).getActiveTab();

    if ( !Ext.isEmpty( processPanel ) ) {
        var win = new Ext.ux.suncode.DocumentationConfigurationWindow( {
          processDefId: processPanel.processDefId
        } );
        win.show();
    } else {
        showWarn( getTranslation( 'Wybierz proces.' ) );
    }
}

function generateZip( config ) {
	  var processDefId = config.processDefId;
    var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.clearJavaClasses();
    mainPanel.clearCustomJS();
    var xpdl = writeXpdl( packageNode, processDefId );
    var formTemplates = Ext.ux.suncode.FormTemplateWriter.write( packageNode, processDefId );
    var processSpecifications = packageNode.getProcessSpecifications( processDefId );
    var formPreviewProcessDataSets = packageNode.getFormPreviewProcessDataSets( processDefId );
    var processGlobalSettings = packageNode.getProcessGlobalSettings( processDefId );
    var attachmentDirectories = packageNode.getAttachmentDirectories( processDefId );
    var xpdlTranslations = Ext.ux.suncode.I18NService.getAllPackageTranslations( processDefId );
    var javaClasses = mainPanel.javaClasses;
    var customJs = mainPanel.customJS
    var packageId = packageNode.attributes.packageId;
    var packageName = packageNode.attributes.packageName;
    var fileName = getXpdlPackageNameTranslation( packageId, packageName );

    if ( !Ext.isEmpty( config.processName ) ) {
    	fileName += ' - ' + getXpdlProcessNameTranslation( processDefId, config.processName );
    }
    
    if ( !Ext.isEmpty( processDefId ) ) {
    	xpdlTranslations = Ext.ux.suncode.I18NService.copyTranslationsWithPackageIdChange( xpdlTranslations,
    			packageId, generateId( packageId + '_' + processDefId, Ext.ux.suncode.Constants.PACKAGE_ID_MAX_LENGTH ) );
    }

    var form = jQuery( '<form/>', { action: 'api/xpdl/generateZip', method:'post' } );
    form.append( jQuery( '<input>', { type: 'text', name: 'fileName', value: fileName } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'packageId', value: packageId } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'xpdl', value: LZString.compressToBase64( Ext.util.JSON.encode( xpdl ) ) } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'formTemplates', value: LZString.compressToBase64( Ext.util.JSON.encode( formTemplates ) ) } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'processSpecifications', value: LZString.compressToBase64( Ext.util.JSON.encode( processSpecifications ) ) } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'formPreviewProcessDataSets', value: LZString.compressToBase64( Ext.util.JSON.encode( formPreviewProcessDataSets ) ) } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'processGlobalSettings', value: LZString.compressToBase64( Ext.util.JSON.encode( processGlobalSettings ) ) } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'xpdlTranslations', value: LZString.compressToBase64( Ext.util.JSON.encode( xpdlTranslations ) ) } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'javaClasses', value: LZString.compressToBase64( Ext.util.JSON.encode( javaClasses ) ) } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'customJs', value: LZString.compressToBase64( Ext.util.JSON.encode( customJs ) ) } ) );
    form.append( jQuery( '<input>', { type: 'text', name: 'attachmentDirectories', value: Ext.util.JSON.encode( attachmentDirectories ) } ) );
    
    var iframe = jQuery( '#download-frame' );
    iframe.empty();
    iframe.append( form );
    
    mainPanel.setCheckSavedOnUnload( false );
    
    form.submit();
    
    var task = new Ext.util.DelayedTask( function() {
        mainPanel.setCheckSavedOnUnload( true );
    }, this );
    task.delay( 2000 );
}

function validateMissingIntegrationComponents( successFunction ) {
  var missingIntegrationComponents = Ext.ux.suncode.IntegrationComponentService.getMissingIntegrationComponents();

  if ( !Ext.isEmpty( missingIntegrationComponents ) ) {
    var warning = getTranslation( 'Pakiet zawiera komponenty integracyjne, których nie ma w systemie.' )
      + ' ' + getTranslation( 'Czy chcesz kontynuować?' );

    Ext.Msg.show( {
      title: '<font weight="bold">' + getTranslation( 'Uwaga' ) + '</font>',
      msg: warning,
      buttons: {
        yes: getTranslation( 'Kontynuuj' ),
        no: getTranslation( 'Pokaż brakujące komponenty' ),
        cancel: getTranslation( 'Anuluj' )
      },
      fn: function( buttonId ) {
        switch ( buttonId ) {
          case 'yes':
            if ( Ext.isFunction( successFunction ) ) {
              successFunction.apply( this, [] );
            }
            break;
          case 'no':
            var win = new Ext.ux.suncode.MissingIntegrationComponentsWindow( {
              missingComponents: missingIntegrationComponents
            } );
            win.show();
            break;
          default:
            break;
        }
      },
      icon: Ext.Msg.WARNING,
      scope: this
    } );
  } else {
    if ( Ext.isFunction( successFunction ) ) {
      successFunction.apply( this, [] );
    }
  }
}

function exportXpdl() {
  var mainPanel = Ext.getCmp( 'main_panel' );

  if ( mainPanel.isProductionSystemType() ) {
    Ext.Ajax.request( {
      url: 'api/packageinfotemplate/get',
      method: 'GET',
      success: function( response, opts ) {
        var packageInfoTemplate = Ext.util.JSON.decode( response.responseText );

        showPackageExportWindow( packageInfoTemplate );
      },
      failure: function( response, opts ) {
        showWarn( getTranslation( 'Wystąpił błąd.' ) );
      }
    } );
  } else {
    showPackageExportWindow( {
      content: '',
      validate: false
    } );
  }
}

function showPackageExportWindow( packageInfoTemplate ) {
  var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();

  var win = new Ext.ux.suncode.PackageExportWindow( {
    packageId: packageNode.attributes.packageId,
    packageName: packageNode.attributes.packageName,
    packageNoOfProcesses: packageNode.childNodes.length,
    packageInfoTemplate: packageInfoTemplate,
    skipExport: false
  } );
  win.show();
}

function executeExportXpdl( config ) {
    var maskId = 'xpdl_export_mask';
    showLoadingMask( maskId, getTranslation( 'Trwa eksportowanie...' ) );

    var mainPanel = Ext.getCmp( 'main_panel' );
    var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();
    var xpdl = Ext.util.JSON.encode( writeXpdl( packageNode ) );
    var formTemplates = Ext.util.JSON.encode( Ext.ux.suncode.FormTemplateWriter.write( packageNode ) );
    var autoJs = Ext.util.JSON.encode( mainPanel.autoJS );
    var customJs = Ext.util.JSON.encode( mainPanel.customJS );
    var javaClasses = Ext.util.JSON.encode( mainPanel.javaClasses );
    var processSpecifications = Ext.util.JSON.encode( packageNode.getProcessSpecifications() );
    var formPreviewProcessDataSets = Ext.util.JSON.encode( packageNode.getFormPreviewProcessDataSets() );
    var processGlobalSettings = Ext.util.JSON.encode( packageNode.getProcessGlobalSettings() );
    var xpdlTranslations = Ext.util.JSON.encode( Ext.ux.suncode.I18NService.getAllPackageTranslations() );
    var attachmentDirectories = packageNode.getAttachmentDirectories();

    Ext.Ajax.request( {
        url: 'api/xpdl/export',
        method: 'POST',
        params: {
            fileName: config.packageName,
            packageId: config.packageId,
            xpdl: LZString.compressToBase64( xpdl ),
            formTemplates: LZString.compressToBase64( formTemplates ),
            autoJs: LZString.compressToBase64( autoJs ),
            customJs: LZString.compressToBase64( customJs ),
            javaClasses: LZString.compressToBase64( javaClasses ),
            packageInfo: LZString.compressToBase64( config.packageInfo ),
            processSpecifications: LZString.compressToBase64( processSpecifications ),
            formPreviewProcessDataSets: LZString.compressToBase64( formPreviewProcessDataSets ),
            processGlobalSettings: LZString.compressToBase64( processGlobalSettings ),
            xpdlTranslations: LZString.compressToBase64( xpdlTranslations ),
            moduleStartTime: Suncode.context( 'pwe' ).moduleStartTime,
            openedAutoSavedPath: Suncode.context( 'pwe' ).openedAutoSavedPath,
            attachmentDirectories: attachmentDirectories
        },
        timeout: 1800000,
        success: function( response, opts ) {
            removeLoadingMask( maskId );
            var packageLoad = Ext.util.JSON.decode( response.responseText );

            if ( packageLoad.loaded ) {
            	Suncode.context( 'pwe' ).openedAutoSavedPath = null;
                var successMessage = getTranslation( 'Pakiet o nazwie' );
                successMessage += ' ' + config.packageName + ' ';
                successMessage += getTranslation( 'został załadowany.' );

                Ext.Msg.show( {
                    title: getTranslation( 'Eksport zakończony sukcesem' ),
                    msg: successMessage,
                    buttons: Ext.Msg.OK,
                    icon: Ext.Msg.INFO
                } );
            } else if ( packageLoad.shouldRefresh ) {
                config = Ext.apply( {
                    xpdl: xpdl,
                    formTemplates: formTemplates,
                    autoJs: autoJs,
                    customJs: customJs,
                    javaClasses: javaClasses,
                    packageVersion: packageLoad.currentVersion,
                    processSpecifications: processSpecifications,
                    formPreviewProcessDataSets: formPreviewProcessDataSets,
                    processGlobalSettings: processGlobalSettings,
                    xpdlTranslations: xpdlTranslations
                }, config );

                refreshXpdl( config );
            } else {
                showPackageExportException( packageLoad );
            }
        },
        failure: function( response, opts ) {
            removeLoadingMask( maskId );

            if ( response.isTimeout ) {
              showWarn( getTranslation( 'Ładowanie pakietu może wciąż trwać. Nie należy restartować systemu.' ) );
            } else {
              showWarn( getTranslation( 'Wystąpił błąd.' ) );
            }
        }
    } );
}

function refreshXpdl( config ) {
    var msg = getTranslation( 'Pakiet już istnieje w wersji' );
    msg += ' ' + config.packageVersion + '. ';
    msg += getTranslation( 'Odświeżyć pakiet?' );

    Ext.Msg.show( {
        title: getTranslation( 'Uwaga' ),
        msg: msg,
        buttons: {
            yes: getTranslation( 'Tak' ),
            no: getTranslation( 'Nie' )
        },
        fn: function( buttonId ) {
            if ( buttonId == 'yes' ) {
                executeRefreshXpdl( config );
            }
        },
        icon: Ext.Msg.QUESTION
    } );
}

function executeRefreshXpdl( config ) {
    var maskId = 'xpdl_refresh_mask';
    showLoadingMask( maskId, getTranslation( 'Trwa odświeżanie...' ) );
    var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();
    var attachmentDirectories = packageNode.getAttachmentDirectories();

    Ext.Ajax.request( {
        url: 'api/xpdl/refresh',
        method: 'POST',
        params: {
            packageId: config.packageId,
            packageVersion: config.packageVersion,
            xpdl: LZString.compressToBase64( config.xpdl ),
            formTemplates: LZString.compressToBase64( config.formTemplates ),
            autoJs: LZString.compressToBase64( config.autoJs ),
            customJs: LZString.compressToBase64( config.customJs ),
            javaClasses: LZString.compressToBase64( config.javaClasses ),
            packageInfo: LZString.compressToBase64( config.packageInfo ),
            processSpecifications: LZString.compressToBase64( config.processSpecifications ),
            formPreviewProcessDataSets: LZString.compressToBase64( config.formPreviewProcessDataSets ),
            processGlobalSettings: LZString.compressToBase64( config.processGlobalSettings ),
            xpdlTranslations: LZString.compressToBase64( config.xpdlTranslations ),
            moduleStartTime: Suncode.context( 'pwe' ).moduleStartTime,
            openedAutoSavedPath: Suncode.context( 'pwe' ).openedAutoSavedPath,
            attachmentDirectories: attachmentDirectories
        },
        timeout: 1800000,
        success: function( response, opts ) {
            removeLoadingMask( maskId );
            var packageLoad = Ext.util.JSON.decode( response.responseText );

            if ( packageLoad.loaded ) {
            	Suncode.context( 'pwe' ).openedAutoSavedPath = null;
            	
                if ( config.nodeToReload ) {
                    config.nodeToReload.reload();
                }

                var successMessage = getTranslation( 'Pakiet o nazwie' );
                successMessage += ' ' + config.packageName + ' ';
                successMessage += getTranslation( 'został odświeżony.' );

                Ext.Msg.show( {
                    title: getTranslation( 'Eksport zakończony sukcesem' ),
                    msg: successMessage,
                    buttons: Ext.Msg.OK,
                    icon: Ext.Msg.INFO
                } );
            } else {
                showPackageExportException( packageLoad );
            }
        },
        failure: function( response, opts ) {
            removeLoadingMask( maskId );

            if ( response.isTimeout ) {
              showWarn( getTranslation( 'Odświeżanie pakietu może wciąż trwać. Nie należy restartować systemu.' ) );
            } else {
              showWarn( getTranslation( 'Wystąpił błąd.' ) );
            }
        }
    } );
}

function exit() {
    if ( Ext.getCmp( 'main_panel' ).isSaved() ) {
        exitProgram();
        return;
    }

    Ext.Msg.show( {
        title: getTranslation( 'Uwaga' ),
        msg: getTranslation( 'Plik jest niezapisany. Zapisać przed wykonaniem operacji?' ),
        buttons: {
            yes: getTranslation( 'Tak' ),
            no: getTranslation( 'Nie' ),
            cancel: getTranslation( 'Anuluj' )
        },
        fn: function( buttonId ) {
            switch ( buttonId ) {
                case 'yes':
                    saveXpdl( exitProgram );
                    break;
                case 'no':
                	resetAutoSaved();
                    exitProgram();
                    break;
                default:
                    break;
            }
        },
        icon: Ext.Msg.QUESTION
    } );
}

function exportJavaClasses() {
    var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();
    writeXpdl( packageNode );
    var mainPanel = Ext.getCmp( 'main_panel' );

    if ( !Ext.isEmpty( mainPanel.javaClasses ) ) {
        var maskId = 'java_classes_export_mask';
        showLoadingMask( maskId, getTranslation( 'Trwa eksportowanie kodów źródłowych...' ) );

        Ext.Ajax.request( {
            url: 'api/javacode/loadToSystem',
            method: 'POST',
            params: {
                javaClasses: Ext.util.JSON.encode( mainPanel.javaClasses )
            },
            timeout: 180000,
            success: function( response, opts ) {
                removeLoadingMask( maskId );

                Ext.Msg.show( {
                    title: getTranslation( 'Eksport zakończony sukcesem' ),
                    msg: getTranslation( 'Wyeksportowano kody źródłowe.' ),
                    buttons: Ext.Msg.OK,
                    icon: Ext.Msg.INFO
                } );
            },
            failure: function( response, opts ) {
                removeLoadingMask( maskId );
                showWarn( getTranslation( 'Wystąpił błąd.' ) );
            }
        } );
    } else {
        showWarn( getTranslation( 'Brak kodów źródłowych.' ) );
    }
}

function exportPackageTranslations() {
	var maskId = 'package_translations_export_mask';
    showLoadingMask( maskId, getTranslation( 'Trwa eksportowanie tłumaczeń...' ) );

    Ext.Ajax.request( {
        url: 'api/xpdl/loadTranslationsToSystem',
        method: 'POST',
        params: {
        	xpdlTranslations: Ext.util.JSON.encode( Ext.ux.suncode.I18NService.getAllPackageTranslations() )
        },
        timeout: 180000,
        success: function( response, opts ) {
            removeLoadingMask( maskId );

            Ext.Msg.show( {
                title: getTranslation( 'Eksport zakończony sukcesem' ),
                msg: getTranslation( 'Wyeksportowano tłumaczenia.' ),
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.INFO
            } );
        },
        failure: function( response, opts ) {
            removeLoadingMask( maskId );
            showWarn( getTranslation( 'Wystąpił błąd.' ) );
        }
    } );
}

function reloadIntegrationComponents() {
	Ext.getCmp( 'main_panel' ).loadIntegrationComponents( new Ext.ux.suncode.RunnableTask( {
        handler: function() {
        	var applicationsButton = Ext.getCmp( 'draw_integration_application_button' );
        	if ( !Ext.isEmpty( applicationsButton ) ) {
        		var categoriesItem = applicationsButton.menu.categoriesItem;
        		applicationsButton.menu.remove( categoriesItem );
        		applicationsButton.menu.add( new Ext.ux.suncode.IntegrationComponentTabPanel( {
                    getCategoriesFunction: function() {
                        return Ext.getCmp( 'main_panel' ).getApplicationsIntegrationComponentCategories();
                    },
                    saveFunction: function( component ) {
                        Ext.ux.suncode.IntegrationComponentService.startDrawingApplication( component );
                    }
                } ) );
        	}
        	
        	var successDialog = Ext.Msg.show( {
                title: getTranslation( 'Sukces' ),
                msg: getTranslation( 'Odświeżono komponenty integracyjne.' ),
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.INFO
            } ).getDialog();
            successDialog.toFront();
            closeDialogOnMaskClick( successDialog );
        },
        scope: this,
        args: []
    } ) );
}

function showIntegrationComponentsPlugins() {
	var win = new Ext.ux.suncode.IntegrationComponentsPluginsWindow();
	win.show();
}

function showMissingIntegrationComponents() {
  var missingIntegrationComponents = Ext.ux.suncode.IntegrationComponentService.getMissingIntegrationComponents();

	var win = new Ext.ux.suncode.MissingIntegrationComponentsWindow( {
    missingComponents: missingIntegrationComponents
  } );
	win.show();
}

function changeShowDeprecatedIntegrationComponents() {
	var item = Ext.getCmp( 'show_deprecated_integration_components_main_menu_item' );
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.changeShowDeprecatedIntegrationComponents();

    if ( mainPanel.getShowDeprecatedIntegrationComponents() ) {
        item.setIconClass( 'x-Module-tickIcon' );
    } else {
        item.setIconClass( 'x-Module-crossIcon' );
    }

    updateUserConfig();
}

function validateXpdl() {
    executeValidateXpdl( {
        doExportXpdl: false,
        doExportPackageTranslations: false
    } );
}

function executeValidateXpdl( config ) {
    var maskId = 'xpdl_validation_mask';
    showLoadingMask( maskId, getTranslation( 'Trwa walidacja...' ) );
    var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();

    Ext.Ajax.request( {
        url: 'api/xpdl/validate',
        method: 'POST',
        params: {
            xpdl: LZString.compressToBase64( Ext.util.JSON.encode( writeXpdl( packageNode ) ) )
        },
        timeout: 180000,
        success: function( response, opts ) {
            removeLoadingMask( maskId );
            var xpdlValidation = Ext.util.JSON.decode( response.responseText );
            var variablesUsageValidation = validateVariablesUsage();

            if ( xpdlValidation.correct && variablesUsageValidation.correct ) {
                if ( config.doExportXpdl ) {
                  validateMissingIntegrationComponents( exportXpdl );
                } else if ( config.doExportPackageTranslations ) {
                  exportPackageTranslations();
                } else {
                    notifyMessageToolbar( getTranslation( 'Walidacja zakończona sukcesem' ), true );
                    clearMessageToolbarWithDelay( 5 );

                    var successDialog = Ext.Msg.show( {
                        title: getTranslation( 'Walidacja zakończona sukcesem' ),
                        msg: getTranslation( 'Walidacja zakończona sukcesem' ) + '.',
                        buttons: Ext.Msg.OK,
                        icon: Ext.Msg.INFO
                    } ).getDialog();
                    successDialog.toFront();
                    closeDialogOnMaskClick( successDialog );
                }
            } else if ( Ext.isObject( xpdlValidation.validationError ) ) {
              markValidationError( xpdlValidation.validationError );
            } else {
              markValidationError( variablesUsageValidation.validationError );
            }
        },
        failure: function( response, opts ) {
            removeLoadingMask( maskId );
            showWarn( getTranslation( 'Wystąpił błąd.' ) );
        }
    } );
}

function validateVariablesUsage() {
  var validation = {
    correct: true
  };

  var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();

  if ( packageNode.hasChildNodes() ) {
    packageNode.eachChild(function ( processNode ) {
      if ( processNode.hasChildNodes() ) {
        processNode.eachChild(function ( activityNode ) {
          var validationError = {
            errorLevel: 'form',
            processDefId: processNode.attributes.processDefId,
            activityDefId: activityNode.attributes.activityDefId,
            validationMessage: getTranslation( 'Akcje formularza/Zdarzenia zawierają zmienne nieistniejące na formularzu' )
          };

          if ( !activityNode.validateVariablesUsage() ) {
            validation = Ext.apply( validation, {
              correct: false,
              validationError: validationError
            } );
            return false;
          }
        } );
      }

      if ( Ext.isObject( validation.validationError ) ) {
        return false;
      }
    } );
  }

  return validation;
}

function markValidationError( validationError ) {
    switch ( validationError.errorLevel ) {
        case 'global':
        case 'process':
        case 'variable':
        case 'table':
        case 'globalRole':
            notifyMessageToolbar( getTranslation( validationError.validationMessage ), false );
            clearMessageToolbarWithDelay( 5 );
            var warnMsg = getTranslation( validationError.validationMessage );
            if ( validationError.errorLevel == 'process' ) {
                changeProcessTab( validationError.processDefId );
            }
            if ( validationError.errorLevel == 'variable' ) {
                changeProcessTab( validationError.processDefId );
                warnMsg += '<br>' + getTranslation( 'Identyfikator' ) + ': ' + validationError.variableId + '.';
            }
            if ( validationError.errorLevel == 'table' ) {
              changeProcessTab( validationError.processDefId );
              warnMsg += '<br>' + getTranslation( 'Identyfikator' ) + ': ' + validationError.tableId + '.';
            }
            if ( validationError.errorLevel == 'globalRole' ) {
                warnMsg += '<br>' + getTranslation( 'Identyfikator' ) + ': ' + validationError.roleId + '.';
            }
            showWarn( warnMsg );
            break;
        case 'role':
            var tab = changeProcessTab( validationError.processDefId );
            var paper = tab.paper;
            var roleHeader = null;
            var set = paper.bpmn.findMapObjectElements( paper, 'roleId', validationError.roleId );

            for ( var i = 0; i < set.length; i++ ) {
                var setType = set[i].data( 'setType' );

                if ( set[i].type == 'path' && ( setType == 'roleHeader' || setType == 'systemHeader' ) ) {
                    roleHeader = set[i];
                    break;
                }
            }

            var headerBox = Raphael.pathBBox( roleHeader.attr( 'path' ) );
            var roleContainer = roleHeader.data( 'roleContainer' );
            var containerBox = Raphael.pathBBox( roleContainer.attr( 'path' ) );

            var animDuration = Ext.getCmp( 'main_panel' ).getAnimationsOn() ? 500 : 1;
            var cr = paper.bpmn.participantCornerRadius;
            var headerFrame = paper.bpmn.roundedRectangle( paper, headerBox.x, headerBox.y, headerBox.width, headerBox.height, cr, 0, 0, cr );
            headerFrame.attr( {
                stroke: '#FFFFFF',
                'stroke-width': 1
            } );
            headerFrame.animate( {
                stroke: '#E16E6E',
                'stroke-width': 3
            }, animDuration );
            paper.validationTrash.push( headerFrame );

            var ctrFrame = paper.bpmn.roundedRectangle( paper, containerBox.x, containerBox.y, containerBox.width, containerBox.height, 0, cr, cr, 0 );
            ctrFrame.attr( {
                stroke: '#FFFFFF',
                'stroke-width': 1
            } );
            ctrFrame.animate( {
                stroke: '#E16E6E',
                'stroke-width': 3
            }, animDuration );
            paper.validationTrash.push( ctrFrame );

            var tooltipText = '<table border="0"><tr><td width="25"><img src="' + getPluginImgPath( 'close' ) + '"></td>';
            tooltipText += '<td><b>' + getTranslation( validationError.validationMessage ) + '</b></td></tr></table>';

            var roleTooltip = new Ext.ux.suncode.TooltipInfo( {
                info: tooltipText
            } );
            scrollToProcessObject( tab, headerBox );
            var tabPos = tab.getPosition();
            var menuXPos = tabPos[0] + parseInt( headerBox.x2 / paper.scale ) - tab.body.dom.scrollLeft;
            var menuYPos = tabPos[1] + parseInt( headerBox.y / paper.scale ) - tab.body.dom.scrollTop;
            roleTooltip.showAt( [ menuXPos, menuYPos ] );
            paper.validationTrash.push( roleTooltip );
            notifyMessageToolbar( getTranslation( validationError.validationMessage ), false );
            clearMessageToolbarWithDelay( 5 );
            break;
        case 'activity':
        case 'activityApplication':
        case 'activityApplicationParam':
        case 'form':
            var tab = changeProcessTab( validationError.processDefId );
            var paper = tab.paper;
            paper.bpmn.eraseTrash( paper.hoverTrash );
            var set = paper.bpmn.findMapObjectElements( paper, 'activityDefId', validationError.activityDefId );
            set.toFront();
            var connector = paper.bpmn.getConnectorFromSet( set );
            var box = connector.getBBox();
            var frame = paper.rect( box.x, box.y, paper.bpmn.activityWidth, paper.bpmn.activityHeight, 10 );
            frame.attr( {
                stroke: '#FFFFFF',
                'stroke-width': 1
            } );
            var animDuration = Ext.getCmp( 'main_panel' ).getAnimationsOn() ? 500 : 1;
            frame.animate( {
                stroke: '#FF9F9F',
                'stroke-width': 3
            }, animDuration );
            paper.validationTrash.push( frame );
            for ( var i = 0; i < connector.deadlineClocks.length; i++ ) {
                connector.deadlineClocks[i].clock.toFront();
            }

            var tooltipMessage = getTranslation( validationError.validationMessage );
            if ( validationError.errorLevel == 'activityApplication' ) {
                tooltipMessage += '<br>' + getTranslation( 'Aplikacja' ) + ': ' + validationError.applicationName;
            }
            if ( validationError.errorLevel == 'activityApplicationParam' ) {
                tooltipMessage += '<br>' + getTranslation( 'Aplikacja' ) + ': ' + validationError.applicationName;
            }

            var tooltipText = '<table border="0"><tr><td width="25"><img src="' + getPluginImgPath( 'close' ) + '"></td>';
            tooltipText += '<td><b>' + tooltipMessage + '</b></td></tr></table>';

            var activityTooltip = new Ext.ux.suncode.TooltipInfo( {
                info: tooltipText
            } );
            scrollToProcessObject( tab, box );
            var tabPos = tab.getPosition();
            var menuXPos = tabPos[0] + parseInt( box.x / paper.scale ) - tab.body.dom.scrollLeft;
            var menuYPos = tabPos[1] + parseInt( box.y2 / paper.scale ) - tab.body.dom.scrollTop;
            activityTooltip.showAt( [ menuXPos, menuYPos ] );
            paper.validationTrash.push( activityTooltip );
            notifyMessageToolbar( getTranslation( validationError.validationMessage ), false );
            clearMessageToolbarWithDelay( 5 );
            break;
        case 'transition':
            var tab = changeProcessTab( validationError.processDefId );
            var paper = tab.paper;
            var set = paper.bpmn.findMapObjectElements( paper, 'transitionId', validationError.transitionId );
            set.toFront();
            var el = set[0];
            var path = el.attr( 'path' );
            var box = Raphael.pathBBox( path );

            var tooltipText = '<table border="0"><tr><td width="25"><img src="' + getPluginImgPath( 'close' ) + '"></td>';
            tooltipText += '<td><b>' + getTranslation( validationError.validationMessage ) + '</b></td></tr></table>';

            var activityTooltip = new Ext.ux.suncode.TooltipInfo( {
                info: tooltipText
            } );
            scrollToProcessObject( tab, box );
            var tabPos = tab.getPosition();
            var menuXPos = tabPos[0] + parseInt( box.x / paper.scale ) - tab.body.dom.scrollLeft;
            var menuYPos = tabPos[1] + parseInt( box.y / paper.scale ) - tab.body.dom.scrollTop;
            activityTooltip.showAt( [ menuXPos, menuYPos ] );
            paper.validationTrash.push( activityTooltip );
            paper.bpmn.changeLineColor( set, '#E16E6E' );
            paper.bpmn.changeLineColorWithDelay( set, '#444444', 5 );
            notifyMessageToolbar( getTranslation( validationError.validationMessage ), false );
            clearMessageToolbarWithDelay( 5 );
            break;
        default:
            break;
    }
}

function showPackageTranslations() {
	var packagePanel = Ext.getCmp( 'package_panel' );
    var packageNode = packagePanel.getRootNode();
	var win = new Ext.ux.suncode.PackageI18NTranslationsWindow( {
		packageNode: packageNode
	} );
    win.show();
}

function showXpdlArchive() {
    var win = new Ext.ux.suncode.XpdlArchiveWindow();
    win.show();
}

function showPackagesHistory() {
    var win = new Ext.ux.suncode.PackagesHistoryWindow();
    win.show();
}

function showChangeCardWindow() {
    var processPanel = Ext.getCmp( 'paper_panel' ).getActiveTab();

    if ( !Ext.isEmpty( processPanel ) ) {
        var packagePanel = Ext.getCmp( 'package_panel' );
        var packageNode = packagePanel.getRootNode();
        var processDefId = processPanel.processDefId;

        var win = new Ext.ux.suncode.ChangeCardWindow( {
            packageId: packageNode.attributes.packageId,
            processDefId: processDefId
        } );
        win.show();
    } else {
        showWarn( getTranslation( 'Wybierz proces.' ) );
    }
}

function changeAdvancedView() {
    var item = Ext.getCmp( 'advanced_view_main_menu_item' );
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.changeAdvancedView();
    var showTooltips = mainPanel.getShowTooltips();

    if ( mainPanel.getAdvancedView() ) {
        Ext.getCmp( 'draw_deadline_line_button' ).show();
        Ext.getCmp( 'draw_subflow_button' ).show();
        item.setIconClass( 'x-Module-tickIcon' );
        updateAllProcessTabTips( showTooltips );
    } else {
        Ext.getCmp( 'draw_deadline_line_button' ).hide();
        Ext.getCmp( 'draw_subflow_button' ).hide();
        item.setIconClass( 'x-Module-crossIcon' );
        updateAllProcessTabTips( showTooltips );
    }

    updateUserConfig();
}

function changeExperimentalView() {
    var item = Ext.getCmp( 'experimental_view_main_menu_item' );
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.changeExperimentalView();

    if ( mainPanel.getExperimentalView() ) {
      item.setIconClass( 'x-Module-tickIcon' );
    } else {
      item.setIconClass( 'x-Module-crossIcon' );
    }

    updateUserConfig();
}

function changeTooltipVisibility() {
    var item = Ext.getCmp( 'tooltip_visibility_main_menu_item' );
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.changeTooltipsVisibility();
    var showTooltips = mainPanel.getShowTooltips();

    if ( showTooltips ) {
        item.setIconClass( 'x-Module-tickIcon' );
        updateAllProcessTabTips( showTooltips );
    } else {
        item.setIconClass( 'x-Module-crossIcon' );
        updateAllProcessTabTips( showTooltips );
    }

    updateUserConfig();
}

function changeAnimationsAbility() {
    var item = Ext.getCmp( 'animations_main_menu_item' );
    var packagePanel = Ext.getCmp( 'package_panel' );
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.changeAnimationsAbility();

    if ( mainPanel.getAnimationsOn() ) {
        item.setIconClass( 'x-Module-tickIcon' );
        packagePanel.animCollapse = true;
        packagePanel.animate = true;
    } else {
        item.setIconClass( 'x-Module-crossIcon' );
        packagePanel.animCollapse = false;
        packagePanel.animate = false;
    }

    updateUserConfig();
}

function changeExportJs() {
    var item = Ext.getCmp( 'export_js_main_menu_item' );
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.changeExportJs();

    if ( mainPanel.getExportJs() ) {
        item.setIconClass( 'x-Module-tickIcon' );
    } else {
        item.setIconClass( 'x-Module-crossIcon' );
    }

    updateUserConfig();
}

function changeExportJavaCode() {
    var item = Ext.getCmp( 'export_java_code_main_menu_item' );
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.changeExportJavaCode();

    if ( mainPanel.getExportJavaCode() ) {
        item.setIconClass( 'x-Module-tickIcon' );
    } else {
        item.setIconClass( 'x-Module-crossIcon' );
    }

    updateUserConfig();
}

function changeDocumentationView() {
    var item = Ext.getCmp( 'documentation_view_main_menu_item' );
    var generateDocumentationItem = Ext.getCmp( 'generate_documentation_main_menu_item' );
    var changeCardItem = Ext.getCmp( 'change_card_main_menu_item' );
    var mainPanel = Ext.getCmp( 'main_panel' );
    mainPanel.changeDocumentationView();

    if ( mainPanel.getDocumentationView() ) {
        item.setIconClass( 'x-Module-tickIcon' );
        generateDocumentationItem.show();
        changeCardItem.show();
    } else {
        item.setIconClass( 'x-Module-crossIcon' );
        generateDocumentationItem.hide();
        changeCardItem.hide();
    }

    updateUserConfig();
}

function changeUseExternalClipboard() {
  var mainPanel = Ext.getCmp( 'main_panel' );
  var currentValue = mainPanel.getUseExternalClipboard();

  if ( !currentValue ) {
    readTextFromClipboard( function() {
      executeChangeUseExternalClipboard();
    }, this, function() {
      showWarn( getTranslation( 'Włącz dostęp do zewnętrznego schowka.' ) );
    }, this );
  } else {
    executeChangeUseExternalClipboard();
  }
}

function executeChangeUseExternalClipboard() {
  var item = Ext.getCmp( 'use_external_clipboard_menu_item' );
  var mainPanel = Ext.getCmp( 'main_panel' );
  mainPanel.changeUseExternalClipboard();

  if ( mainPanel.getUseExternalClipboard() ) {
    item.setIconClass( 'x-Module-tickIcon' );
  } else {
    item.setIconClass( 'x-Module-crossIcon' );
  }

  updateUserConfig();
}

function simulation() {
    var simulationConfigWindow = new Ext.ux.suncode.SimulationConfigWindow();
    simulationConfigWindow.show();
}

function changeCompatibilityMode( id ) {
	var mainPanel = Ext.getCmp( 'main_panel' );
    var prevMode = mainPanel.getCompatibilityMode();
    if ( prevMode.getId() == id ) {
        return;
    }

    mainPanel.setCompatibilityMode( id );
    var previousItem = Ext.getCmp( 'compatibility_mode_' + prevMode.getId() + '_main_menu_item' );
    previousItem.setIconClass( undefined );
    var currentItem = Ext.getCmp( 'compatibility_mode_' + id + '_main_menu_item' );
    currentItem.setIconClass( 'x-Module-tickIcon' );
    var loadJavaCodeToSystemItem = Ext.getCmp( 'export_java_classes_main_menu_item' );
    var exportJsItem = Ext.getCmp( 'export_js_main_menu_item' );
    var exportJavaCodeItem = Ext.getCmp( 'export_java_code_main_menu_item' );
    var currentMode = mainPanel.getCompatibilityMode();
    
    if ( currentMode.showLoadJavaCodeToSystem() ) {
    	loadJavaCodeToSystemItem.show();
    } else {
    	loadJavaCodeToSystemItem.hide();
    }
    
    if ( currentMode.showExportJs( mainPanel.getExportJs() ) ) {
    	exportJsItem.show();
    } else {
    	exportJsItem.hide();
    }
    
    if ( currentMode.showExportJavaCode( mainPanel.getExportJavaCode() ) ) {
    	exportJavaCodeItem.show();
    } else {
    	exportJavaCodeItem.hide();
    }

    updateUserConfig();
}

function changeHoverColor() {
    closeColorChooser();

    var mainPanel = Ext.getCmp( 'main_panel' );
    var item = Ext.getCmp( 'hover_color_main_menu_item' );
    var colorChooser = new Ext.ux.suncode.ColorChooser( {
        title: getTranslation( 'Kolor podświetlenia' ),
        e: {
            getXY: function() {
                return item.getEl().getXY();
            }
        },
        color: mainPanel.getHoverColor(),
        saveFunction: function( choosenColor ) {
            mainPanel.setHoverColor( choosenColor );
            var processTab = Ext.getCmp( 'paper_panel' ).getActiveTab();

            if ( processTab ) {
                var paper = processTab.paper;

                for ( var i = 0; i < paper.resizerTrash.length; i++ ) {
                    var el = paper.resizerTrash[i];
                    el.attr( {
                        fill: choosenColor
                    } );
                }
            }

            item.setText( createMenuColorBox( getTranslation( 'Kolor podświetlenia' ), choosenColor ) );
            updateUserConfig();
        }
    } );
    colorChooser.show();
}

function changeIncomingTransitionColor() {
  closeColorChooser();

  var mainPanel = Ext.getCmp( 'main_panel' );
  var item = Ext.getCmp( 'incoming_transition_color_main_menu_item' );
  var colorChooser = new Ext.ux.suncode.ColorChooser( {
    title: getTranslation( 'Kolor połączeń wchodzących' ),
    e: {
      getXY: function() {
        return item.getEl().getXY();
      }
    },
    color: mainPanel.getIncomingTransitionColor(),
    saveFunction: function( choosenColor ) {
      mainPanel.setIncomingTransitionColor( choosenColor );
      item.setText( createMenuColorBox( getTranslation( 'Kolor połączeń wchodzących' ), choosenColor ) );
      updateUserConfig();
    }
  } );
  colorChooser.show();
}

function changeOutgoingTransitionColor() {
  closeColorChooser();

  var mainPanel = Ext.getCmp( 'main_panel' );
  var item = Ext.getCmp( 'outgoing_transition_color_main_menu_item' );
  var colorChooser = new Ext.ux.suncode.ColorChooser( {
    title: getTranslation( 'Kolor połączeń wychodzących' ),
    e: {
      getXY: function() {
        return item.getEl().getXY();
      }
    },
    color: mainPanel.getOutgoingTransitionColor(),
    saveFunction: function( choosenColor ) {
      mainPanel.setOutgoingTransitionColor( choosenColor );
      item.setText( createMenuColorBox( getTranslation( 'Kolor połączeń wychodzących' ), choosenColor ) );
      updateUserConfig();
    }
  } );
  colorChooser.show();
}

function changeXpdlTranslationLanguage( language ) {
  var mainPanel = Ext.getCmp( 'main_panel' );
  var prevLanguage = mainPanel.getXpdlTranslationLanguage();
  if ( prevLanguage == language ) {
    return;
  }

  mainPanel.setXpdlTranslationLanguage( language );
  Ext.ux.suncode.I18NService.setXpdlTranslationLanguage( language );
  var previousItem = Ext.getCmp( 'xpdl_translation_language_' + prevLanguage + '_main_menu_item' );
  previousItem.setIconClass( undefined );
  var currentItem = Ext.getCmp( 'xpdl_translation_language_' + language + '_main_menu_item' );
  currentItem.setIconClass( 'x-Module-tickIcon' );

  updateUserConfig();
  updateXpdlTranslationsInEditor();
}

function updateXpdlTranslationsInEditor() {
  var mainPanel = Ext.getCmp( 'main_panel' );
  var paperPanel = Ext.getCmp( 'paper_panel' );
  var activeTab = paperPanel.getActiveTab();
  var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();

  if ( packageNode.hasChildNodes() ) {
    var packageId = packageNode.attributes.packageId;
    packageNode.setText( getXpdlPackageNameTranslation( packageId, packageNode.attributes.packageName ) );

    packageNode.eachChild( function ( processNode ) {
      var processDefId = processNode.attributes.processDefId;
      var processName = processNode.attributes.processName;
      var processColor = processNode.attributes.processColor;
      var processTab = paperPanel.findProcessTab( processDefId );
      paperPanel.activate( processTab );
      var paper = processTab.paper;
      processNode.setText( processNode.getNodeText( processDefId, processName, processColor ) );
      paperPanel.updateProcessTab( processDefId, processDefId, processName, processColor );

      if ( processNode.hasChildNodes() ) {
        Ext.each( processNode.attributes.participants, function( role, index, roles ) {
          var roleId = role.roleId;
          paper.bpmn.updateRole( paper, roleId, roleId, role.roleName );
        } );


        processNode.eachChild( function ( activityNode ) {
          var activityDefId = activityNode.attributes.activityDefId;
          var activityName = activityNode.attributes.activityName;
          activityNode.setText( activityNode.getNodeText(processDefId, activityDefId, activityName ) );

          paper.bpmn.updateActivity( paper, {
            processDefId: processDefId,
            oldActDefId: activityDefId,
            newActDefId: activityDefId,
            actName: activityName
          } );
        } );

        processNode.sortActivities();
      }
    } );

    packageNode.sortProcesses();
  }

  paperPanel.activate( activeTab );
  updateAllProcessTabTips( mainPanel.getShowTooltips() );
}

function changeDistinctionColor() {
    closeColorChooser();

    var mainPanel = Ext.getCmp( 'main_panel' );
    var item = Ext.getCmp( 'distinction_color_main_menu_item' );
    var colorChooser = new Ext.ux.suncode.ColorChooser( {
        title: getTranslation( 'Kolor wyróżnienia' ),
        e: {
            getXY: function() {
                return item.getEl().getXY();
            }
        },
        color: mainPanel.getDistinctionColor(),
        saveFunction: function( choosenColor ) {
            mainPanel.setDistinctionColor( choosenColor );
            item.setText( createMenuColorBox( getTranslation( 'Kolor wyróżnienia' ), choosenColor ) );
            setDistinctionColorCssRules( choosenColor );
            updateUserConfig();
        }
    } );
    colorChooser.show();
}

function restoreFormsDefaultLook() {
  Ext.ux.suncode.GlobalSettingsService.resetGlobalSettings();
  var packageNode = Ext.getCmp( 'package_panel' ).getRootNode();

  if ( packageNode.hasChildNodes() ) {
    packageNode.eachChild( function( processNode ) {
      processNode.restoreFormsDefaultLook();
    }, this );
  }

  Ext.getCmp( 'main_panel' ).setSaved( false );
}

function updateUserConfig() {
    var userConfig = Ext.getCmp( 'main_panel' ).getUserConfig();

    Ext.Ajax.request( {
        url: 'api/userconfig/set',
        method: 'POST',
        params: userConfig,
        success: function( response, opts ) {
            if ( !parseBoolean( response.responseText ) ) {
                showWarn( getTranslation( 'Wystąpił błąd.' ) );
            }
        },
        failure: function( response, opts ) {
        	showWarn( getTranslation( 'Wystąpił błąd.' ) );
        }
    } );
}

function tutorials() {
    var win = new Ext.ux.suncode.TutorialsWindow();
    win.show();
}

function help() {
    var win = new Ext.ux.suncode.HelpWindow( {} );
    win.show();
}

function about() {
    var win = new Ext.ux.suncode.AboutWindow();
    win.show();
}