/*
 * Decompiled with CFR 0.152.
 */
package com.suncode.plusocr.suncodeocr.domain;

import com.suncode.plusocr.suncodeocr.domain.Formatters;
import java.math.BigInteger;
import java.util.Locale;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Validators {
    private static final Logger log = LoggerFactory.getLogger(Validators.class);
    public static final Predicate<Object> IBAN_VALIDATOR = iban -> {
        if (iban == null) {
            log.info("IBAN_VALIDATOR -> Reject null input");
            return false;
        }
        if (iban instanceof String[]) {
            String[] ibanArray = (String[])iban;
            if (ibanArray.length == 0) {
                log.info("IBAN_VALIDATOR -> Reject empty array input");
                return false;
            }
            for (String singleIban : ibanArray) {
                if (singleIban != null && Validators.validateIban(singleIban)) continue;
                log.info("IBAN_VALIDATOR -> Reject invalid IBAN in array: {}", (Object)singleIban);
                return false;
            }
            return true;
        }
        return Validators.validateIban(iban.toString());
    };
    public static final Predicate<Object> TAX_VALIDATOR = taxId -> {
        if (taxId == null) {
            log.info("TAX_VALIDATOR -> Reject null input");
            return false;
        }
        String input = (String)Formatters.TAX_FORMATTER.apply(taxId);
        if (StringUtils.isBlank((CharSequence)input)) {
            log.info("TAX_VALIDATOR -> Reject empty input after cleaning");
            return false;
        }
        if (!input.matches("^[A-Z0-9]+$")) {
            log.info("TAX_VALIDATOR -> Reject input containing invalid characters (like colon): {}", (Object)input);
            return false;
        }
        String countryCode = "";
        String number = input;
        Pattern countryPattern = Pattern.compile("^([A-Z]{2})([A-Z0-9]+)$");
        Matcher countryMatcher = countryPattern.matcher(input);
        if (countryMatcher.matches()) {
            countryCode = countryMatcher.group(1);
            number = countryMatcher.group(2);
        }
        if (number.length() < 4) {
            log.info("TAX_VALIDATOR -> Reject number too short (less than 4 characters): {}", (Object)number.length());
            return false;
        }
        if ("PL".equals(countryCode)) {
            return Validators.validatePolishNIP(number);
        }
        if (countryCode.isEmpty() && number.matches("\\d{1,9}")) {
            log.info("TAX_VALIDATOR -> Reject simple numeric input without country code: {}", (Object)number);
            return false;
        }
        if (!number.matches("[A-Z0-9]{4,20}")) {
            log.info("TAX_VALIDATOR -> Reject input not matching alphanumeric pattern: {}", (Object)number);
            return false;
        }
        return true;
    };

    private static boolean validateIban(String iban) {
        boolean isValid;
        String trimmed = iban.replaceAll("\\s+", "").toUpperCase(Locale.ROOT);
        if (!trimmed.matches("[A-Z0-9]+")) {
            log.info("IBAN_VALIDATOR -> Reject IBAN containing invalid characters: {}", (Object)trimmed);
            return false;
        }
        if (trimmed.length() < 15 || trimmed.length() > 34) {
            log.info("IBAN_VALIDATOR -> Reject IBAN with invalid length {}: {}", (Object)trimmed.length(), (Object)trimmed);
            return false;
        }
        String rearranged = trimmed.substring(4) + trimmed.substring(0, 4);
        StringBuilder numericIban = new StringBuilder();
        for (char ch : rearranged.toCharArray()) {
            if (Character.isLetter(ch)) {
                numericIban.append(ch - 65 + 10);
                continue;
            }
            numericIban.append(ch);
        }
        BigInteger ibanNumber = new BigInteger(numericIban.toString());
        boolean bl = isValid = ibanNumber.mod(BigInteger.valueOf(97L)).intValue() == 1;
        if (!isValid) {
            log.info("IBAN_VALIDATOR -> Reject IBAN with invalid checksum: {}", (Object)trimmed);
        }
        return isValid;
    }

    private static boolean validatePolishNIP(String nip) {
        boolean isValid;
        if (!nip.matches("\\d{10}")) {
            log.info("TAX_VALIDATOR -> Polish NIP validation failed: not exactly 10 digits");
            return false;
        }
        int[] weights = new int[]{6, 5, 7, 2, 3, 4, 5, 6, 7};
        int sum = 0;
        for (int i = 0; i < 9; ++i) {
            sum += (nip.charAt(i) - 48) * weights[i];
        }
        int checkDigit = sum % 11;
        boolean bl = isValid = checkDigit != 10 && checkDigit == nip.charAt(9) - 48;
        if (!isValid) {
            log.info("TAX_VALIDATOR -> Polish NIP validation failed: invalid checksum");
        }
        return isValid;
    }

    private Validators() {
    }
}

