#!/usr/bin/env node

/**
 * Script to generate PDF from KSeF XML using ksef-pdf-generator library
 * Usage: node generate-pdf.js [nrKSeF] [qrCode]
 * Reads XML from stdin, writes PDF to stdout
 */

const path = require('path');

const SKIP_MARKER = "SKIP";
const args = process.argv.slice(2);
const nrKSeF = args[0] || '';
const qrCode = args[1] || '';
const qrCode2 = args[2] || '';
const invoiceLanguage = args[3] || '';

const scriptDir = __dirname;
const libraryPath = path.join(scriptDir, 'lib', 'ksef-pdf-generator.es.js');

let xmlContent = '';
process.stdin.setEncoding('utf8');

process.stdin.on('data', (chunk) => {
    xmlContent += chunk;
});

process.stdin.on('end', () => {
    generatePdf();
});

process.stdin.on('error', (error) => {
    console.error(`Failed to read XML from stdin: ${error.message}`);
    process.exit(1);
});

class File {
    constructor(content, name, options = {}) {
        this.name = name;
        this.content = content;
        this.size = Buffer.byteLength(content, 'utf8');
        this.type = options.type || 'text/xml';
        this.lastModified = Date.now();
    }
}

class FileReader {
    constructor() {
        this.result = null;
        this.onload = null;
        this.onerror = null;
    }

    readAsText(file) {
        setTimeout(() => {
            try {
                this.result = typeof file === 'string' ? file : file.content || file;
                if (this.onload) {
                    this.onload({ target: { result: this.result } });
                }
            } catch (error) {
                if (this.onerror) {
                    this.onerror(error);
                }
            }
        }, 0);
    }
}

class Blob {
    constructor(parts = [], options = {}) {
        this.parts = parts;
        this.type = options.type || '';
        this.size = 0;
        if (Array.isArray(parts)) {
            this.size = parts.reduce((acc, part) => {
                if (typeof part === 'string') {
                    return acc + Buffer.byteLength(part, 'utf8');
                } else if (Buffer.isBuffer(part)) {
                    return acc + part.length;
                } else if (part instanceof Uint8Array) {
                    return acc + part.length;
                }
                return acc;
            }, 0);
        }
    }

    async arrayBuffer() {
        const buffers = this.parts.map(part => {
            if (typeof part === 'string') {
                return Buffer.from(part, 'utf8');
            } else if (Buffer.isBuffer(part)) {
                return part;
            } else if (part instanceof Uint8Array) {
                return Buffer.from(part);
            }
            return Buffer.from(String(part));
        });
        return Buffer.concat(buffers).buffer;
    }
}

global.File = File;
global.FileReader = FileReader;
global.Blob = Blob;
global.window = global;
global.document = {
    createElement: () => ({}),
    getElementById: () => null,
    body: { appendChild: () => {}, removeChild: () => {} }
};

async function generatePdf() {
    try {
        const { generateInvoice, generatePDFUPO } = await import('file://' + libraryPath);
        const xmlFile = new File(xmlContent, 'input.xml', { type: 'text/xml' });

        const upoMarkerPattern = /<[^:>]*:?Potwierdzenie/;
        const upoShortMarkerPattern = /<[^:>]*:?UPO/;
        const isUPO = upoMarkerPattern.test(xmlContent) || upoShortMarkerPattern.test(xmlContent);

        let pdfBlob;
        if (isUPO) {
            pdfBlob = await generatePDFUPO(xmlFile);
        } else {
            const additionalData = {};
            if (nrKSeF && nrKSeF !== SKIP_MARKER) {
                additionalData.nrKSeF = nrKSeF;
            }
            if (qrCode && qrCode !== SKIP_MARKER) {
                additionalData.qrCode = qrCode;
            }
            if (qrCode2 && qrCode2 !== SKIP_MARKER) {
                additionalData.qrCode2 = qrCode2;
            }
            if (invoiceLanguage && invoiceLanguage !== SKIP_MARKER) {
                additionalData.language = invoiceLanguage;
            }
            pdfBlob = await generateInvoice(xmlFile, additionalData, 'blob');
        }

        let pdfBuffer;
        if (pdfBlob instanceof Blob) {
            const arrayBuffer = await pdfBlob.arrayBuffer();
            pdfBuffer = Buffer.from(arrayBuffer);
        } else if (Buffer.isBuffer(pdfBlob)) {
            pdfBuffer = pdfBlob;
        } else if (pdfBlob instanceof Uint8Array) {
            pdfBuffer = Buffer.from(pdfBlob);
        } else {
            throw new Error('Unexpected PDF blob type');
        }

        process.stdout.write(pdfBuffer);
    } catch (error) {
        console.error(`Failed to generate PDF: ${error.message}`);
        console.error(error.stack);
        process.exit(1);
    }
}