package com.plusmpm.directorymonitor.xml;

import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.XMLReaderFactory;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

@Slf4j
public class XmlValueExtractor
    extends DefaultHandler
{

    @Getter
    private List<Element> capturedXMLElements;

    private final boolean ignoreNamespaces;

    private XmlValueExtractor( Element startingElement, boolean ignoreNamespaces )
    {
        this.capturedXMLElements = new ArrayList<>( 1 );
        this.capturedXMLElements.add( startingElement );

        this.ignoreNamespaces = ignoreNamespaces;
    }

    public static Vector<String> getValuesByXmlTagPathWithDelimiter(
        Element startingElement, boolean ignoreNamespaces,
        String xmlTagDelimiterPath, String xmlTagPath, String defaultValue
    )
    {
        log.info( "Getting values by delimiter xml tag path [{}] "
                      + "with xml tag path [{}] "
                      + "and default value placeholder [{}]",
                  xmlTagDelimiterPath, xmlTagPath, defaultValue );

        List<Element> capturedXMLElementsByDelimiter = getCapturedElementsByXmlTagPath(
            startingElement, ignoreNamespaces,
            xmlTagDelimiterPath
        );

        Vector<String> values = new Vector<>();
        for ( Element xmlDelimitedElement : capturedXMLElementsByDelimiter )
        {
            List<String> valuesByDelimiter = getValuesByXmlTagPath(
                xmlDelimitedElement, ignoreNamespaces,
                xmlTagPath
            );

            if ( !valuesByDelimiter.isEmpty() )
            {
                if ( valuesByDelimiter.size() >= 2 )
                {
                    log.warn( "Found more than one value by xml tag path [{}] - getting first value from {}",
                              xmlTagPath, valuesByDelimiter );
                }

                values.add( valuesByDelimiter.get( 0 ) );
            }
            else
            {
                values.add( defaultValue );
            }
        }

        log.info( "Resolved values: {}", values );
        return values;
    }

    public static Vector<String> getValuesByXmlTagPath(
        Element startingElement, boolean ignoreNamespaces,
        String xmlTagPath
    )
    {
        log.info( "Getting values by xml tag path [{}]", xmlTagPath );


        List<Element> capturedXMLElements = getCapturedElementsByXmlTagPath(
            startingElement, ignoreNamespaces,
            xmlTagPath
        );

        Vector<String> values = new Vector<>();
        for ( Element xmlElement : capturedXMLElements )
        {
            Node childNode = xmlElement.getFirstChild();
            if ( childNode != null )
            {
                values.add( childNode.getNodeValue() );
            }
            else
            {
                values.add( "" );
            }
        }

        log.info( "Resolved values: {}", values );
        return values;
    }

    public static Vector<String> getValuesByXmlAttributePathWithDelimiter(
        Element startingElement, boolean ignoreNamespaces,
        String xmlTagDelimiterPath, String xmlTagPath,
        String xmlAttribute, String defaultValue
    )
    {
        log.info( "Getting values by delimiter xml tag path [{}] "
                      + "with xml tag path [{}] "
                      + "and attribute [{}]"
                      + "and default value placeholder [{}]",
                  xmlTagDelimiterPath, xmlTagPath, xmlAttribute, defaultValue );

        List<Element> capturedXMLElementsByDelimiter = getCapturedElementsByXmlTagPath(
            startingElement, ignoreNamespaces,
            xmlTagDelimiterPath
        );

        Vector<String> values = new Vector<>();
        for ( Element xmlDelimitedElement : capturedXMLElementsByDelimiter )
        {
            List<String> valuesByDelimiter = getValuesByXmlAttributePath(
                xmlDelimitedElement, ignoreNamespaces,
                xmlTagPath, xmlAttribute
            );

            if ( !valuesByDelimiter.isEmpty() )
            {
                if ( valuesByDelimiter.size() >= 2 )
                {
                    log.warn( "Found more than one value by xml tag path [{}] "
                                  + "and attribute [{}] - getting first value from {}",
                              xmlTagPath, xmlAttribute, valuesByDelimiter );
                }

                values.add( valuesByDelimiter.get( 0 ) );
            }
            else
            {
                values.add( defaultValue );
            }
        }

        log.info( "Resolved values: {}", values );
        return values;
    }

    public static Vector<String> getValuesByXmlAttributePath(
        Element startingElement, boolean ignoreNamespaces,
        String xmlTagPath, String xmlAttribute
    )
    {
        log.info( "Getting values by xml tag path [{}] and attribute [{}]",
                  xmlTagPath, xmlAttribute );

        List<Element> capturedXMLElements = getCapturedElementsByXmlTagPath(
            startingElement, ignoreNamespaces,
            xmlTagPath
        );

        Vector<String> values = new Vector<>();
        for ( Element xmlElement : capturedXMLElements )
        {
            String value = ignoreNamespaces
                ? getUnboundAttributeValue( xmlElement, unbound( xmlAttribute ) )
                : xmlElement.getAttribute( xmlAttribute );

            if ( value != null )
            {
                values.add( value );
            }
        }

        log.info( "Resolved values: {}", values );
        return values;
    }

    public static List<Element> getCapturedElementsByXmlTagPath(
        Element startingElement, boolean ignoreNamespaces,
        String xmlTagPath
    )
    {
        log.info( "Getting captured elements by xml tag path [{}]", xmlTagPath );
        XmlValueExtractor extractor = new XmlValueExtractor( startingElement, ignoreNamespaces );

        try
        {
            // utworzenie obiektu reprezentującego parser
            XMLReader parser = XMLReaderFactory.createXMLReader();

            // rejestracja handlera w parserze
            parser.setContentHandler( extractor );
            parser.setErrorHandler( extractor );
            parser.setFeature( "http://xml.org/sax/features/namespaces", false );
            parser.setFeature( "http://xml.org/sax/features/namespace-prefixes", true );

            // start parsera -typ parsera podajemy z linii polecenia:
            // java -Dorg.xml.sax.driver=org.apache.xerces.parsers.SAXParser SAXExample plik.xml
            parser.parse( new InputSource( new StringReader( xmlTagPath ) ) );
        }
        catch ( SAXParseException e )
        {
            return extractor.getCapturedXMLElements();
        }
        catch ( Exception e )
        {
            log.error( e.getLocalizedMessage(), e );
        }

        return new ArrayList<>();
    }

    @Override
    public void startDocument()
    {

    }

    @Override
    public void endDocument()
    {

    }

    @Override
    public void startElement( String namespaceURL, String localName,
                              String qName, Attributes attributes )
    {
        if ( ignoreNamespaces )
        {
            parseWithoutNamespaces( qName, attributes );
        }
        else
        {
            parseWithNamespaces( qName, attributes );
        }
    }

    private void parseWithoutNamespaces( String qName, Attributes attributes )
    {
        String unboundTagName = unbound( qName );

        List<Element> newXMLElements = new ArrayList<>();
        for ( Element vXMLElement : capturedXMLElements )
        {
            NodeList nodeList = vXMLElement.getElementsByTagName( "*" );
            for ( int j = 0; j < nodeList.getLength(); j++ )
            {
                Element element = (Element) nodeList.item( j );

                String elementTagName = element.getTagName();
                String unboundElementTagName = unbound( elementTagName );

                if ( !unboundElementTagName.equals( unboundTagName ) )
                {
                    continue;
                }

                if ( areUnboundAttributesMatched( element, attributes ) )
                {
                    newXMLElements.add( element );
                }
            }
        }
        capturedXMLElements.clear();
        capturedXMLElements = newXMLElements;
    }

    private boolean areUnboundAttributesMatched( Element element, Attributes attributes )
    {
        boolean attributesMatched = true;
        for ( int k = 0; k < attributes.getLength(); k++ )
        {
            String attributeQualifiedName = attributes.getQName( k );
            if ( attributeQualifiedName.startsWith( "xmlns:" ) )
            {
                continue;
            }

            String unboundAttributeName = unbound( attributeQualifiedName );
            String unboundAttributeValue = attributes.getValue( k );

            String elementAttributeValue = getUnboundAttributeValue( element, unboundAttributeName );

            if ( !elementAttributeValue.equals( unboundAttributeValue ) )
            {
                attributesMatched = false;
                break;
            }
        }
        return attributesMatched;
    }

    private void parseWithNamespaces( String qName, Attributes attributes )
    {
        List<Element> newXMLElements = new ArrayList<>();
        for ( Element vXMLElement : capturedXMLElements )
        {
            NodeList nodeList = vXMLElement.getElementsByTagName( qName );
            for ( int j = 0; j < nodeList.getLength(); j++ )
            {
                Element element = (Element) nodeList.item( j );

                if ( areAttributesMatched( element, attributes ) )
                {
                    newXMLElements.add( element );
                }
            }
        }
        capturedXMLElements.clear();
        capturedXMLElements = newXMLElements;
    }

    private boolean areAttributesMatched( Element element, Attributes attributes )
    {
        boolean attributesMatched = true;
        for ( int k = 0; k < attributes.getLength(); k++ )
        {
            String attributeQualifiedName = attributes.getQName( k );
            String attributeValue = attributes.getValue( k );

            if ( attributeQualifiedName.startsWith( "xmlns:" ) )
            {
                continue;
            }

            String elementAttributeValue = element.getAttribute( attributeQualifiedName );
            if ( !elementAttributeValue.equals( attributeValue ) )
            {
                attributesMatched = false;
                break;
            }
        }
        return attributesMatched;
    }

    private static String getUnboundAttributeValue( Element element, String unboundAttributeName )
    {
        NamedNodeMap elementAttributes = element.getAttributes();

        String elementAttributeValue = "";
        for (int i = 0; i < elementAttributes.getLength(); ++i )
        {
            Node elementAttributeNode = elementAttributes.item( i );

            String unboundElementAttributeName = unbound( elementAttributeNode.getNodeName() );
            String unboundElementAttributeValue = elementAttributeNode.getNodeValue();

            if ( unboundElementAttributeName.equals( unboundAttributeName ) )
            {
                elementAttributeValue = unboundElementAttributeValue;
                break;
            }
        }
        return elementAttributeValue;
    }

    private static String unbound( String tagOrAttributeName )
    {
        return tagOrAttributeName.substring( tagOrAttributeName.indexOf( ":" ) + 1 );
    }

    @Override
    public void endElement( String namespaceURL, String localName, String qName )
    {
        int i = 0;
        while ( capturedXMLElements.size() > i + 1 )
        {
            Element oEl = capturedXMLElements.get( i );
            oEl = (Element) oEl.getParentNode();
            if ( oEl == null )
            {
                capturedXMLElements.remove( i );
            }
            else
            {
                capturedXMLElements.set( i, oEl );
                i++;
            }
        }
    }

    @Override
    // nierozpoznane znaczki...
    public void characters( char[] ch, int start, int length )
    {
        int i = 0;
        while ( capturedXMLElements.size() > i + 1 )
        {
            String sValue = null;
            for ( int j = start; j < start + length; j++ )
            {
                sValue += ch[j];
            }
            Element oEl = capturedXMLElements.get( i );
            if ( oEl.getNodeValue().compareToIgnoreCase( sValue ) != 0 )
            {
                capturedXMLElements.remove( i );
            }
            else
            {
                i++;
            }
        }
    }

    // obsluga bledow
    /*
     * public void warning(SAXParseException ex) {...} public void error(SAXParseException ex) throws SAXException{}
     * public void fatalError(SAXParseException ex) throws SAXException{}
     */

}
