/*
 * Decompiled with CFR 0.152.
 */
package com.suncode.dbexplorer.alias;

import com.google.common.collect.ImmutableMap;
import com.suncode.dbexplorer.alias.Alias;
import com.suncode.dbexplorer.alias.AliasNotFoundException;
import com.suncode.dbexplorer.alias.AliasService;
import com.suncode.dbexplorer.alias.Table;
import com.suncode.dbexplorer.alias.TablesSet;
import com.suncode.dbexplorer.alias.data.DataController;
import com.suncode.dbexplorer.alias.dto.AliasDto;
import com.suncode.dbexplorer.alias.dto.ColumnDto;
import com.suncode.dbexplorer.alias.dto.SchemaDto;
import com.suncode.dbexplorer.alias.dto.SimpleTableDto;
import com.suncode.dbexplorer.alias.dto.TableDto;
import com.suncode.dbexplorer.alias.dto.TablesSetDto;
import com.suncode.dbexplorer.alias.exception.AliasExistsException;
import com.suncode.dbexplorer.alias.exception.AliasNotActiveException;
import com.suncode.dbexplorer.alias.exception.InvalidJdbcUrlException;
import com.suncode.dbexplorer.alias.settings.ColumnSettings;
import com.suncode.dbexplorer.alias.settings.TableSettings;
import com.suncode.dbexplorer.audit.AuditTypes;
import com.suncode.dbexplorer.context.UserContext;
import com.suncode.dbexplorer.database.ConnectionString;
import com.suncode.dbexplorer.database.Database;
import com.suncode.dbexplorer.database.DatabaseFactory;
import com.suncode.dbexplorer.database.DatabaseType;
import com.suncode.dbexplorer.database.exception.DatabaseNotAvailableException;
import com.suncode.dbexplorer.util.authorization.AuthorizationHelper;
import com.suncode.dbexplorer.util.web.rest.RestController;
import com.suncode.pwfl.audit.builder.ManualAuditBuilder;
import com.suncode.pwfl.translation.Translator;
import com.suncode.pwfl.translation.Translators;
import jakarta.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.HttpStatusCode;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

@Controller
public class AliasController
extends RestController {
    private static final Logger log = LoggerFactory.getLogger(AliasController.class);
    private static final Pattern JDBC_URL_PATTERN = Pattern.compile("^jdbc:(postgresql|postgres|sqlserver|oracle|db2):.*", 2);
    private final AliasService aliasService;
    private final DatabaseFactory databaseFactory;
    private final AuthorizationHelper authorizationHelper;

    @RequestMapping(value={"/aliases"}, method={RequestMethod.POST})
    @ResponseBody
    public ResponseEntity<?> add(@RequestBody AliasDto newAlias) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        Date started = new Date();
        try {
            boolean isJdbcUrlConfig = newAlias.getIsJdbcUrlConfig();
            if (isJdbcUrlConfig) {
                this.validateJdbcUrl(newAlias.getJdbcUrl());
            }
            Alias alias = new Alias(newAlias.getName(), ConnectionString.toDomain(newAlias.getConnectionString()));
            alias.setIsActive(newAlias.getIsActive());
            alias.setIsJdbcUrlConfig(isJdbcUrlConfig);
            alias.setLogging(newAlias.getLogging());
            this.aliasService.addAlias(alias);
            this.auditAliasOperation(started, AuditTypes.AUDIT_ALIAS_ADD, alias, null);
            AliasDto aliasDto = new AliasDto(alias);
            aliasDto.setDisplayedName(this.resolveDisplayedName(aliasDto));
            return new ResponseEntity((Object)aliasDto, (HttpStatusCode)HttpStatus.OK);
        }
        catch (AliasExistsException ex) {
            return new ResponseEntity((Object)ImmutableMap.of((Object)"aliasAlreadyExists", (Object)true), (HttpStatusCode)HttpStatus.NOT_ACCEPTABLE);
        }
    }

    private void validateJdbcUrl(String jdbcUrl) throws InvalidJdbcUrlException {
        if (StringUtils.isBlank((CharSequence)jdbcUrl) || !JDBC_URL_PATTERN.matcher(jdbcUrl).matches()) {
            throw new InvalidJdbcUrlException();
        }
    }

    @RequestMapping(value={"/system/aliases"}, method={RequestMethod.POST})
    @ResponseBody
    public ResponseEntity<?> addSystemAlias() {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        Date started = new Date();
        try {
            Alias systemAlias = this.aliasService.addSystemAlias();
            this.auditSystemAliasOperation(started, AuditTypes.AUDIT_ALIAS_ADD, systemAlias);
            AliasDto aliasDto = new AliasDto(systemAlias);
            aliasDto.setDisplayedName(this.resolveDisplayedName(aliasDto));
            return new ResponseEntity((Object)aliasDto, (HttpStatusCode)HttpStatus.OK);
        }
        catch (AliasExistsException ex) {
            return new ResponseEntity((Object)ImmutableMap.of((Object)"aliasAlreadyExists", (Object)true), (HttpStatusCode)HttpStatus.NOT_ACCEPTABLE);
        }
    }

    @RequestMapping(value={"/aliases/{aliasId}"}, method={RequestMethod.PUT})
    @ResponseBody
    public AliasDto update(@PathVariable Long aliasId, @RequestBody AliasDto updatedAlias) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        Date started = new Date();
        Alias oldAlias = this.getAlias(aliasId);
        Alias alias = this.getAlias(aliasId);
        boolean isJdbcUrlConfig = updatedAlias.getIsJdbcUrlConfig();
        if (isJdbcUrlConfig) {
            this.validateJdbcUrl(updatedAlias.getJdbcUrl());
        }
        alias.setName(updatedAlias.getName());
        alias.setConnectionStringDomain(ConnectionString.toDomain(updatedAlias.getConnectionString()));
        alias.setIsActive(updatedAlias.getIsActive());
        alias.setLogging(updatedAlias.getLogging());
        alias.setIsJdbcUrlConfig(isJdbcUrlConfig);
        this.aliasService.updateAlias(alias);
        this.auditAliasOperation(started, AuditTypes.AUDIT_ALIAS_UPDATE, alias, oldAlias);
        AliasDto aliasDto = new AliasDto(alias);
        aliasDto.setDisplayedName(this.resolveDisplayedName(aliasDto));
        return aliasDto;
    }

    @RequestMapping(value={"/aliases/{aliasId}"}, method={RequestMethod.DELETE})
    @ResponseBody
    public ResponseEntity<Void> delete(@PathVariable Long aliasId) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        Date started = new Date();
        Alias aliasToDelete = this.aliasService.getAlias(aliasId);
        this.aliasService.deleteAlias(aliasId);
        if (aliasToDelete.getIsSystemAlias().booleanValue()) {
            this.auditSystemAliasOperation(started, AuditTypes.AUDIT_ALIAS_DELETE, aliasToDelete);
        } else {
            this.auditAliasOperation(started, AuditTypes.AUDIT_ALIAS_DELETE, aliasToDelete, null);
        }
        return new ResponseEntity((HttpStatusCode)HttpStatus.NO_CONTENT);
    }

    @RequestMapping(value={"/aliases"}, method={RequestMethod.GET})
    @ResponseBody
    public List<AliasDto> getAll(HttpServletRequest request) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        log.info("Requesting {}" + request.getRequestURI());
        List<Alias> aliases = this.aliasService.getAliases();
        List<AliasDto> aliasDtos = aliases.stream().map(alias -> {
            try {
                return AliasDto.from(alias);
            }
            catch (Exception e) {
                log.info("Exception while fetching alias: {}, message: {}", (Object)alias.getName(), (Object)e.getMessage());
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toList());
        aliasDtos.forEach(aliasDto -> aliasDto.setDisplayedName(this.resolveDisplayedName((AliasDto)aliasDto)));
        return aliasDtos;
    }

    @RequestMapping(value={"/aliases/{aliasId}/refreshschema"}, method={RequestMethod.POST})
    @ResponseBody
    public void refreshSchema(@PathVariable Long aliasId) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        this.validateAliasIsActive(aliasId);
        Alias alias = this.aliasService.getAlias(aliasId);
        if (!this.databaseFactory.isAvailable(alias.getWrappedConnectionString())) {
            throw new DatabaseNotAvailableException();
        }
        Database database = this.databaseFactory.create(alias.getWrappedConnectionString(), true);
        database.updateSchema();
    }

    @RequestMapping(value={"/aliases/{aliasId}/tablessets"}, method={RequestMethod.GET})
    @ResponseBody
    public List<TablesSetDto> getTablesSets(@PathVariable Long aliasId) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        this.validateAliasIsActive(aliasId);
        Alias alias = this.getAlias(aliasId);
        for (TablesSet tableSet : alias.getTablesSets()) {
            this.normalizeTablesInSet(tableSet);
        }
        return TablesSetDto.from(alias);
    }

    @RequestMapping(value={"/aliases/{aliasId}/tablessets"}, method={RequestMethod.POST})
    @ResponseBody
    public TablesSetDto addTablesSet(@PathVariable Long aliasId, @RequestBody TablesSetDto newSet) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        this.validateAliasIsActive(aliasId);
        Date started = new Date();
        TablesSet set = new TablesSet(newSet.getName(), newSet.getTables());
        Alias alias = this.getAlias(aliasId);
        alias.addTablesSet(set);
        this.aliasService.updateAlias(alias);
        this.auditTableSetOperation(started, AuditTypes.AUDIT_TABLE_SET_ADD, set, null);
        return new TablesSetDto(alias, set);
    }

    @RequestMapping(value={"/aliases/{aliasId}/tablessets/{setId}"}, method={RequestMethod.PUT})
    @ResponseBody
    public TablesSetDto updateTablesSet(@PathVariable Long aliasId, @PathVariable Long setId, @RequestBody TablesSetDto newSet) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        this.validateAliasIsActive(aliasId);
        Date started = new Date();
        Alias alias = this.getAlias(aliasId);
        TablesSet set = alias.getTablesSet(setId);
        LinkedHashMap<String, Object> params = new LinkedHashMap<String, Object>();
        params.put("dbex.audit.tableset.oldtables", set.getTables().stream().map(SimpleTableDto::toString).collect(Collectors.joining(", ")));
        set.setTablesNames(new ArrayList<SimpleTableDto>());
        this.aliasService.updateAlias(alias);
        set.setName(newSet.getName());
        set.setTablesNames(newSet.getTables());
        this.aliasService.updateAlias(alias);
        this.auditTableSetOperation(started, AuditTypes.AUDIT_TABLE_SET_UPDATE, set, params);
        return new TablesSetDto(alias, set);
    }

    @RequestMapping(value={"/aliases/{aliasId}/tablessets/{setId}"}, method={RequestMethod.DELETE})
    @ResponseBody
    public void deleteTablesSet(@PathVariable Long aliasId, @PathVariable Long setId) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        this.validateAliasIsActive(aliasId);
        Date started = new Date();
        Alias alias = this.getAlias(aliasId);
        TablesSet set = alias.getTablesSet(setId);
        if (set != null) {
            alias.removeTablesSet(set);
        }
        this.aliasService.updateAlias(alias);
        this.auditTableSetOperation(started, AuditTypes.AUDIT_TABLE_SET_DELETE, set, null);
    }

    @RequestMapping(value={"/aliases/{aliasId}/tables"}, method={RequestMethod.GET})
    @ResponseBody
    public List<TableDto> getTables(@PathVariable Long aliasId) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        this.validateAliasIsActive(aliasId);
        Alias alias = this.getAlias(aliasId);
        Set<Table> tables = alias.getTables(this.databaseFactory);
        return TableDto.from(alias, tables);
    }

    @RequestMapping(value={"/aliases/{aliasId}/schemas"}, method={RequestMethod.GET})
    @ResponseBody
    public List<SchemaDto> getSchemas(@PathVariable Long aliasId) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        this.validateAliasIsActive(aliasId);
        Alias alias = this.getAlias(aliasId);
        boolean available = this.isAliasAvailable(alias);
        if (available) {
            return SchemaDto.from(alias, alias.getSchemas(this.databaseFactory));
        }
        return Arrays.asList(SchemaDto.unavailable());
    }

    private boolean isAliasAvailable(Alias alias) {
        return this.databaseFactory.isAvailable(alias.getWrappedConnectionString());
    }

    @RequestMapping(value={"/aliases/{aliasId}/tables"}, method={RequestMethod.PUT})
    @ResponseBody
    public void updateTable(@PathVariable Long aliasId, @RequestBody TableDto newTable) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        this.validateAliasIsActive(aliasId);
        Date started = new Date();
        Alias alias = this.getAlias(aliasId);
        TableSettings oldSettings = this.getAlias(aliasId).getSettings().getTableSettings(newTable.getSchema(), newTable.getName());
        TableSettings settings = alias.getSettings().getTableSettings(newTable.getSchema(), newTable.getName());
        if (newTable.getDisplayName() != null) {
            settings.setDisplayName(newTable.getDisplayName().trim());
        }
        if (newTable.getLogging() != null) {
            settings.setLogging(newTable.getLogging());
        }
        this.aliasService.updateAlias(alias);
        this.auditTableUpdate(started, alias.getId(), oldSettings, settings);
    }

    @RequestMapping(value={"/aliases/{aliasId}/columns"}, method={RequestMethod.PUT})
    @ResponseBody
    public void updateColumn(@PathVariable Long aliasId, @RequestBody ColumnDto newColumn) {
        this.authorizationHelper.assertFullAdministrationRights(() -> {});
        this.validateAliasIsActive(aliasId);
        Date started = new Date();
        Alias alias = this.getAlias(aliasId);
        ColumnSettings oldSettings = this.getAlias(aliasId).getSettings().getTableSettings(newColumn.getSchemaName(), newColumn.getTableName()).getColumnSettings(newColumn.getName());
        TableSettings tableSettings = alias.getSettings().getTableSettings(newColumn.getSchemaName(), newColumn.getTableName());
        ColumnSettings settings = tableSettings.getColumnSettings(newColumn.getName());
        if (newColumn.getDisplayName() != null) {
            settings.setDisplayName(newColumn.getDisplayName().trim());
        }
        this.aliasService.updateAlias(alias);
        this.auditColumnUpdate(started, alias.getId(), tableSettings, oldSettings, settings, true);
    }

    @ResponseBody
    @ExceptionHandler(value={AliasNotActiveException.class})
    public ResponseEntity<?> handleAliasNotActive(AliasNotActiveException e) {
        return new ResponseEntity((Object)e.getMessage(), (HttpStatusCode)HttpStatus.BAD_REQUEST);
    }

    private void validateAliasIsActive(Long aliasId) {
        Alias alias = this.aliasService.getAlias(aliasId);
        if (!alias.getIsActive().booleanValue()) {
            Translator translator = Translators.get(DataController.class);
            throw new AliasNotActiveException(translator.getMessage("dbex.alias.notActive.exception", new Object[]{alias.getName()}));
        }
    }

    private Alias getAlias(Long aliasId) {
        Alias alias = this.aliasService.getAlias(aliasId);
        if (alias == null) {
            throw new AliasNotFoundException(aliasId);
        }
        return alias;
    }

    private void normalizeTablesInSet(TablesSet set) {
        Alias alias = set.getAlias();
        Set<SimpleTableDto> tables = set.getTables();
        for (SimpleTableDto table : tables) {
            if (table.getSchema() != null) continue;
            table.setSchema(this.getDefaultSchema(alias));
        }
    }

    private String getDefaultSchema(Alias alias) {
        Database database = this.databaseFactory.create(alias.getWrappedConnectionString());
        return database.getDefaultSchemaName();
    }

    @ResponseBody
    @ExceptionHandler(value={DatabaseNotAvailableException.class})
    public ResponseEntity<?> handleDatabaseNotAvailable(Exception ex) {
        return this.handle(ex, ImmutableMap.of((Object)"databaseNotAvailable", (Object)true));
    }

    private ResponseEntity<?> handle(Exception ex, Object responseObject) {
        log.error(ex.getMessage(), (Throwable)ex);
        return new ResponseEntity(responseObject, (HttpStatusCode)HttpStatus.NOT_ACCEPTABLE);
    }

    private void auditAliasOperation(Date started, AuditTypes type, Alias alias, Alias oldAlias) {
        String jdbcUrl;
        String isJdbcUrlConfigParameter;
        String logging;
        String isActive;
        String databaseType;
        String password;
        String user;
        String databasePort;
        String databaseAddress;
        String aliasRealName;
        String aliasName;
        LinkedHashMap<String, Object> params = new LinkedHashMap<String, Object>();
        String changePattern = "%s -> %s";
        boolean isJdbcUrlConfig = alias.getIsJdbcUrlConfig();
        Integer port = alias.getConnectionStringDomain().getPort();
        DatabaseType dbType = alias.getConnectionStringDomain().getType();
        if (oldAlias == null) {
            aliasName = alias.getName();
            aliasRealName = alias.getConnectionStringDomain().getCatalog();
            databaseAddress = alias.getConnectionStringDomain().getHost();
            databasePort = port != null ? port.toString() : "";
            user = alias.getConnectionStringDomain().getUser();
            password = "********";
            databaseType = isJdbcUrlConfig ? "" : dbType.name();
            isActive = alias.getIsActive().toString();
            logging = alias.getLogging().toString();
            isJdbcUrlConfigParameter = String.valueOf(isJdbcUrlConfig);
            jdbcUrl = isJdbcUrlConfig ? alias.getConnectionStringDomain().getJdbcUrl() : "";
        } else {
            aliasName = String.format(changePattern, oldAlias.getName(), alias.getName());
            aliasRealName = String.format(changePattern, oldAlias.getConnectionStringDomain().getCatalog(), alias.getConnectionStringDomain().getCatalog());
            databaseAddress = String.format(changePattern, oldAlias.getConnectionStringDomain().getHost(), alias.getConnectionStringDomain().getHost());
            Integer oldPort = oldAlias.getConnectionStringDomain().getPort();
            databasePort = String.format(changePattern, oldPort != null ? oldPort.toString() : "", port != null ? port.toString() : "");
            user = String.format(changePattern, oldAlias.getConnectionStringDomain().getUser(), alias.getConnectionStringDomain().getUser());
            password = String.format(changePattern, "********", "********");
            DatabaseType oldDbType = oldAlias.getConnectionStringDomain().getType();
            databaseType = String.format(changePattern, oldAlias.getIsJdbcUrlConfig() != false ? "" : oldDbType.name(), isJdbcUrlConfig ? "" : dbType.name());
            isActive = String.format(changePattern, oldAlias.getIsActive(), alias.getIsActive());
            logging = String.format(changePattern, oldAlias.getLogging(), alias.getLogging());
            isJdbcUrlConfigParameter = String.format(changePattern, oldAlias.getIsJdbcUrlConfig(), isJdbcUrlConfig);
            String oldJdbcUrl = oldAlias.getIsJdbcUrlConfig() != false ? oldAlias.getConnectionStringDomain().getJdbcUrl() : "";
            String newJdbcUrl = isJdbcUrlConfig ? alias.getConnectionStringDomain().getJdbcUrl() : "";
            jdbcUrl = String.format(changePattern, oldJdbcUrl, newJdbcUrl);
        }
        params.put("dbex.audit.alias.id", alias.getId());
        params.put("dbex.audit.alias.name", aliasName);
        params.put("dbex.audit.alias.realName", aliasRealName);
        params.put("dbex.audit.alias.databaseAddress", databaseAddress);
        params.put("dbex.audit.alias.databasePort", databasePort);
        params.put("dbex.audit.alias.isSystem", alias.getIsSystemAlias());
        params.put("dbex.audit.alias.user", user);
        params.put("dbex.audit.alias.password", password);
        params.put("dbex.audit.alias.databaseType", databaseType);
        params.put("dbex.audit.alias.active", isActive);
        params.put("dbex.audit.alias.table.logging", logging);
        params.put("dbex.audit.alias.isJdbcUrlConfig", isJdbcUrlConfigParameter);
        params.put("dbex.audit.alias.jdbcUrl", jdbcUrl);
        ManualAuditBuilder.getInstance().type(type.getValue()).username(UserContext.userName()).success(true).params(params).started(started).stopped(new Date()).build().log();
    }

    private void auditSystemAliasOperation(Date started, AuditTypes type, Alias alias) {
        LinkedHashMap<String, Object> params = new LinkedHashMap<String, Object>();
        params.put("dbex.audit.alias.id", alias.getId());
        params.put("dbex.audit.alias.name", alias.getName());
        params.put("dbex.audit.alias.connectionString", alias.getWrappedConnectionString().getJdbcUrl());
        params.put("dbex.audit.alias.isSystem", alias.getIsSystemAlias());
        params.put("dbex.audit.alias.user", alias.getWrappedConnectionString().getUser());
        params.put("dbex.audit.alias.password", "********");
        params.put("dbex.audit.alias.databaseType", alias.getWrappedConnectionString().getType().name());
        params.put("dbex.audit.alias.active", alias.getIsActive());
        params.put("dbex.audit.alias.table.logging", alias.getLogging());
        ManualAuditBuilder.getInstance().type(type.getValue()).username(UserContext.userName()).success(true).params(params).started(started).stopped(new Date()).build().log();
    }

    private void auditTableSetOperation(Date started, AuditTypes type, TablesSet set, Map<String, Object> moreParams) {
        LinkedHashMap<String, Object> params = new LinkedHashMap<String, Object>();
        params.put("dbex.audit.tablesetname", set.getName());
        params.put("dbex.audit.tableset.tables", set.getTables().stream().map(SimpleTableDto::toString).collect(Collectors.joining(", ")));
        if (moreParams != null) {
            params.putAll(moreParams);
        }
        ManualAuditBuilder.getInstance().type(type.getValue()).username(UserContext.userName()).success(true).params(params).started(started).stopped(new Date()).build().log();
    }

    private void auditTableUpdate(Date started, Long id, TableSettings oldSettings, TableSettings settings) {
        LinkedHashMap<String, Object> params = new LinkedHashMap<String, Object>();
        String changePattern = "%s -> %s";
        params.put("dbex.audit.alias.id", id);
        params.put("dbex.audit.alias.table.tableName", settings.getTableName());
        if (StringUtils.isNotBlank((CharSequence)oldSettings.getDisplayName()) || StringUtils.isNotBlank((CharSequence)settings.getDisplayName())) {
            params.put("dbex.audit.alias.table.displayedName", String.format(changePattern, StringUtils.defaultString((String)oldSettings.getDisplayName()), settings.getDisplayName()));
        }
        params.put("dbex.audit.alias.table.logging", String.format(changePattern, oldSettings.isLogging(), settings.isLogging()));
        ManualAuditBuilder.getInstance().type(AuditTypes.AUDIT_TABLE_CHANGE.getValue()).username(UserContext.userName()).success(true).params(params).started(started).stopped(new Date()).build().log();
    }

    private void auditColumnUpdate(Date started, Long id, TableSettings tableSettings, ColumnSettings oldColumnSettigns, ColumnSettings columnSettings, boolean success) {
        LinkedHashMap<String, Object> params = new LinkedHashMap<String, Object>();
        String changePattern = "%s -> %s";
        params.put("dbex.audit.alias.id", id);
        params.put("dbex.audit.alias.column.tableName", tableSettings.getTableName());
        params.put("dbex.audit.alias.column.columnName", columnSettings.getColumnName());
        params.put("dbex.audit.alias.column.displayedName", String.format(changePattern, oldColumnSettigns.getDisplayName(), columnSettings.getDisplayName()));
        ManualAuditBuilder.getInstance().type(AuditTypes.AUDIT_COLUMN_CHANGE.getValue()).username(UserContext.userName()).success(success).params(params).started(started).stopped(new Date()).build().log();
    }

    private String resolveDisplayedName(AliasDto alias) {
        Translator translator = Translators.get(AliasController.class);
        return alias.getIsActive() != false ? alias.getName() : alias.getName() + " (" + translator.getMessage("dbex.alias.notActive") + ")";
    }

    @ResponseStatus(value=HttpStatus.BAD_REQUEST)
    @ExceptionHandler(value={InvalidJdbcUrlException.class})
    @ResponseBody
    public ResponseEntity<?> handleNotFullRightsException() {
        return new ResponseEntity((Object)ImmutableMap.of((Object)"invalidJdbcUrl", (Object)true), (HttpStatusCode)HttpStatus.BAD_REQUEST);
    }

    @Autowired
    public AliasController(AliasService aliasService, DatabaseFactory databaseFactory, AuthorizationHelper authorizationHelper) {
        this.aliasService = aliasService;
        this.databaseFactory = databaseFactory;
        this.authorizationHelper = authorizationHelper;
    }
}

