package com.suncode.dbexplorer.util.persistence;

import java.io.IOException;
import java.io.Serializable;
import java.util.List;

import org.hibernate.EmptyInterceptor;
import org.hibernate.Interceptor;
import org.hibernate.type.Type;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;

/**
 * Interceptor {@link Interceptor} odpowiedzialny za serializację i deserializację obiektu
 * {@linkplain Json}.
 * 
 * @author Cezary Kozar 16 sty 2016
 */
@SuppressWarnings( "serial" )
public class JsonInterceptor
    extends EmptyInterceptor
{
    private final ObjectMapper mapper = new ObjectMapper();

    @Override
    public boolean onSave( Object entity, Serializable id, Object[] state, String[] propertyNames, Type[] types )
    {
        return serialize( findJsonObjects( state ) );
    }

    @Override
    public boolean onFlushDirty( Object entity, Serializable id, Object[] currentState,
                                 Object[] previousState,
                                 String[] propertyNames, Type[] types )
    {
        return serialize( findJsonObjects( currentState ) );
    }

    @Override
    public boolean onLoad( Object entity, Serializable id, Object[] state, String[]
                           propertyNames, Type[] types )
    {
        return deserialize( findJsonObjects( state ) );
    }

    private boolean serialize( List<Json<?>> jsons )
    {
        if ( jsons.isEmpty() )
        {
            return false;
        }

        for ( Json<?> json : jsons )
        {
            try
            {
                json.setSerialized( mapper.writeValueAsBytes( json ) );
            }
            catch ( IOException e )
            {
                throw new RuntimeException( e );
            }
        }
        return true;
    }

    @SuppressWarnings( { "rawtypes", "unchecked" } )
    private boolean deserialize( List<Json<?>> jsons )
    {
        if ( jsons.isEmpty() )
        {
            return false;
        }

        for ( Json json : jsons )
        {
            try
            {
                Json deserialized = mapper.readValue( json.getSerialized(), json.getClass() );
                json.apply( deserialized );
            }
            catch ( IOException e )
            {
                throw new RuntimeException( e );
            }
        }
        return true;
    }

    private List<Json<?>> findJsonObjects( Object[] props )
    {
        List<Json<?>> jsons = Lists.newArrayList();
        for ( Object prop : props )
        {
            if ( prop instanceof Json )
            {
                jsons.add( (Json<?>) prop );
            }
        }
        return jsons;
    }
}