package com.suncode.dbexplorer.internal;

import java.util.List;

import com.suncode.dbexplorer.util.persistence.PrefixTableNamingStrategy;
import org.hibernate.Interceptor;
import org.hibernate.boot.model.naming.CamelCaseToUnderscoresNamingStrategy;
import org.hibernate.boot.model.naming.Identifier;
import org.hibernate.engine.jdbc.env.spi.JdbcEnvironment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.orm.hibernate5.HibernateTransactionManager;
import org.springframework.orm.hibernate5.LocalSessionFactoryBean;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.annotation.EnableTransactionManagement;

import com.suncode.dbexplorer.util.persistence.CompositeInterceptor;
import com.suncode.dbexplorer.util.persistence.EntityInjector;
import com.suncode.dbexplorer.util.persistence.JsonInterceptor;
import com.suncode.plugin.framework.util.ClassLoaderUtils;
import com.suncode.plugin.framework.web.mvc.MvcModuleConfigurerAdapter;
import com.suncode.pwfl.persistence.support.hibernate.SystemBaseSessionFactoryBean;

/**
 * Konfiguracja kontekstu.
 * 
 * @author Cezary Kozar 16 sty 2016
 */
@Configuration
@EnableTransactionManagement
public class AppConfig
    extends MvcModuleConfigurerAdapter
{
    private static final String PREFIX = "dbex_";

    private final Logger logger = LoggerFactory.getLogger( getClass() );

    @Autowired
    private ApplicationContext context;

    @Bean
    public LocalSessionFactoryBean sessionFactory()
    {
        SystemBaseSessionFactoryBean sessionFactoryBean = new SystemBaseSessionFactoryBean(true);
        sessionFactoryBean.setPackagesToScan( "com.suncode.dbexplorer" );

        sessionFactoryBean.setPhysicalNamingStrategy(new PrefixTableNamingStrategy(
                PREFIX,
                new CamelCaseToUnderscoresNamingStrategy()
        ));

        Interceptor json = new JsonInterceptor();
        Interceptor injector = new EntityInjector( context );
        sessionFactoryBean.setEntityInterceptor( new CompositeInterceptor( json, injector ) );
        return sessionFactoryBean;
    }

    @Bean
    public PlatformTransactionManager transactionManager()
    {
        return new HibernateTransactionManager( sessionFactory().getObject() );
    }

    @Override
    public void configureMessageConverters( List<HttpMessageConverter<?>> converters )
    {
        // PWFL-3214
        // W celu wykorzystania DownloadResource musimy zarejestrowac odpowiedni MessageConverter
        String className = "com.suncode.pwfl.web.support.io.DownloadResourceHttpMessageConverter";
        try
        {
            Class<?> clazz = ClassLoaderUtils.loadSystemClass( className );
            converters.add( 0, (HttpMessageConverter<?>) clazz.newInstance() );
        }
        catch ( Exception e )
        {
            logger.warn( "Could not find class [{}] in system classloader. Excel export will fail.", className, e );
        }
    }
}
