package com.suncode.dbexplorer.database.schema;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;

public class TableSchema
{
    private final String schema;
    
    private final String name;
    
    private final boolean view;

    private final Map<String, ColumnSchema> columns;

    private final PrimaryKey primaryKey;

    public TableSchema( String schema, String name, boolean view, List<ColumnSchema> columns, PrimaryKey primaryKey )
    {
        this.schema = schema;
        this.name = name;
        this.primaryKey = primaryKey;
        this.view = view;

        Map<String, ColumnSchema> columnsMap = new LinkedHashMap<String, ColumnSchema>();
        for ( ColumnSchema column : columns )
        {
            columnsMap.put( column.getName(), column );
        }
        this.columns = ImmutableMap.copyOf( columnsMap );
    }

    public String getSchema()
    {
        return schema;
    }

    public String getName()
    {
        return name;
    }
    
    public String getFullName()
    {
        StringBuilder result = new StringBuilder();
        if( !StringUtils.isEmpty( schema ) )
        {
            result.append( schema );
            result.append( "." );
        }
        
        result.append( name );
        return result.toString();
    }
    
    public boolean isView()
    {
        return view;
    }

    public PrimaryKey getPrimaryKey()
    {
        return primaryKey;
    }

    public boolean hasPrimaryKey()
    {
        return primaryKey != null;
    }

    public ColumnSchema[] getPrimaryKeyColumns()
    {
        Assert.state( hasPrimaryKey(), "[Assertion failed] - this state invariant must be true" );

        List<ColumnSchema> result = new ArrayList<ColumnSchema>();
        for ( String pkColumn : primaryKey.getColumns() )
        {
            result.add( this.columns.get( pkColumn ) );
        }
        return Iterables.toArray( result, ColumnSchema.class );
    }

    public List<ColumnSchema> getColumns()
    {
        return new ArrayList<ColumnSchema>( columns.values() );
    }

    public ColumnSchema getColumn( String name )
    {
        if ( !columns.containsKey( name ) )
        {
            throw new IllegalArgumentException( "Column [" + name + "] is not part of [" + this.name + "] table" );
        }
        return columns.get( name );
    }

    public boolean hasColumn( String name )
    {
        return columns.containsKey( name );
    }

    public Set<String> getColumnNames()
    {
        Set<String> names = Sets.newHashSet();
        for ( ColumnSchema column : columns.values() )
        {
            names.add( column.getName() );
        }
        return names;
    }
}
