package com.suncode.dbexplorer.database.internal.type;

import java.sql.Types;
import java.util.Map;

import org.springframework.util.Assert;

import com.google.common.collect.Maps;
import com.suncode.dbexplorer.database.type.BinaryDataType;
import com.suncode.dbexplorer.database.type.BooleanDataType;
import com.suncode.dbexplorer.database.type.DataType;
import com.suncode.dbexplorer.database.type.DateDataType;
import com.suncode.dbexplorer.database.type.DateTimeDataType;
import com.suncode.dbexplorer.database.type.FloatDataType;
import com.suncode.dbexplorer.database.type.IntegerDataType;
import com.suncode.dbexplorer.database.type.NativeType;
import com.suncode.dbexplorer.database.type.StringDataType;
import com.suncode.dbexplorer.database.type.TimeDataType;
import com.suncode.dbexplorer.database.type.UnknownDataType;

public abstract class DefaultDataTypeRegistry
    implements DataTypeRegistry
{

    private final Map<Class<? extends DataType>, DataTypeHandler> typeHandlers = Maps.newHashMap();

    private final Map<Integer, Class<? extends DataType>> sqlTypeMapping = Maps.newHashMap();

    private final Map<String, Class<? extends DataType>> nativeTypeMapping = Maps.newHashMap();

    public DefaultDataTypeRegistry()
    {

        // string
        registerTypeHandler( StringDataType.class, new StringDataTypeHandler() );

        mapType( Types.CHAR, StringDataType.class );
        mapType( Types.NCHAR, StringDataType.class );
        mapType( Types.VARCHAR, StringDataType.class );
        mapType( Types.NVARCHAR, StringDataType.class );
        mapType( Types.LONGVARCHAR, StringDataType.class );
        mapType( Types.LONGNVARCHAR, StringDataType.class );

        // numeric
        registerTypeHandler( IntegerDataType.class, new IntegerDataTypeHandler() );
        registerTypeHandler( FloatDataType.class, new FloatDataTypeHandler() );

        mapType( Types.TINYINT, IntegerDataType.class );
        mapType( Types.SMALLINT, IntegerDataType.class );
        mapType( Types.INTEGER, IntegerDataType.class );
        mapType( Types.BIGINT, IntegerDataType.class );

        mapType( Types.DECIMAL, FloatDataType.class );
        mapType( Types.NUMERIC, FloatDataType.class );
        mapType( Types.DOUBLE, FloatDataType.class );
        mapType( Types.FLOAT, FloatDataType.class );
        mapType( Types.REAL, FloatDataType.class );

        // date time
        registerTypeHandler( DateTimeDataType.class, new DateTimeDataTypeHandler() );
        registerTypeHandler( DateDataType.class, new DateDataTypeHandler() );
        registerTypeHandler( TimeDataType.class, new TimeDataTypeHandler() );

        mapType( Types.TIMESTAMP, DateTimeDataType.class );
        mapType( Types.DATE, DateDataType.class );
        mapType( Types.TIME, TimeDataType.class );

        // binary
        registerTypeHandler( BinaryDataType.class, new BinaryDataTypeHandler() );

        mapType( Types.BINARY, BinaryDataType.class );
        mapType( Types.VARBINARY, BinaryDataType.class );
        mapType( Types.LONGVARBINARY, BinaryDataType.class );
        mapType( Types.BLOB, BinaryDataType.class );
        mapType( Types.CLOB, BinaryDataType.class );
        mapType( Types.NCLOB, BinaryDataType.class );

        // boolean
        registerTypeHandler( BooleanDataType.class, new BooleanDataTypeHandler() );

        mapType( Types.BIT, BooleanDataType.class );
        mapType( Types.BOOLEAN, BooleanDataType.class );

        // unknown
        registerTypeHandler( UnknownDataType.class, new UnknownDataTypeHandler() );
    }

    @Override
    public DataType getType( NativeType nativeType )
    {
        Class<? extends DataType> dataTypeClass = nativeTypeMapping.get( nativeType.getName() );
        if ( dataTypeClass == null )
        {
            dataTypeClass = sqlTypeMapping.get( nativeType.getSqlType() );
            if ( dataTypeClass == null )
            {
                // TODO: log
                dataTypeClass = UnknownDataType.class;
            }
        }

        DataTypeHandler handler = getTypeHandler( dataTypeClass );
        return handler.create( nativeType );
    }

    @Override
    public DataTypeHandler getTypeHandler( DataType dataType )
    {
        Assert.notNull( dataType, "[Assertion failed] - this argument is required; it must not be null" );
        return getTypeHandler( dataType.getClass() );
    }

    protected DataTypeHandler getTypeHandler( Class<?> dataTypeClass )
    {
        DataTypeHandler handler = typeHandlers.get( dataTypeClass );
        if ( handler == null )
        {
            throw new IllegalStateException( "There is not matching type handler registered for given type ("
                + dataTypeClass.getName() + ")" );
        }
        return handler;
    }

    protected void mapType( int jdbcType, Class<? extends DataType> dataTypeClass )
    {
        Assert.notNull( dataTypeClass, "[Assertion failed] - this argument is required; it must not be null" );

        if ( !typeHandlers.containsKey( dataTypeClass ) )
        {
            throw new IllegalStateException( "There is not matching type handler registered for "
                + dataTypeClass.getName() );
        }
        sqlTypeMapping.put( jdbcType, dataTypeClass );
    }

    protected void mapNativeType( String nativeTypeName, Class<? extends DataType> dataTypeClass )
    {
        Assert.hasText( nativeTypeName, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );
        Assert.notNull( dataTypeClass, "[Assertion failed] - this argument is required; it must not be null" );

        if ( !typeHandlers.containsKey( dataTypeClass ) )
        {
            throw new IllegalStateException( "There is not matching type handler registered for "
                + dataTypeClass.getName() );
        }
        nativeTypeMapping.put( nativeTypeName, dataTypeClass );
    }

    protected void registerTypeHandler( Class<? extends DataType> dataTypeClass, DataTypeHandler typeHandler )
    {
        Assert.notNull( dataTypeClass, "[Assertion failed] - this argument is required; it must not be null" );
        Assert.notNull( typeHandler, "[Assertion failed] - this argument is required; it must not be null" );

        typeHandlers.put( dataTypeClass, typeHandler );
    }
}
