package com.suncode.dbexplorer.database.internal.support;

import com.google.common.base.Optional;
import com.google.common.collect.Iterables;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeanWrapperImpl;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.util.Assert;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ResultSetMapper
{
	private final ResultSet resultSet;

	private final Map<String, Extractor> mappings = new HashMap<>();

	private ResultSetMapper(ResultSet resultSet)
	{
		Assert.notNull( resultSet, "Result set cannot be null" );
		this.resultSet = resultSet;
	}

	public static ResultSetMapper map(ResultSet resultSet)
	{
		return new ResultSetMapper(resultSet);
	}

	public ResultSetMapper mapString(String column)
	{
		return mapString(column, null);
	}

	public ResultSetMapper mapString(String column, String alias)
	{
		mappings.put(column, new StringExtractor(alias));
		return this;
	}

	public ResultSetMapper mapInt(String column)
	{
		return mapInt(column, null);
	}

	public ResultSetMapper mapInt(String column, String alias)
	{
		mappings.put(column, new IntExtractor(alias));
		return this;
	}

	public ResultSetMapper mapYesNoBoolean(String column, String alias)
	{
		mappings.put(column, new YesNoExtractor(alias));
		return this;
	}

	public <T> List<T> beans(Class<T> type)
			throws SQLException
	{
		try ( resultSet )
		{
			Assert.state( !resultSet.isClosed(), "Result set cannot be closed" );
			Assert.state( !mappings.isEmpty(), "Mappings shouldn't be empty" );

			List<T> results = new ArrayList<>();
			while ( resultSet.next() )
			{
				MutablePropertyValues properties = new MutablePropertyValues();
				for ( Map.Entry<String, Extractor> entry : mappings.entrySet() )
				{
					properties.add( entry.getValue().alias, entry.getValue().extract( resultSet, entry.getKey() ) );
				}

				T row = BeanUtils.instantiate( type );
				BeanWrapperImpl beanWrapper = new BeanWrapperImpl( row );
				beanWrapper.setPropertyValues( properties );
				results.add( row );
			}
			return results;
		}
	}

	@SuppressWarnings("unchecked")
	public <T> List<T> single()
			throws SQLException
	{
		try ( resultSet )
		{
			Assert.state( !resultSet.isClosed(), "Result set cannot be closed" );
			Assert.state( mappings.size() == 1, "Mappings shouldn't be of size 1" );

			String column = Iterables.getFirst( mappings.keySet(), null );
			Extractor extractor = mappings.get( column );

			List<T> results = new ArrayList<>();
			while ( resultSet.next() )
			{
				results.add( (T) extractor.extract( resultSet, column ) );

			}
			return results;
		}
	}

	private abstract static class Extractor
	{
		String alias;

		Extractor( String alias )
		{
			this.alias = alias;
		}

		abstract Object extract( ResultSet rs, String column )
			throws SQLException;
	}

	private static class IntExtractor
			extends Extractor
	{

		IntExtractor(String alias)
		{
			super(alias);
		}

		@Override
		Integer extract(ResultSet rs, String column)
				throws SQLException
		{
			return rs.getInt(column);
		}
	}

	private static class StringExtractor
			extends Extractor
	{
		StringExtractor(String alias)
		{
			super(alias);
		}

		@Override
		String extract(ResultSet rs, String column)
				throws SQLException
		{
			return rs.getString(column);
		}
	}

	private static class YesNoExtractor
			extends Extractor
	{
		YesNoExtractor(String alias)
		{
			super(alias);
		}

		@Override
		Optional<Boolean> extract(ResultSet rs, String column)
				throws SQLException
		{
			String value = rs.getString(column);
			if ("YES".equals(value))
			{
				return Optional.of( Boolean.TRUE );
			}
			else if ("NO".equals(value))
			{
				return Optional.of( Boolean.FALSE );
			}
			return Optional.absent();
		}
	}
}
