package com.suncode.dbexplorer.database.internal.query;

import com.google.common.collect.Lists;
import com.suncode.dbexplorer.database.DatabaseSession;
import com.suncode.dbexplorer.database.Record;
import com.suncode.dbexplorer.database.RecordId;
import com.suncode.dbexplorer.database.internal.DatabaseImplementor;
import com.suncode.dbexplorer.database.internal.type.DataTypeHandler;
import com.suncode.dbexplorer.database.query.Condition;
import com.suncode.dbexplorer.database.query.Order;
import com.suncode.dbexplorer.database.query.Page;
import com.suncode.dbexplorer.database.query.Pagination;
import com.suncode.dbexplorer.database.query.QueryContext;
import com.suncode.dbexplorer.database.query.QueryParameter;
import com.suncode.dbexplorer.database.query.SelectQuery;
import com.suncode.dbexplorer.database.schema.ColumnSchema;
import com.suncode.dbexplorer.database.type.BasicDataType;
import com.suncode.dbexplorer.database.type.DataType;
import org.apache.ibatis.jdbc.SQL;
import org.hibernate.SQLQuery;
import org.hibernate.transform.ResultTransformer;
import org.hibernate.type.StandardBasicTypes;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

public class SelectQueryImpl
    extends AbstractQuery
    implements SelectQuery
{
    private List<Condition> whereConditions = new ArrayList<Condition>();

    private List<Order> orders = new ArrayList<Order>();

    private boolean includeBinary;

    private int counter = 0;

    private Map<String, String> aliases = new HashMap<String, String>();

    private Map<String, String> aliases2 = new HashMap<String, String>();

    public SelectQueryImpl( DatabaseSession session, DatabaseImplementor implementor )
    {
        super( session, implementor );
    }

    @Override
    public SelectQuery from( String table )
    {
        Assert.hasText( table, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );

        TablePathInfo pathInfo = getPathInfo( table );
        return from( pathInfo.getSchema(), pathInfo.getName() );
    }

    @Override
    public SelectQuery from( String schema, String table )
    {
        Assert.hasText( schema, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );
        Assert.hasText( table, "[Assertion failed] - this String argument must have text; it must not be null, empty, or blank" );

        this.rootTable = session.getDatabase().getSchema( schema ).getTable( table );
        return this;
    }

    @Override
    public SelectQuery where( Condition... conditions )
    {
        for ( Condition condition : conditions )
        {
            where( condition );
        }
        return this;
    }

    @Override
    public SelectQuery where( Condition condition )
    {
        Assert.notNull( condition, "[Assertion failed] - this argument is required; it must not be null" );
        whereConditions.add( condition );
        return this;
    }

    @Override
    public SelectQuery addOrder( Order order )
    {
        return addOrder( new Order[] { order } );
    }

    @Override
    public SelectQuery addOrder( Order... orders )
    {
        for ( Order order : orders )
        {
            Assert.notNull( order, "[Assertion failed] - this argument is required; it must not be null" );
            this.orders.add( order );
        }
        return this;
    }

    @Override
    public SelectQuery includeBinary()
    {
        this.includeBinary = true;
        return this;
    }

    @Override
    public Record uniqueRecord()
    {
        List<Record> records = list();
        if ( records.size() > 1 )
        {
            throw new IllegalStateException( "Non unique result set: found " + records.size() + " matching records" );
        }
        return records.isEmpty() ? null : records.get( 0 );
    }

    @Override
    public long count()
    {
        SQLQuery count = prepareSqlQuery( true );
        return (Long) count.uniqueResult();
    }

    @Override
    @SuppressWarnings( "unchecked" )
    public List<Record> list()
    {

        SQLQuery query = prepareSqlQuery( false );
        return query.list();
    }

    @Override
    @SuppressWarnings( "unchecked" )
    public Page<Record> page( Pagination pagination )
    {
        Assert.notNull( pagination, "[Assertion failed] - this argument is required; it must not be null" );

        if ( !pagination.hasOrder() )
        {
            String column;
            if ( rootTable.hasPrimaryKey() )
            {
                column = rootTable.getPrimaryKeyColumns()[0].getName();
            }
            else
            {
                column = rootTable.getColumns().get( 0 ).getName();
            }

            pagination.addOrder( Order.ASC( column ) );
        }
        this.orders.addAll( 0, pagination.getOrder() );

        SQLQuery query = prepareSqlQuery( false );
        query.setFirstResult( pagination.getStart() );
        query.setMaxResults( pagination.getLimit() );

        List<Record> records = query.list();
        long size = records.size();

        // strona nie 0 i < max to też

        if ( size == pagination.getLimit() || pagination.getStart() != 0 )
        {
            // TODO: refactor
            SQLQuery count = prepareSqlQuery( true );
            size = (Long) count.uniqueResult();
        }

        return new Page<Record>( records, size );
    }

    private SQLQuery prepareSqlQuery( boolean count )
    {

        // TODO: stworzyc odpowiednik org.apache.ibatis.jdbc.SQL żeby pozbyc się zależności
        // TODO: dodac aliasy kolumn
        // TODO: pomijanie danych binarnych - tylko dostęp bezpośredni poprzez podanie takiej
        // kolumny w select?
        QueryContext queryContext = new QueryContextImpl( rootTable, implementor );
        SQL rawSql = buildSql( count, queryContext );

        StringBuilder sql = new StringBuilder();
        StringTokenizer tokenizer = new StringTokenizer( rawSql.toString().replace( "\n", " " ), " \"=><()", true );

        List<QueryParameter> parameters = Lists.newArrayList();
        for ( Condition condition : whereConditions )
        {
            Collections.addAll( parameters, condition.getParameters( queryContext ) );
        }

        int paramCount = 0;
        List<BindParam> binded = Lists.newArrayList();
        while ( tokenizer.hasMoreTokens() )
        {
            String token = tokenizer.nextToken();
            if ( token.equals( "?" ) )
            {
                QueryParameter param = parameters.get( paramCount );
                String paramName = "param" + paramCount;
                paramCount++;
                binded.add( new BindParam( paramName, param ) );

                sql.append( ":" + paramName );
                continue;
            }
            sql.append( token );
        }

        SQLQuery sqlQuery = session.hibernateSession().createSQLQuery( sql.toString() );

        if ( !count )
        {
            sqlQuery.setResultTransformer( new RecordResultTransformer() );
        }

        for ( BindParam param : binded )
        {

            DataType type = param.parameter.getType();
            DataTypeHandler typeHandler = implementor.getTypeRegistry().getTypeHandler( type );

            typeHandler.bindParameter( type, param.name, param.parameter.getValue(), sqlQuery );
        }

        if ( count )
        {
            sqlQuery.addScalar( "count", StandardBasicTypes.LONG );
        }
        else
        {
            // TODO: skalary po faktycznych kolumnach z select + typy
            for ( ColumnSchema column : rootTable.getColumns() )
            {

                DataType type = column.getType();
                DataTypeHandler typeHandler = implementor.getTypeRegistry().getTypeHandler( type );

                typeHandler.setScalar( type, aliases.get( column.getName() ), sqlQuery );
            }
        }
        return sqlQuery;
    }

    private SQL buildSql( boolean count, QueryContext queryContext )
    {
        SQL sql = new SQL();

        // SELECT
        if ( count )
        {
            sql.SELECT( "count(1) as " + implementor.escapeColumnName( "count" ) );
        }
        else
        {
            for ( String column : rootTable.getColumnNames() )
            {
                ColumnSchema col = rootTable.getColumn( column );
                if ( BasicDataType.is( col.getType(), BasicDataType.BINARY ) && !includeBinary )
                {
                    continue;
                }
                String alias = column.replaceAll( "[^\\w]", "_" ) + counter++;
                aliases.put( column, alias );
                aliases2.put( alias, column );
                sql.SELECT( implementor.escapeColumnName( column ) + " as " + alias );
            }
        }

        // FROM
        sql.FROM( implementor.escapeTableName( rootTable.getFullName() ) );

        // WHERE
        for ( Condition condition : whereConditions )
        {
            String conditionSql = condition.toSql( queryContext );
            sql.WHERE( conditionSql );
        }

        // ORDER BY
        if ( !count )
        {
            for ( Order order : orders )
            {
                sql.ORDER_BY( implementor.escapeColumnName( order.getColumn() ) + " " + order.getOrderType() );
            }
        }

        return sql;
    }

    private static class BindParam
    {
        String name;

        QueryParameter parameter;

        public BindParam( String name, QueryParameter parameter )
        {
            this.name = name;
            this.parameter = parameter;
        }
    }

    @SuppressWarnings( "serial" )
    private class RecordResultTransformer
        implements ResultTransformer
    {

        @Override
        public Object transformTuple( Object[] tuple, String[] aliases )
        {

            Record record = session.createRecord( rootTable.getSchema(), rootTable.getName() );
            for ( int i = 0; i < aliases.length; i++ )
            {
                record.set( SelectQueryImpl.this.aliases2.get( aliases[i] ), tuple[i] );
            }

            if ( record.hasId() )
            {
                RecordId id = record.getId();
                record.setId( id );
            }

            return record;
        }

        @Override
        @SuppressWarnings( "rawtypes" )
        public List transformList( List collection )
        {
            return collection;
        }
    }
}
